/** @file common.h
 * Common declarations.
 *
 * Basic data types, constants, externals, utility methods.
 * Common to both the generator and the modules.
 */ 
#ifndef __COMMON_H
#define __COMMON_H

#include <string>
#include <vector>
#include <sstream>


//--------------------------------------------------------------------------
// Type definitions

namespace rpg {
	
	/// Signed integer datatype.
	typedef long int rpg_int_t;
	/// Unsigned integer datatype.
	typedef unsigned long int rpg_uint_t;
	/// Float datatype.
	typedef double rpg_float_t;
	/// String datatype.
	typedef std::string rpg_str_t;	
	/// Boolean datatype.
	typedef bool rpg_bool_t;

	/// List of integers datatype.
	typedef std::vector<rpg_int_t> rpg_list_int_t;
	/// List of floats datatype.
	typedef std::vector<rpg_float_t> rpg_list_float_t;
	/// List of strings datatype.
	typedef std::vector<rpg_str_t> rpg_list_str_t;

	/// Exception datatype.
	enum rpg_exception_t {
		INVALID_SOURCE_ACCESS,
		UNIX_SYSCALL_ERROR,
		INTERNAL_ERROR,
		INVALID_CONFIGURATION_OPTION,
		UNKNOWN_MODULE
	};
}


//--------------------------------------------------------------------------
// Constant definitions

/// Minimum value of the uint data type.
#define RPG_UINT_MIN     0x00000000
/// Maximum value of the uint data type.
#define RPG_UINT_MAX     0xFFFFFFFF

/// System path separator.
#define PATH_SEPARATOR "/"

/** Output formatting, used both from individual modules and the app template */

#define OUTPUT_SEPARATOR 			            ";"

#define OUTPUT_MONOTONIC_MEASURE	            "monotonic"
#define OUTPUT_THREADTIME_MEASURE	            "threadtime"

#define OUTPUT_RESULT_OK_COUNT		            "result-ok-count"
#define OUTPUT_RESULT_ERROR_LATENT_COUNT		"result-error-latent-count"
#define OUTPUT_RESULT_ERROR_EFFECTIVE_COUNT     "result-error-effective-count"

#define OUTPUT_CONTEXT_CONFIG	                "config"
#define OUTPUT_CONTEXT_ISOLATED	                "isolated"
#define OUTPUT_CONTEXT_SHARED	                "shared"

#define OUTPUT_SYNCHRONIZED						"synchronized"

//--------------------------------------------------------------------------
// Utility methods

/** Write output line of the app to given stream*/
#define output(stream, module, context, attribute, value) \
		stream << (module) << OUTPUT_SEPARATOR << (context) << OUTPUT_SEPARATOR << (attribute) << OUTPUT_SEPARATOR << (value) << endl


namespace rpg {

	/* Stringification template */
	template <typename T>
	inline rpg::rpg_str_t toString (const T& value) {
		std::stringstream ss;
		ss << value;
		return ss.str ();
	}

	template <typename T>
	inline rpg::rpg_str_t toString (const std::vector<T> & vec) {
		std::stringstream ss;
		ss << "( ";

		const char * sep = "";

		for (typename std::vector <T> :: const_iterator
					iter = vec.begin(); iter != vec.end (); iter++) {

			ss << *iter << sep;
			sep = ", ";
		}

		ss << " )";
		return ss.str();
	}
}

//--------------------------------------------------------------------------
// External definitions

extern volatile bool rpg_terminate;
extern rpg::rpg_uint_t errors;
extern void set_err_prefix(rpg::rpg_str_t& error_prefix);
extern void error(rpg::rpg_str_t msg);
extern void error(rpg::rpg_str_t msg, rpg::rpg_uint_t line);
extern void error_syscall(rpg::rpg_str_t msg);

#endif
