#include <iostream>
#include <fstream>

#include "common.h"
#include "config.h"
#include "gener.h"
#include "lang.h"
#include "parser.h"
#include "rand.h"

#include <time.h>

using namespace std;
using namespace rpg;

#include <stdio.h>
#include <stdlib.h>

bool parse_params(int argc, char* argv[], rpg_str_t& lang_name,
		rpg_str_t& config_file, rpg_str_t& gen_path);

int main(int argc, char* argv[]) {

	/* The command line contains the configuration file name
	 * and the output path, neither of which has a default.
	 * Before them can an optional parameter -l or --lang "language"
	 * specifying the generated language, with cpp (default) for C++, or
	 * java for java language
	 */
	
	rpg_str_t lang_name;
	rpg_str_t config_file;
	rpg_str_t gen_path;
	
	if(!parse_params(argc, argv, lang_name, config_file, gen_path) ||
		!lang::init(lang_name))
	{
		error ("Expecting an optional -l or --lang parameter with java or cpp value, and the configuration file name and the output path on the command line.");
		return (-1);
	}
	
	if (gen_path [gen_path.length() - 1] != PATH_SEPARATOR [0]) gen_path.append (PATH_SEPARATOR);

	// First, parse the configuration file.

	FILE *input = fopen (config_file.c_str (), "r");
	if (input == NULL) {
		error ("Cannot read the configuration file.");
		return (-1);
	}
	
	rpg_context_t context(sprovider, cprovider);

	config_set_in (input);
	if (config_parse (context) != 0 || errors != 0) {
		error ("Error parsing the configuration file.");
		fclose (input);
		return (-1);
	}
	
	fclose (input);

	config_lex_destroy ();

    // Get the generator seed from the configuration.

	rpg_data_source_t seed_gen (sprovider.createSource(GROUP_CORE, ITEM_RANDOM_SEED));
	rand_init (seed_gen->getIntValue ());
	architecture_generator * archGen = architecture_generator::create_instance();

	// Get the configuration of the individual modules.

	try {	
		if (!archGen->load()) {
			cerr << "Failure while loading modules" << endl;
			return EXIT_FAILURE;
			// TODO: Unify the use of error functions for error reporting.
		}
	} catch (rpg_exception_t e) {
		cerr << "Exception raised while loading modules" << endl;
		return EXIT_FAILURE;
		// TODO: Unify the use of error functions for error reporting.
	}

	// Generate the architecture.

	try {
		/* generate the slot modules and calls between them */
		archGen->generateArchitecture();

		/* generate source file */
		rpg_str_t source_file_output_name = gen_path + lang_generator->getOutputMainFile(); 
		ofstream source_code_file_output (source_file_output_name.c_str(), ios::trunc);	
		if (!source_code_file_output.is_open()) {
			cerr << "Generation error: Cannot open file " << source_file_output_name << endl;
			return EXIT_FAILURE;
			// TODO: Unify the use of error functions for error reporting.
		}

		rpg_str_t template_file_input_name = gen_path + lang_generator->getInputTemplateFile();;
		ifstream source_code_file_template (template_file_input_name.c_str (), ios :: in);
		if (!source_code_file_template.is_open()) {
			cerr << "Generation error: Cannot open template file." << endl;
			source_code_file_output.close();
			return EXIT_FAILURE;
			// TODO: Unify the use of error functions for error reporting.
		}

		archGen->generateSource(source_code_file_output, source_code_file_template);
		source_code_file_output.close();
		source_code_file_template.close();

		/* generate configuration file */
		rpg_str_t config_file_name = gen_path + OUTPUT_CONFIG_FILE;
		ofstream configuration_file (config_file_name.c_str(), ios::trunc);
		if (!configuration_file.is_open()) {
			cerr << "Generation error: Cannot open configuration file for write: " << config_file_name << endl;
			return EXIT_FAILURE;
			// TODO: Unify the use of error functions for error reporting.
		}
		archGen->generateConfig (configuration_file);

		// Create specialized configuration items in the application file.
		//
		// The specialized configuration items are of two kinds.
		// One is generated from the generator config.
		// One is copied from the generator config.

        configuration_file << GEN_GLOBAL_CONFIGURATION_CAPTION << endl;

        // TODO A hack that allows us to use the dataSource::generateConfigAndXML method without an XML output.
        // The support for dumping dataSource configuration should be refactored to improve the code here.

        stringstream dummy_output_stream;

        #define COPY_CONFIG_VALUE(name,var) \
			rpg_data_source_t var (sprovider.createSource (GROUP_CORE, name)); \
            if (!var->isVoid ()) { \
                configuration_file << GROUP_APP GEN_DOUBLEDOTS name GEN_ASSIGN; \
                var->generateConfigAndXML (configuration_file, dummy_output_stream); \
            }

		// Experiment size related settings.

		COPY_CONFIG_VALUE (ITEM_MIN_CYCLES,              mincycles)
        COPY_CONFIG_VALUE (ITEM_MIN_TIME,                mintime)
        COPY_CONFIG_VALUE (ITEM_FREE_ROUND_COUNT,        warmup)
        COPY_CONFIG_VALUE (ITEM_MEASURE,                 cycles)
        COPY_CONFIG_VALUE (ITEM_TIME_LIMIT,              time_limit)
        COPY_CONFIG_VALUE (ITEM_VALUES_STORAGE_MAX_SIZE, values_max_cnt)

		// Thread pool related settings.

        COPY_CONFIG_VALUE (ITEM_THREAD_COUNT,        thread_count)
        COPY_CONFIG_VALUE (ITEM_CLIENT_WAIT_TIME_EX, client_wait_ex)
        COPY_CONFIG_VALUE (ITEM_CLIENT_COUNT,        client_count)

        // Close the thing ...

		configuration_file << GEN_RANDOM_SEED << endl;
		configuration_file.close();

		/* generate XML definition */
		rpg_str_t xml_file_name = gen_path + OUTPUT_XML_FILE; 
		ofstream xml_file (xml_file_name.c_str(), ios :: trunc);
		if (xml_file.is_open()) {
			archGen->generateXMLArchDef (xml_file);
			xml_file.close();
		} else {
			cerr << "Generation error: Cannot open file " << xml_file_name << endl;
			return EXIT_FAILURE;
			// TODO: Unify the use of error functions for error reporting.
		}

		delete archGen;

	} catch (rpg_exception_t e) {
		cerr << "Exception raised while loading modules and generating application" << endl;
		return EXIT_FAILURE;
		// TODO: Unify the use of error functions for error reporting.
	}
	
	return EXIT_SUCCESS;
}


/** Parses the program parameters.
  * @param argc Main argc argument.
  * @param argv Main argv argument.
  * @param lang_name Generated language name parsed from the parameters.
  * @param config_file Config file name parsed from the parameters.
  * @param gen_path Output file name parsed from the parameters.
  * @return True if ok, false if arguments are in unexpected format.
  */
bool parse_params(int argc, char* argv[], rpg_str_t& lang_name,
		rpg_str_t& config_file, rpg_str_t& gen_path) {
	if (argc == 3)
	{
		// C++ is default
		lang_name = LANG_CPP_NAME;
		config_file = argv [1];
		gen_path = argv [2];
	}
	else if(argc == 5)
	{
		rpg_str_t param_name = argv [1];
		if(param_name.compare("-l") != 0 && param_name.compare("--lang") != 0 )
			return false;
		
		lang_name = argv [2];
		config_file = argv [3];
		gen_path = argv [4];
	}
	else
		return false;
	return true;
}
