#include "common.h"
#include "gener.h"
#include "config.h"
#include "parser.h"
#include "lang.h"

#include <iostream>
#include <cctype>

using namespace std;
using namespace rpg;

auto_ptr<lang> rpg::lang_generator;

bool lang::init(rpg_str_t lang_name)
{
	if(lang_name.compare(LANG_CPP_NAME) == 0) lang_generator.reset(new lang_cpp());
	else if (lang_name.compare(LANG_JAVA_NAME) == 0) lang_generator.reset(new lang_java());
	else return false;

	return true;
}

/// A macro to output the mangled module class name.
#define MANGLED_MODULE_CLASS(t,n) (t).getClassName() << "_instantiated_" << (n)

rpg_str_t lang_cpp::getOutputMainFile() {
	return LANG_OUTPUT_MAIN_FILE_CPP;
}

rpg_str_t lang_cpp::getInputTemplateFile() {
	return LANG_INPUT_TEMPLATE_FILE_CPP;
}

/** Generates definition of the module and all of the submodules.
  * @param o Stream, to which is output printed.
  */
void lang_cpp::generateDefinition(std::ostream& o, const string &indent, rpg_slot_module_t* module) {
	// Unfortunately, the definition of the virtual methods has to be
	// repeated here to force the compiler to generate the externals.
	// Whenever the module interface is changed, this needs to be
	// changed as well.
	o << indent << "class " << MANGLED_MODULE_CLASS (module->type, module->name) << " : public " << module->type.getClassName() << "{" << endl
	  << indent << "\tpublic:" << endl
	  << indent << "\t" << MANGLED_MODULE_CLASS (module->type, module->name) << " (const std::string &);" << endl
	  << indent << "\tvirtual void init (int count, ...);" << endl
	  << indent << "\tvirtual void deinit ();" << endl
	  << indent << "\tvirtual int internal_work ();" << endl
	  << indent << "\tvirtual ~" << MANGLED_MODULE_CLASS (module->type, module->name) << " ();" << endl
	  << indent << "};" << endl;
}

/** Generates declaration of the module.
  * @param o Stream, to which is output printed.
  */
void lang_cpp::generateDeclaration(std::ostream& o, const string &indent, rpg_slot_module_t* module) {
	o << indent << MANGLED_MODULE_CLASS (module->type, module->name) << " " << module->name << "(" << GEN_QUOTE(module->name) << ");" << endl;
}

/** Generates initialization code of the module.
  * @param o Stream, to which is output printed.
  */
void lang_cpp::generateInitialization(std::ostream& o, const string &indent, rpg_slot_module_t* module) {
	/* generate initialization */
	o << indent << module->name << ".init( " << module->number_of_slots;
	for (rpg_uint_t i = 0; i < module->number_of_slots; i++) {
		o << ", &" << module->slots[i]->name;
	}
	o << " );" << endl;
}

/** Generates deinitialization code of the module.
  * @param o Stream, to which is output printed.
  */
void lang_cpp::generateDeinitialization(std::ostream& o, const string &indent, rpg_slot_module_t* module) {
	o << indent << module->name << ".deinit();" << endl;
}

/** Generates code measuring module working time.
  * @param o Stream, to which code is printed.
  */
void lang_cpp::generateMeasure(std::ostream& o, const string &indent, rpg_slot_module_t* module) {
    /* non-architectural modules should be always measured in isolation */
    if (module->number_of_slots == 0) {
        o << indent << module->name << ".measure_isolated ();" << endl;
    } else {
    	/* architectural modules only when a flag is set */
    	o << indent << "if (measure_isolated_architectures) " << module->name << ".measure_isolated ();" << endl;
    }
}

/** Generates call to the clear function of the module.
  * @param o Stream, to which is output printed.
  */
void lang_cpp::generateClear(std::ostream& o, const string &indent, rpg_slot_module_t* module) {
    o << indent << module->name << ".clear_measurements ();" << endl;
}

/** Generates call to the work function of the module.
  * @param o Stream, to which is output printed.
  */
void lang_cpp::generateWork(std::ostream& o, const string &indent, rpg_slot_module_t* module) {
	o << indent << "work (&" << module->name << ");" << endl;
}

/** Generates code for print module working time.
  * @param o Stream, to which code is printed.
  */
void lang_cpp::generatePrintTimes(std::ostream &o, const string &indent, rpg_slot_module_t *module) {
	o << indent << module->name << ".print_measurements (\"" OUTPUT_CONTEXT_SHARED "\");" << endl;
}

/** Generates code for printing module-spefific information
  * @param o Stream, to which code is printed.
  */
void lang_cpp::generatePrintConfig (std::ostream &o, const string &indent, rpg_slot_module_t *module) {
	o << indent << module->name << ".print_configuration ();" << endl;
}

rpg_str_t lang_java::getOutputMainFile() {
	return LANG_OUTPUT_MAIN_FILE_JAVA;
}
rpg_str_t lang_java::getInputTemplateFile() {
	return LANG_INPUT_TEMPLATE_FILE_JAVA;
}

/** In Java, module definition means loading module class using a module-specific classloader to isolate the modules */
void lang_java::generateDefinition(std::ostream& o, const string &indent,
        		rpg_slot_module_t* module) {
	/* First, derive the module class name with the first character in uppercase, because
	 * in the config file, modules are writen in lower case. Then add "Module".
	 */
	rpg_str_t package_name = module->type.getClassName();
	rpg_str_t class_name = package_name;
	if(class_name.length() > 1)	{
		class_name.replace(0, 1, 1, std::toupper(class_name[0]));
		class_name.append("Module");
	}

	/* Now, load the module class using a new instance of ModuleClassLoader (we don't need to store the
	 * classloader reference).
	 *
	 * Note that the first line relies on the fact that the indent string already contains the static variable type
	 * declaration (to cope with the lack of preprocessing in Java). */
	o << indent << "<Module> " << module->name << "Class; " << endl;
	/* This is a bit ugly since we have to catch the exception. Having just one static { } block would need two expand macros
	 * and functions instead of one. */
	o << "\tstatic { " << endl << "\t\ttry {" << endl;
	o << "\t\t\t" << module->name << "Class = (Class <Module>)" <<
			"new ModuleClassLoader().loadClass(\"rpg.modules." << package_name << "." << class_name << "\");" << endl;
	o << "\t\t} catch (ClassNotFoundException e) {" << endl << "\t\t\tthrow new RuntimeException (e);"
			<< endl << "\t\t}" << endl << "\t}" << endl;
};

/** Generates a declaration of a module, using a class loaded by the module-specific classloader. */
void lang_java::generateDeclaration(std::ostream& o, const string &indent,
		rpg_slot_module_t* module) {
	/*
	 * Example for a module Module1 of class something:
	 *
	 * Module Module1 = Module1Class.newInstance();
	 */
	o << indent << "Module " << module->name << ";" << endl;
	/* This is a bit ugly since we have to catch the exception. Having just one try-catch block would need two expand macros
	 * and functions instead of one. */
	o << indent << "try {" << endl;
	o << indent << "\t" << module->name << " = " << module->name << "Class.newInstance();" << endl;
	o << indent << "} catch (Exception e) {" << endl;
	o << indent << "\tthrow new RuntimeException (e);" << endl;
	o << indent << "}" << endl;
}

void lang_java::generateInitialization(std::ostream& o, const string &indent,
		rpg_slot_module_t* module) {
	/* set module name */
	o << indent << module->name << ".setName(" << GEN_QUOTE(module->name);
	o << ");" << endl;

	/* generate initialization */
	if (module->slots != NULL) {
		/* generate initialization of this module */
		o << indent << module->name << ".init ( ";
		for (rpg_uint_t i = 0; i < module->number_of_slots; i++) {
			if (i != 0)
				o << ", ";
			o << module->slots[i]->name;			
		}
		o << " );" << endl;
	}
	else {
		o << indent << module->name << ".init ();" << endl;
	}
}

void lang_java::generateDeinitialization(std::ostream& o, const string &indent,
		rpg_slot_module_t* module) {
	// TODO: no java modules define any deinitialization yet
}


void lang_java::generateMeasure(std::ostream& o, const string &indent,
    rpg_slot_module_t* module) {
    /* only non-architectural modules should be measured in isolation */
    if (module->number_of_slots == 0) {
        o << indent << module->name << ".measureIsolated ();" << endl;
    } else {
    	/* architectural modules only when a flag is set */
    	o << indent << "if (Main.measureIsolatedArchitectures) " << module->name << ".measureIsolated ();" << endl;
    }
}

void lang_java::generateClear(std::ostream &o, const string &indent, rpg_slot_module_t *module) {
    o << indent << module->name << ".clearMeasurements ();" << endl;
}

void lang_java::generateWork(std::ostream& o, const string &indent,
	rpg_slot_module_t* module) {
	o << indent << module->name << endl;
}

void lang_java::generatePrintTimes(std::ostream &o, const string &indent, rpg_slot_module_t *module) {
	o << indent << module->name << ".printMeasurements (OUTPUT_CONTEXT_SHARED);" << endl;
}

void lang_java::generatePrintConfig (std::ostream &o, const string &indent, rpg_slot_module_t *module) {
	o << indent << module->name << ".printConfiguration ();" << endl;
}
