/** @file gener.cpp
 * Random program generator.
 *
 * This file provides implementation of architecture generator classes and
 * classes holding module information.
 */

#include "rand.h"
#include "common.h"
#include "gener.h"
#include "config.h"
#include "parser.h"
#include "lang.h"

#include <sstream>
#include <cassert>
#include <iostream>
#include <algorithm>
#include <list>

using namespace std;
using namespace rpg; 

/** A list of module parameters that are used only in generator */
static const rpg_str_t
generator_parameters_array[] = { ITEM_CLASS_NAME, ITEM_PROBABILITY, ITEM_SLOTS };

/** A set of module parameters that are used only in generator */
set<rpg_str_t>
rpg_formal_module_t::generator_parameters (generator_parameters_array,
		generator_parameters_array + sizeof(generator_parameters_array)/sizeof(rpg_str_t));

/* ---                 formal module class implementation           --- */

/** Creates new module type instance.
 * @param name_ Name of the module.
 * @param classname_ Class name of the module. Class name is real class type, modules
 * with differrent module name can have same class.
 * @param slots_ Number of the slots of the module - number of submodules,
 * which will this type of module call.
 * @param probability_ Probability of the occurence of the module in the
 * random module selection.
 * @param archGen_ Architecture generator that created this module
 */
rpg_formal_module_t::rpg_formal_module_t(const rpg_str_t& name_,
										 const rpg_str_t& classname_,
										 dataSource* slots_,
										 rpg_uint_t probability_,
										 architecture_generator& archGen_)
										 : archGen(archGen_), slots(slots_), expected_duration(0) {
    classname = classname_;
	name = name_;
	probability = probability_;
}

/** Returns Name of the module.
  * @return Name of the module.
  */
const rpg_str_t& rpg_formal_module_t::getName(void) {
	return name;
}

/** Returns class name of the module.
  * @return Class name of the module.
  */
const rpg_str_t& rpg_formal_module_t::getClassName(void) {
	return classname;
}

/** Returns number of the slots of the module.
  * @return Number of the slots.
  */
rpg_uint_t rpg_formal_module_t::getSlots(void) {
	return slots->getIntValue ();
}

/** Returns probability of the module.
  * @return probability of the module occurence.
  */
rpg_uint_t rpg_formal_module_t::getProbability(void) {
	return probability;
}

/** Returns reference to the archicture generator, which owns this module.
  * @return Storage reference.
  */
architecture_generator& rpg_formal_module_t::getArchGen(void) {
	return archGen;
}

/** Creates new instance of slot module from formal module
  * @param slot_count (default - autogenerated) number of submodules - generate_test_arch overrides this for the root module
  * @return Formal module.
  */
rpg_slot_module_t* rpg_formal_module_t::createModule(rpg_uint_t slot_count) {
	rpg_uint_t slot_number = (slot_count == (rpg_uint_t) -1) ? slots->getIntValue () : slot_count;
	rpg_str_t m_name	   = archGen.createModuleName();
	return new rpg_slot_module_t(*this, slot_number, m_name);
}

/** Create data source of a given configuration item of the module */
dataSource * rpg_formal_module_t::createItemDataSource (const rpg_str_t & item) {
	return sprovider.createSource (name, item);
}

/* ---                 slot module class implementation             --- */

/** Constructor of slot module.
  * @param type_ Formal type of the module (formal module class instance).
  * @param slot_count Number of the submodules.
  * @param name_ Name of the module.
  */
rpg_slot_module_t::rpg_slot_module_t(rpg_formal_module_t& type_, rpg_uint_t slot_count,
									 rpg_str_t& name_)
	: name(name_), type(type_),  slots (NULL), refcount (0)  {

	change_slot_count (slot_count);
	/* initially, inherit the expected duration from the formal module */
	expected_duration = type.expected_duration;
}

void
rpg_slot_module_t::change_slot_count (rpg_uint_t slot_count) {
	if (slots != NULL) {
		delete[] slots;
	}
	number_of_slots = slot_count;
	/* Initialize slot array */
	if (slot_count > 0) {
		slots = new rpg_slot_module_t*[slot_count];
		for (rpg_uint_t i = 0; i < slot_count; i++) {
			slots[i] = NULL;
		}
	} else {
		slots = NULL;
	}
}

/** Destructor, frees memory. */
rpg_slot_module_t::~rpg_slot_module_t() {
	if (slots != NULL) {
		/* we can't delete the submodules in non-tree architectures */
		delete[] slots;
	}
}

/** Generates XML definition of the current module and
  * all of its subnodes.
  *
  * Note that XML description of module parameters has to be generated in generateConfig() first,
  * before calling this method. This is needed for making the xml and config in sync wrt parameters
  * randomized at generation time.
  *
  * @param o Stream, to which is output printed.
  * @param oldStyleSlots Embed submodules instead of referencing them by id. @see architecture_generator::generateTreeXml
  */
void rpg_slot_module_t::generateXMLdef(std::ostream& o, bool oldStyleSlots) {
	o << "<module" << XML_ATTR ("name", type.getName ()) << XML_ATTR ("id", name)
		<< XML_ATTR ("classname", type.getClassName ());
	if (!oldStyleSlots) o << XML_ATTR ("refcount", refcount);
	o << ">";

	o << "<params>";
	o << xmlparamdef;
	o << "</params>";

	/* generate the submodules */
	if (slots != NULL) {
		for (rpg_uint_t i = 0; i < number_of_slots; i++) {
			if (oldStyleSlots) {
				slots[i]->generateXMLdef(o, true);
			} else {
				o << "<slot" << XML_ATTR("target-id",slots[i]->name) << " />";
			}
		}
	}

	o << "</module>";
}


/** A helper function to collect item names.
 *
 * @param source The source name::value map.
 * @param names The target name set.
 */
void collect_item_names (const rpg_source_item_map_t &source, set<string> &items)
{
    if (&source != NULL)
    {
        for (rpg_source_item_map_t::const_iterator i = source.begin () ; i != source.end () ; i ++)
        {
            items.insert (i->first);
        }
    }
}


/** Generates configuration parameters of the module (without submodules).
 *
 * In addition to the configuration output, the xml definition is stored in xmlparamdef.
 *
 * @param o Stream, to which is output printed.
 */
void rpg_slot_module_t::generateConfig(std::ostream& o) {
	/* generate configuration of this module */
	o << "# configuration parameters of module " << name << endl;

	// This is a somewhat inefficient solution. Since we do not know what parameters
	// a module has, we simply take all parameters present in the configuration file.
	// This might not look so bad, except it forces us to replicate all default
	// parameters as well ...

	// TODO This could be improved by copying at least the system default parameters
	// as default parameters and not as module specific parameters, but for that,
	// the code that dumps a parameter would have to be outside a module.

	rpg_source_item_map_t *this_group_items = sprovider.getItemMap (type.getName ());
	rpg_source_item_map_t *group_default_items = sprovider.getItemMap (GROUP_DEFAULT "-" + type.getName ());
	rpg_source_item_map_t *system_default_items = sprovider.getItemMap(GROUP_DEFAULT);

	// Merge the names of all items into one set.
    set<string> items;
    collect_item_names (*this_group_items, items);
    collect_item_names (*group_default_items, items);
    collect_item_names (*system_default_items, items);

    // Dump configuration parameters with collected names.
    if (items.size () > 0)
    {
        stringstream s;
        for (set<string>::iterator i = items.begin () ; i != items.end () ; i ++)
        {
            // Do not copy generator parameters.
            if (rpg_formal_module_t::generator_parameters.count (*i) == 0)
            {
                generateParam (o, *i, s);
            }
            xmlparamdef = s.str ();
        }
    }

	o << endl << endl;
}

/** Generates parameters of the single module to the config file and to the xml output stream.
 *
 * Both have to be generated at the same time once, so that parameters randomized at generation time
 * are in sync.
 *
 * @param o Output stream with configuration file.
 * @param param_name Name of the parameter.
 * @param xmlparamdef XML definition output stream.
 */
void
rpg_slot_module_t::generateParam(std::ostream& con, const rpg_str_t& param_name, std::ostream& xmlparamdef) {

	const rpg_str_t& module_class_name = type.getName ();
	rpg_data_source_t param(sprovider.createSource (module_class_name, param_name));

	con << name << GEN_DOUBLEDOTS << param_name << GEN_ASSIGN;
	xmlparamdef << "<param name=\"" << param_name << "\" ";

	param->generateConfigAndXML (con, xmlparamdef);

	xmlparamdef << "</param>";
}

//--------------------------------------------------------------------------
// Architecture Generator Base Class Implementation

/** Create architecture generator of a subclass defined by ITEM_ARCHITECTURE_GENERATOR config option.
 *
 * Defaults to 'tree' architecture when missing.
 */
architecture_generator *
architecture_generator::create_instance () {
	rpg_data_source_t dsGenName (sprovider.createSource(GROUP_CORE, ITEM_ARCHITECTURE_GENERATOR, true));
	rpg_str_t genName;

	genName = dsGenName->getStringValue ();

	if (genName == "tree") {
		architecture_generator * gen = new tree_architecture_generator ();
		gen->generateTreeXml = true;
		return gen;
	} else
	if (genName == "testing") {
		return new test_architecture_generator ();
	} else
	if (genName == "tree-new") {
		return new tree_architecture_generator ();
	} else
	if (genName == "bottom-up") {
		return new bottom_up_architecture_generator ();
	} else {
		error("Invalid value of configuration " ITEM_ARCHITECTURE_GENERATOR);
		throw INVALID_CONFIGURATION_OPTION;
	}
}

/** Loads module list from configuration file. If error is detected, exception 
  * thrown, or false is returned. Note: All architecture modules must have at
  * least one slot.
  * @return True, if load was successful.
  */
bool architecture_generator::load(void) {
	/* first, get the module name */
	rpg_data_source_t module_name_list(sprovider.createSource(GROUP_CORE, ITEM_MODULES));

	/* load all modules */
	for (rpg_uint_t i = 0; i < module_name_list->array_size(); i++) {
		/* get the module name and load its data */
		const rpg_str_t& name = (*module_name_list)[i];

		rpg_data_source_t slots (sprovider.createSource (name, ITEM_SLOTS));
		rpg_data_source_t probability (sprovider.createSource (name, ITEM_PROBABILITY));
		rpg_data_source_t real_classname (sprovider.createSource (name, ITEM_CLASS_NAME));

		/* check, if classname, slot number, probability are correctly specified */
		if (!probability->isInteger()) {
			error("Probability resource of module " + name + " is not integer");
			return false;
		}

		if (!slots->isInteger()) {
			error("Slots resource of module " + name + " is not integer");
			return false;
		}

		if (!real_classname->isString()) {
			error("Classname resource of module " + name + " is not string");
			return false;
		}

		rpg_uint_t i_probability = probability->getIntValue ();
		rpg_uint_t i_slots = slots->getIntValue ();
		rpg_str_t s_r_classname = real_classname->getStringValue ();

		/* create module and insert it into architecture or module list */
		rpg_formal_module_t* module = new rpg_formal_module_t(name, s_r_classname, slots.release(), i_probability, *this);
		if (i_slots == 0) {
			/* for non-architectural modules, read and set expected duration */
			rpg_data_source_t ds_expected_duration (
					sprovider.createSource (name, ITEM_EXPECTED_DURATION, true));

			module->expected_duration = ds_expected_duration->getIntValue ();

			modules.push_back(module);
			module_total_probability += i_probability;
		} else {
			architectures.push_back(module);
			arch_total_probability += i_probability;
		}
	}

	return true;
}

/** Creates unique module name among all modules created by this generator
  * @return Module name.
  */
rpg_str_t architecture_generator::createModuleName(void) {
	stringstream s;
	s << "Module" << name_id++;
	return s.str();
}

/** Constructor, which initializes name generator, random generator etc. */
architecture_generator::architecture_generator () :
	name_id (1),
	arch_total_probability (0),
	module_total_probability (0),
	root_module (NULL),
	generateTreeXml (false) {
}

/** Destructor, destroys all formal modules and architectures, and the slot modules
 * recursively from the root module.
 */
architecture_generator::~architecture_generator(void) {
	/* first delete generated modules, they have pointers to formal modules */
	for (std::vector <rpg_slot_module_t*> :: iterator it = generated_modules.begin();
			it != generated_modules.end(); it++) {
		delete *it;
	}

	/* now delete formal modules */
	std::vector<rpg_formal_module_t*> :: iterator iter;
	for (iter = modules.begin(); iter != modules.end(); iter++) {
		delete *iter;
	}

	for (iter = architectures.begin(); iter != architectures.end(); iter++) {
		delete *iter;
	}
}

/** Generates source file of the architecture using given template file.

  * @param o Output stream, on which is source code written.
  * @param i Input stream containing source code template.
  */
void
architecture_generator::generateSource (std::ostream &o, std::istream &i)
{
	// Copy the template file line by line, catching and expanding lines that contain generator macros.
    // The generation is simplified in that it assumes everything on the line before the macro is just
    // indentation (but that is not checked). Also, everything on the line after the macro is discarded
	// to allow compilable dummy macro values in the Java template.
	while (i)
	{
		string line;
		getline(i, line);

		size_t found_position;
		bool found_indication = false;

		// Note that the statement boundaries in the macros are tricky.

        #define GENERATE_FOR_ONE_MODULE(module,function) \
            lang_generator->generate##function (o, line.substr (0, found_position), module);

        #define GENERATE_FOR_ALL_MODULES(function) \
            for (std::vector<rpg_slot_module_t *>::iterator i = generated_modules.begin () ; i < generated_modules.end () ; i ++) \
                GENERATE_FOR_ONE_MODULE (*i, function)

        #define GENERATE_CONDITION(trigger) \
            if ((found_position = line.find (trigger)) != string::npos)

        #define GENERATE_CONDITION_FOR_ROOT_MODULE(trigger,function) \
            GENERATE_CONDITION (trigger) \
            { \
                GENERATE_FOR_ONE_MODULE (root_module, function) \
                found_indication = true; \
            }

        #define GENERATE_CONDITION_FOR_ALL_MODULES(trigger,function) \
            GENERATE_CONDITION (trigger) \
            { \
                GENERATE_FOR_ALL_MODULES (function) \
                found_indication = true; \
            }

		GENERATE_CONDITION_FOR_ALL_MODULES (GEN_DEFINE, Definition)
		GENERATE_CONDITION_FOR_ALL_MODULES (GEN_DECLARE, Declaration)
		GENERATE_CONDITION_FOR_ALL_MODULES (GEN_INIT, Initialization)
		GENERATE_CONDITION_FOR_ALL_MODULES (GEN_DEINIT, Deinitialization)
		GENERATE_CONDITION_FOR_ALL_MODULES (GEN_MEASURE, Measure)
		GENERATE_CONDITION_FOR_ALL_MODULES (GEN_CLEAR, Clear)
		GENERATE_CONDITION_FOR_ROOT_MODULE (GEN_WORK, Work)
		GENERATE_CONDITION_FOR_ALL_MODULES (GEN_PRINTTIMES, PrintTimes)
		GENERATE_CONDITION_FOR_ALL_MODULES (GEN_PRINTCONFIG, PrintConfig)

		if (!found_indication) o << line << endl;
	}
}

/** Generates configuration file for all modules generated in the architecture.
 *
 * @param o Output stream.
 */
void
architecture_generator::generateConfig (std::ostream& o) {
	for (vector<rpg_slot_module_t *>::iterator i = generated_modules.begin (); i != generated_modules.end (); i++) {
		(*i)->generateConfig(o);
	}
}

/** Generates XML definition of the current architecture.
 *
 * @see generateTreeXml
 *
 * @param o Output stream for the XML
 */
void
architecture_generator::generateXMLArchDef (std::ostream& o) {
	o << "<?xml version=\"1.0\" encoding=\"UTF-8\"?>" << endl;
	if (generateTreeXml) {
		o << "<architecture>" << endl;
		root_module->generateXMLdef(o, true);
	} else {
		o << "<architecture" << XML_ATTR("root-module-id",root_module->name) << ">" << endl;
		for (vector<rpg_slot_module_t *>::iterator i = generated_modules.begin (); i != generated_modules.end (); i++) {
			(*i)->generateXMLdef(o, false);
		}
	}
	o << "</architecture>" << endl;
}

rpg_slot_module_t*
architecture_generator::create_module (const std::vector <rpg_formal_module_t*> & candidates) {
	rpg_uint_t sum_prob = 0;

	for (std::vector <rpg_formal_module_t*> :: const_iterator iter = candidates.begin();
			iter != candidates.end(); iter++) {
		sum_prob += (*iter)->getProbability();
	}

	return create_module (candidates, sum_prob);
}

rpg_slot_module_t*
architecture_generator::create_module (const std::vector <rpg_formal_module_t*> & candidates, rpg_uint_t total_prob) {
	rpg_uint_t module_hit = rand_l () % total_prob;
	rpg_uint_t accumulated_prob = 0;

	for (std::vector <rpg_formal_module_t*> :: const_iterator iter = candidates.begin();
			iter != candidates.end(); iter++) {

		accumulated_prob += (*iter)->getProbability();

		/* check, if we hit the chosen module */
		if (accumulated_prob >= module_hit) {
			return (*iter)->createModule();
		}
	}

	assert (false);
}

/** Calculate and optionally set expected duration of an architectural module from all or subset of its submodules
 *
 * @param arch The architectural module processed
 * @param num_connected_slots Number of slots that are already connected. When set to -1 (default), all slots are considered.
 * @param set If true, set the expected_duration field to the calculated result.
 */
rpg_uint_t
architecture_generator::calculateExpectedDuration (rpg_slot_module_t * arch, rpg_int_t num_connected_slots, bool set) {

	const rpg_str_t & className = arch->type.getClassName();

	if (num_connected_slots == -1) {
		num_connected_slots = arch->number_of_slots;
	}

	rpg_uint_t duration = 0;
	if (className == ARCH_CLASS_SEQUENCE) {
		for (unsigned i = 0; i < num_connected_slots; i++) {
			duration += arch->slots[i]->expected_duration;
		}
	} else if (className == ARCH_CLASS_LOOP) {
		if (num_connected_slots == 1) {
			/* get the mean value of loop iterations of this loop module */
			rpg_data_source_t dsLoopCount (arch->type.createItemDataSource (CONFIG_LOOPCOUNT));
			rpg_uint_t meanLoopCount = dsLoopCount->getIntMean ();
			/* see if the expected duration of candidate times loop count exceeds max duration */
			duration = arch->slots[0]->expected_duration * meanLoopCount;
		}
		/* else leave the duration at 0 */
	} else if (className == ARCH_CLASS_BRANCH) {
		/* for branches, we take the maximum, not mean
		 * TODO: see if this is right... for loops we use mean count and not maximum, because e.g. geometric distribution
		 * has unlimited maximum, but very improbable, yet probability of executing a too long branch is much higher?
		 */
		for (unsigned i = 0; i < num_connected_slots; i++) {
			duration = max (duration, arch->slots[i]->expected_duration);
		}
	} else {
		error ("Unknown architectural module class in calculateExpectedDuration(): " + className);
		throw UNKNOWN_MODULE;
	}

	if (set) arch->expected_duration = duration;

	return duration;
}

/** Calculate expected durations of an architectural module from the already connected submodules, and a candidate module */
rpg_uint_t
architecture_generator::calculateExpectedDuration (rpg_slot_module_t * arch, rpg_int_t num_connected_slots, rpg_slot_module_t * candidate) {
	assert (num_connected_slots < (rpg_int_t) arch->number_of_slots);
	/*
	 * This is a bit hacky, but saves repeated code. We put the candidate temporarily as a new slot,
	 * and call the other method variant with num_connected_slots + 1
	 */
	/* just in case, backup the pointer */
	rpg_slot_module_t * slot_saved = arch->slots [num_connected_slots];

	arch->slots [num_connected_slots] = candidate;
	rpg_uint_t duration = calculateExpectedDuration (arch, num_connected_slots + 1, false);

	/* restore the pointer */
	arch->slots [num_connected_slots] = slot_saved;

	return duration;
}


//--------------------------------------------------------------------------
// Tree Architecture Generator Class Implementation

tree_architecture_generator::tree_architecture_generator () :
	arch_probabilities (sprovider.createSource(GROUP_CORE, ITEM_ARCH_PROB)),
	arch_prob_gen (sprovider.createSource(GROUP_CORE, ITEM_ARCH_PROB_GEN)) {
}

/** Generates tree of modules and architectures.
  * @return Slot module instance containing the tree.
  */
void tree_architecture_generator::generateArchitecture () {
	root_module = generate (0);
}

/** Generates subtree of given modules on level level.
 *
 * @param level Level of the subtree.
 * @return Pointer to the slot module generated on this level.
 */
rpg_slot_module_t* tree_architecture_generator::generate(rpg_uint_t level) {
	/* generate a module */
	rpg_slot_module_t * module = choose_random(level);

	/* put the module also to the list of all generated modules */
	generated_modules.push_back (module);
	/* all modules except the root one are called once */
	if (level > 0) module->refcount = 1;

	/* fill all its slots with random modules */
	for (rpg_uint_t i = 0; i < module->number_of_slots; i++) {
		module->slots[i] = generate (level + 1);
	}

	/* if it was an architectural module, calculate its expected duration */
	if (module->number_of_slots > 0) calculateExpectedDuration (module);

	/* return the pointer */
	return module;
}

/** Based on the level, it chooses new module and creates its instance.
  * @param level Current level of recursion while generating.
  * @return New instance of the module.
  */
rpg_slot_module_t* tree_architecture_generator::choose_random (rpg_uint_t level) {
	/* first choose, if we create architecture module or regular module */
	rpg_uint_t hit = arch_prob_gen->getIntValue ();
	rpg_uint_t border;
	rpg_uint_t borders_size = arch_probabilities->array_size();
	if (level >= borders_size) {
		border = arch_probabilities->access_int_list()[borders_size-1];
	} else {
		border = arch_probabilities->access_int_list()[level];
	}

	/* choose list, from which the module is selected, and a module from that list */
	if (hit <= border) {
		return create_module (architectures, arch_total_probability);
	} else {
		return create_module (modules, module_total_probability);
	}

	error("Internal error - insufficient modules or architectures probably");
	throw INTERNAL_ERROR;
}

//--------------------------------------------------------------------------
// Test Architecture Generator Class Implementation

test_architecture_generator::test_architecture_generator () {
	rpg_data_source_t _test_arch_root (sprovider.createSource(GROUP_CORE, ITEM_TEST_ARCH_ROOT));
	test_arch_root = _test_arch_root->getStringValue ();
	rpg_data_source_t _test_arch_dummy (sprovider.createSource(GROUP_CORE, ITEM_TEST_ARCH_DUMMY));
	test_arch_dummy = _test_arch_dummy->getStringValue ();
	rpg_data_source_t _test_arch_instances (sprovider.createSource(GROUP_CORE, ITEM_TEST_ARCH_INSTANCES));
	test_arch_instances = _test_arch_instances->getIntValue ();
}

/** Generates testing architecture ree of modules and architectures.
  * @return Slot module instance containing the tree.
  */
void test_architecture_generator::generateArchitecture () {
	/*
	 *  the root module will have as many slots as there are architectures and modules, times
	 *  the specified number of instances per module (this overrides the generated parameter value)
	 *  it's not a problem that the submodules will include another instances the root and dummy too
	 */
	rpg_uint_t root_slots = (architectures.size() + modules.size()) * test_arch_instances;

	/* create instance of the root architecture module as given by test_arch_root */
	for (std::vector<rpg_formal_module_t*> :: iterator iter = architectures.begin(); iter != architectures.end(); iter++) {
		if ((*iter)->getName() == test_arch_root) {
			root_module = (*iter)->createModule(root_slots);
			generated_modules.push_back (root_module);
			break;
		}
	}
	if (root_module == NULL) {
		error("Internal error - testing architecture root module not found");
		throw INTERNAL_ERROR;
	}

	/* get the dummy module factory as given by test_arch_dummy */
	rpg_formal_module_t* dummy_factory = NULL;
	for (std::vector<rpg_formal_module_t*> :: iterator iter = modules.begin(); iter != modules.end(); iter++) {
		if ((*iter)->getName() == test_arch_dummy) {
			dummy_factory = *iter;
			break;
		}
	}
	if (dummy_factory == NULL) {
		error("Internal error - testing architecture dummy module not found");
		throw INTERNAL_ERROR;
	}

	/* now populate all slots of root module */
	unsigned current_slot = 0;
	/* start with architectures */
	for (std::vector<rpg_formal_module_t*> :: iterator iter = architectures.begin(); iter != architectures.end(); iter++) {
		/* do as many instances of each module as told */
		for (unsigned i = 0; i < test_arch_instances; i++) {
			rpg_slot_module_t* arch = (*iter)->createModule();
			generated_modules.push_back (arch);
			/* populate the slots with dummies */
			for (unsigned j = 0; j < arch->number_of_slots; j++) {
				rpg_slot_module_t* dummy = dummy_factory->createModule();
				generated_modules.push_back (dummy);
				dummy->refcount = 1;
				arch->slots[j] = dummy;
			}
			if (arch->number_of_slots > 0) {
				calculateExpectedDuration (arch);
			}
			root_module->slots[current_slot] = arch;
			current_slot++;
		}
	}

	/* next the working modules */
	for (std::vector<rpg_formal_module_t*> :: iterator iter = modules.begin(); iter != modules.end(); iter++) {
		/* do as many instances of each module as told */
		for (unsigned i = 0; i < test_arch_instances; i++) {
			rpg_slot_module_t* module = (*iter)->createModule();
			assert(module->number_of_slots == 0);
			generated_modules.push_back (module);
			module->refcount = 1;
			root_module->slots[current_slot] = module;
			current_slot++;
		}
	}

	/* calculate the total expected duration of the root module */
	calculateExpectedDuration (root_module);
}

//--------------------------------------------------------------------------
// Bottom Up Architecture Generator Class Implementation

bottom_up_architecture_generator::bottom_up_architecture_generator () {
	rpg_data_source_t _workers_count (sprovider.createSource (GROUP_CORE, ITEM_WORKER_MODULES_COUNT));
	workers_count = _workers_count->getIntValue ();

	connected_module_reuse.reset (sprovider.createSource (GROUP_CORE, ITEM_CONNECTED_MODULE_REUSE));

	rpg_data_source_t _maximum_expected_duration (sprovider.createSource (GROUP_CORE, ITEM_MAX_EXPECTED_DURATION, true));
	maximum_expected_duration = _maximum_expected_duration->getIntValue ();
}

rpg_uint_t
bottom_up_architecture_generator::determineReusals (bool * module_reusals, rpg_uint_t num_slots) {
	unsigned newly_connected_count = 0;
	unsigned actual_num_slots = 0;
	for (rpg_uint_t i = 0; i < num_slots; i++) {
		// TODO: This means that a module connected early has higher chance of being connected again. Is it a problem?
		/*
		 * we add up current count of connected modules and the newly connected count to determine
		 * if reuse is possible at this step
		 */
		bool reuse = connected_modules.size() + newly_connected_count > 0 && connected_module_reuse->getBoolValue ();

		module_reusals [i] = reuse;
		if (!reuse) newly_connected_count++;

		/*
		 * If we just decided to connect more unconnected modules to the architecture, than are available
		 * then we end here. But we still allowed the slot to be an already-connected module.
		 */
		if (newly_connected_count > unconnected_modules.size ()) {
			break;
		} else {
			actual_num_slots++;
		}
	}

	return actual_num_slots;
}

/** Filter modules from already connected modules based on the type of the architectural module
 * that would connect them.
 */
void
bottom_up_architecture_generator::filterConnectedModules
	(rpg_slot_module_t * arch, rpg_uint_t slot_index, std::vector <rpg_slot_module_t *> & candidates) {

	const rpg_str_t archClass = arch->type.getClassName ();

	for (std::vector <rpg_slot_module_t*> :: iterator it = connected_modules.begin ();
			it != connected_modules.end (); it++) {

		if (archClass == ARCH_CLASS_BRANCH) {
			/*
			 * for branch modules, having the same module in multiple slots makes little sense
			 * so we check all previous slots
			 */
			if (std::find (arch->slots, arch->slots + slot_index, *it) != arch->slots + slot_index) {
				continue;
			}
		} else if (archClass == ARCH_CLASS_SEQUENCE) {
			/*
			 * for sequential modules, having the same module in adjacent slots is like loop with count=2
			 * so we check previous slot
			 */
			if (slot_index > 0 && arch->slots [slot_index-1] == *it) {
				continue;
			}
		}
		candidates.push_back (*it);
	}
}

/** Filter all candidates */
void
bottom_up_architecture_generator::filterCandidates (rpg_slot_module_t * arch, rpg_uint_t num_connected_slots,
		std::vector <rpg_slot_module_t *> & candidates, std::vector <rpg_slot_module_t *> & filtered_candidates) {
	const rpg_str_t archClass = arch->type.getClassName ();

	/* further filtering of candidates based on the selected module */
	for (std::vector <rpg_slot_module_t*> :: iterator
			it = candidates.begin (); it != candidates.end (); it++) {

		rpg_slot_module_t & candidate = **it;

		if (archClass == ARCH_CLASS_LOOP) {
			/* two nested loops are the same as one loop */
			if (candidate.type.getClassName () == ARCH_CLASS_LOOP) {
//				cout << "filtering nested loop" << endl;
				continue;
			}
		}
		/* filtering based on maximum expected durations */
		if (maximum_expected_duration > 0) {
			if (calculateExpectedDuration (arch, num_connected_slots, *it) > maximum_expected_duration) {
//				cout << "filtering " << archClass << " slot candidate due to duration " << endl;
				continue;
			}
		}
		filtered_candidates.push_back (*it);
	}
}

/** Create a new architectural module connecting exiting modules as its slots */
rpg_slot_module_t *
bottom_up_architecture_generator::createArchModule () {

	unsigned num_slots = 0;
	bool * module_reusals = NULL;
	rpg_slot_module_t * arch = NULL;

	while (true) {
		/* cleanup after previous attempt is easier this way */
		if (module_reusals) {
			delete [] module_reusals;
			module_reusals = NULL;
		}
		if (arch) {
			delete arch;
			arch = NULL;
		}

		/* create an architectural module */
		arch = create_module (architectures, arch_total_probability);

		/*
		 * determine for each slot if we will connect an unconnected or already connected module to it
		 * we have to do it *before* checking if the architecture doesn't have more slots than remaining unconnected modules
		 * because we first need to now how many unconnected modules will be used
		 */
		num_slots = arch->number_of_slots;
		module_reusals = new bool [num_slots];
		unsigned actual_num_slots = determineReusals (module_reusals, num_slots);

		bool success = true;
		/* if we had to use less slots than available */
		if (actual_num_slots < num_slots) {
//			cout << "too many slots" << endl;
			/* If the architecture module parameter allows less slots that were generated */
			if (arch->type.slots->getIntMin () <= actual_num_slots) {
				arch->change_slot_count (actual_num_slots);
				num_slots = actual_num_slots;
			} else {
				/* this module cannot have so few slots, we have to try again */
//				cout << "need new arch module" << endl;
				success = false;
			}
		}

		/* we need to track modules that were newly connect, in case of "backtracking" */
		std::list <rpg_slot_module_t *> newly_connected_modules;
		/* for each of its slots, connect it to something */
		if (success) for (rpg_uint_t i = 0; i < num_slots; i++) {

			const rpg_str_t archClass = arch->type.getClassName ();
			std::vector <rpg_slot_module_t*> candidates;

			/* determine if we connect to an already connected module */
			const bool reuse_connected = module_reusals [i];
			if (reuse_connected) {

				/* filter out modules that make no sense to reuse at this point */
				filterConnectedModules (arch, i, candidates);
			} else {
				candidates = unconnected_modules;
			}

			std::vector <rpg_slot_module_t *> filtered_candidates;
			filterCandidates (arch, i, candidates, filtered_candidates);

			/* if there are no candidates start over */
			if (filtered_candidates.size() == 0) {
//				cout << "no candidates left" << endl;
				success = false;
				break;
			}

			/* choose module from the candidates randomly */
			const rpg_uint_t cand_index = rand_l () % filtered_candidates.size ();
			rpg_slot_module_t * slot_module = filtered_candidates [cand_index];
			arch->slots[i] = slot_module;

			if (! reuse_connected) {
				/* the module is no longer unconnected */
				std::vector <rpg_slot_module_t *> :: iterator
					erase_it = find (unconnected_modules.begin (), unconnected_modules.end (), slot_module);
				assert (erase_it != unconnected_modules.end ());
				unconnected_modules.erase (erase_it);

				connected_modules.push_back (slot_module);
				newly_connected_modules.push_back (slot_module);
			}
		}

		if (success) {
			cout << ".";
			cout.flush ();
			break;
		} else {
			/* everything that was newly connected is no longer connected */
			for (std::list <rpg_slot_module_t *> :: iterator it = newly_connected_modules.begin ();
					it != newly_connected_modules.end (); it++) {

				cout << "<" << endl;

				std::vector <rpg_slot_module_t *> :: iterator
					erase_it = find (connected_modules.begin (), connected_modules.end (), *it);
				assert (erase_it != connected_modules.end ());
				connected_modules.erase (erase_it);

				unconnected_modules.push_back (*it);
			}
			cout << "X";
			cout.flush ();
			continue;
		}
	}

	delete[] module_reusals;
	return arch;
}

void
bottom_up_architecture_generator::generateArchitecture () {
	/* generate the given amount of the worker modules */
	for (rpg_uint_t i = 0; i < workers_count; i++) {
		rpg_slot_module_t * worker = create_module (modules, module_total_probability);
		generated_modules.push_back (worker);
		unconnected_modules.push_back (worker);
	}

	/* connect until everything but one module is connected */
	while (unconnected_modules.size () != 1) {

		rpg_slot_module_t * arch = createArchModule ();

		/* increase refcount for all submodules of the arch */
		for (rpg_uint_t i = 0; i < arch->number_of_slots; i++) {
			arch->slots [i]->refcount++;
		}

		calculateExpectedDuration (arch);

		generated_modules.push_back (arch);
		/* the architecture itself is unconnected, and cannot put it there earlier than now to prevent self-cycles */
		unconnected_modules.push_back (arch);
	}

	/* the only unconnected module (which is the last created architecture) becomes the root */
	root_module = unconnected_modules.front ();
}
