// Random number generator definitions
package rpg;

import java.util.concurrent.atomic.AtomicInteger;

/**
 * Contains a custom random number generator.
 *
 */
public class Rand {
	/// Multiplication constant of the random generator.
	private static final long RND_MULTIPLIER = 1664525;
	/// Additive constant of the random generator.
	private static final long RND_CONSTANT = 1013904223;

	private static long seed;

	private static AtomicInteger randLock = new AtomicInteger(0);
	private static final int RAND_LOCKED = 1;
	private static final int RAND_UNLOCKED = 0;

	/**
	 * Initializes the seed of the random number generator.
	 *
	 * @param seed_ a new seed to use
	 */
	@SuppressWarnings("empty-statement")
	public static void setSeed(long seed_) {
		while (randLock.compareAndSet(RAND_UNLOCKED, RAND_LOCKED)) {};

		seed = seed_;

		randLock.set(RAND_UNLOCKED);
	}

	/**
	 * Generates a random number from the range of C++ unsigned 4-byte integer
	 * It is supposed to be equivalent to unsigned long that has no defined
	 * size, so 4 bytes are chosen.
	 *
	 * @return the random number
	 */
	@SuppressWarnings("empty-statement")
	// FIXME: this has returned negative number!
	public static long randUint32 () {
		while (randLock.compareAndSet(RAND_UNLOCKED, RAND_LOCKED)) {};

		seed = (RND_MULTIPLIER * seed + RND_CONSTANT) % 0x100000000l;
		long retVal = seed;

		randLock.set(RAND_UNLOCKED);
		
		return retVal;
	}


	/**
	 * Generates a random floating point value in the range between 0 and 1.
	 *
	 * @return the random number
	 */
	public static double randDouble() {
		/* generate random number between 0 and 1 */
		double sum = 0.0;
		double iter = 1.0 / 2.0;

		long rnd = randUint32();

		for (int i = 0; i < 32; i++) {
			if ((rnd & 1) > 0) {
				sum += iter;
			}

			iter /= 2.0;
			rnd = rnd >> 1;
		}
		return sum;
	}


	/**
	 * Generates a random number from exponential distribution.

	 * @param lambda a parameter of the exponential distribution
	 * @return the random number
	 */
	public static double randExp(double lambda) {

		double rnd = randDouble();

		double argument = 1.0 - rnd; // Redundant?
		return - Math.log(argument) * lambda;
	}


	/**
	 * Generates random numbers from geometric distribution.
	 *
	 * We use the variant Y (as denoted on Wikipedia) = number of failures
	 * before the first success. This is more general, as the possible
	 * outcomes include also zero, which is applicable for loops.
	 * For loops, p is the probability of exiting the loop (thus including
	 * the test before the first iteration).
	 *
	 * @param p The probability of success on each trial.
	 * @return the random number
	 */
	public static double randGeom(double p) {

		double rnd = randDouble();

		double u = 1.0 - rnd; // Redundant?
		double d = 1.0 - p;
		/* the log/log formula calculates X, we want Y = X - 1 */
		return Math.ceil(Math.log(u) / Math.log(d) - 1.0);
	}
}
