package rpg;

import java.lang.management.ManagementFactory;
import java.lang.management.ThreadMXBean;


/**
 * A timer class for a global time.
 */
class MonotonicTimer {
	private	long startTime;
	/**
	 * Starts the timer.
	 */
	public	void start() {
		startTime = System.nanoTime();
	}
	/**
	 * Returns the time that occured after the last call to start().
	 */
	long stop() {
		long endTime = System.nanoTime();
		/* simple overflow check */
		if (endTime > startTime) {
			return endTime - startTime;
		}
		else {
			return startTime - endTime;
		}
	}
};

/**
 * A timer class for a thread time.
 */
class ThreadCputimeTimer {
	private	long startTime;
	/**
	 * Starts the timer.
	 */
	public	void start() {
		ThreadMXBean bean = ManagementFactory.getThreadMXBean();
		if(!bean.isCurrentThreadCpuTimeSupported())
			startTime = 0;
		else
			startTime = bean.getCurrentThreadCpuTime( );
	}
	/**
	 * Returns the time that occured after the last call to start().
	 */
	long stop() {
		long endTime;
		ThreadMXBean bean = ManagementFactory.getThreadMXBean();
		if(!bean.isCurrentThreadCpuTimeSupported())
			endTime = 0;
		else
			endTime = bean.getCurrentThreadCpuTime( );

		/* simple overflow check */
		if (endTime > startTime) {
			return endTime - startTime;
		} else {
			return startTime - endTime;
		}
	}
};

/**
 * A parent class for all the benchmarks.
 *
 */
public abstract class ModuleTimed extends ModuleBase {
	
	/** Mutex for printing. */
	protected static final Object printMutex = new Object ();
		
	/** The maximum number of measured values that should be stored */
	protected static int maxMeasuredValues = 0;
	
	/**
	 * Returns the maximum number of values that should be measured. After this
	 * number is reached, simulation begins to end.
	 */
	public static int getMaxMeasuredValues() {
		return maxMeasuredValues;
	}

	/**
	 * Sets the maximum number of values that should be measured. After this
	 * number is reached, simulation begins to end.
	 */
	public static void setMaxMeasuredValues(int maxMeasuredValues_) {
		maxMeasuredValues = maxMeasuredValues_;
	}

	/** The wall clock timer used to time an individual invocation. */
	protected MonotonicTimer mt;
	/** The thread clock timer used to time an individual invocation. */
	protected ThreadCputimeTimer tt;

	/** Storage for measured times of the wall clock. */
	protected long[] monotonicMeasure;
	
	/** Storage for measured times of the thread clock. */
	protected long[] threadtimeMeasure;
	
	/** Counter of stored measurements */
	protected int measuredValues;

	public ModuleTimed () {
		mt = new MonotonicTimer();
		tt = new ThreadCputimeTimer();

		monotonicMeasure = new long [maxMeasuredValues];
		threadtimeMeasure = new long [maxMeasuredValues];
		
		measuredValues = 0;
	}

	@Override
	public void printMeasurements (String context) {
		// Nest the methods properly.
		super.printMeasurements (context);

		// Printing should be serialized.
		synchronized (printMutex) {

			final String monotonicPrefix = ConfigReader.outputCreatePrefix (name, context, ConfigReader.OUTPUT_MONOTONIC_MEASURE); 
			for (int i = 0; i < measuredValues; i++) {
				ConfigReader.outputWithPrefix (monotonicPrefix, monotonicMeasure [i]);
			}

			final String threadtimePrefix = ConfigReader.outputCreatePrefix (name, context, ConfigReader.OUTPUT_THREADTIME_MEASURE); 
			for (int i = 0; i < measuredValues; i++) {
				ConfigReader.outputWithPrefix (threadtimePrefix, threadtimeMeasure [i]);
			}
		}
	}
	
	@Override
	public void clearMeasurements () {
		// Nest the methods properly.
		super.clearMeasurements ();
		
		measuredValues = 0;
	}
	
	@Override
	protected boolean beforeInternalWork (SessionStateHolder sessionState) {
		// Start the timers for the duration of the work.
		mt.start ();
		tt.start ();
		
		// Nest the callbacks properly.
		return super.beforeInternalWork (sessionState);
	}
	
	@Override
	protected void afterInternalWork (SessionStateHolder sessionState) {
		// Nest the callbacks properly.
		super.afterInternalWork (sessionState);
		
		// Collect the measurements.
		long mTime = mt.stop();
		long tTime = tt.stop();
	
	    // Measurements are not collected when buffers are full or when shutdown is in progress.
		// Results are printed after shutdown, collecting during shutdown could therefore
		// distort the results.
		if (!rpgTerminate && measuredValues < maxMeasuredValues) {
			monotonicMeasure [measuredValues] = mTime;
			threadtimeMeasure [measuredValues] = tTime;
			measuredValues ++;
		}
	}
}
