package rpg;

import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;

/** A classloader for isolation of modules.
 * 
 * Assumes that the module-specific classes are located in one directory.
 * Tries to load these first, with a fallback to default parent classloader,
 * which will load the shared classes without isolation.
 * When each module instance is loaded a different instance of this classloader,
 * the static data will be isolated. 
 * 
 * Based on http://tech.puredanger.com/2006/11/09/classloader/
 */
public class ModuleClassLoader extends URLClassLoader {

	/** To ease classloader instantiation calls (which is inserted by the rpg generator),
	 * we use constructor with no parameters and fix the path with module-specific classes
	 * here.
	 */
	static private URL[] modulesPath; 
		
	static {
		try {
			modulesPath = new URL[1];
			modulesPath[0] = new URL ("file:./modules/");
		} catch (MalformedURLException e) {
			// this should not really happen
			throw new RuntimeException (e);
		}
	}
	
	public ModuleClassLoader () {
		super (modulesPath);
	}

	@SuppressWarnings("unchecked")
	@Override
	synchronized public Class<?> loadClass (String name) throws ClassNotFoundException {
		// try to reuse already loaded classes first
		Class cls = findLoadedClass (name);
		
		if (cls == null) {
			// try to load the class locally
			try {
				cls = findClass (name);
			} catch (ClassNotFoundException e) {
				// delegate to parent
				cls = super.loadClass (name);
			}
		}
		
		return cls;
	}
	
	/* This has to be also overriden to be safe */
	@Override
	public URL getResource(String name) {
		// Try to find it locally first
		URL url = findResource(name);

		// If not, try parent
		if(url == null) {
			url = super.findResource(name);
		}

		return url;
	} 
}
