package rpg;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;


/** Provider of work locks for modules - all instances of a module object of the same name share a lock.
 * 
 * This is a singleton object.
 */
class LockProvider {
	
	static private Map<String, Lock> locks = new HashMap<String, Lock>();
	
	/** Get an lock for module of given name. Created on demand. */
	static synchronized Lock getLock (String moduleName) {
		Lock l = locks.get (moduleName);
		if (l == null) {
			l = new ReentrantLock();
			locks.put (moduleName, l);
		}
		return l;
	}
}

public abstract class ModuleBase implements Module {

	/** Pointer to the lock synchronizing the internal_work method
	 *
	 * Initialized and used only for modules annotated as @Synchronized
	 */
	protected Lock workLock = null;
	
	protected String name;
	
	/** Denotes whether the application is in the tear-down phase.
	 * 
	 * This variable controls whether measurements should be collected and
	 * whether further work should be performed. It is set by the worker manager
	 * and used both by the manager and the modules.
	 * 
	 * The variable is placed in this class due to dependencies.
	 */
	public static volatile boolean rpgTerminate = false;

	/** Set name and also get the module lock, if the module is annotated @Synchronized. 
	 * 
	 * @see rpg.Module#setName(java.lang.String)
	 */
	public final void setName (String name) {
		this.name = name;
		
		/* Now that we know the name, we can get the appropriate lock for this module, if it's synchronized. */
		if (this.getClass().isAnnotationPresent(Synchronized.class)) {
			workLock = LockProvider.getLock (name);
		}
	}

	/* (non-Javadoc)
	 * @see rpg.Module#getName()
	 */
	public final String getName () {
		return name;
	}	
	
	private final boolean isSynchronized () {
		return (workLock != null);
	}
	
	/* (non-Javadoc)
	 * @see rpg.Module#init()
	 */
	public abstract void init ();

	/** Since most modules are workers, we define this method here to avoid repeating.
	 * Only architectural modules should override it. 
	 * 
	 * @see rpg.Module#init(rpg.Module)
	 */
	public void init (Module... modules) {
		throw new AssertionError ("Called init (...) with submodules on a worker module");
	}

	/** Module workload. */
	protected abstract void internalWork (SessionStateHolder sessionState);

	/** The callback before the internal workload method is called. */
	protected boolean beforeInternalWork (SessionStateHolder sessionState) {
	    // Bail out if terminating.
	    // Makes terminating faster.
		return rpgTerminate;
	}
	
	/// The callback after the internal workload function is called.
	protected void afterInternalWork (SessionStateHolder sessionState) {}
	
	/* (non-Javadoc)
	 * @see rpg.Module#measuredWork(rpg.SessionStateHolder)
	 */
	public final void measuredWork (SessionStateHolder sessionState) {
		if (isSynchronized ()) {
			workLock.lock ();
		}

		try {
			// Execute whatever callbacks we have around the work.
			boolean bail_out = beforeInternalWork (sessionState);
			if (!bail_out) internalWork (sessionState);
			afterInternalWork (sessionState);

		} finally {
			if (isSynchronized ()) {
				workLock.unlock ();
			}
		}
	}
	
	/* (non-Javadoc)
	 * @see rpg.Module#measureIsolated()
	 */
	public final void measureIsolated () {
		// Query the duration of the warmup workload and the isolated workload from the configuration.
		int warmup = ConfigReader.getIntItem (ConfigReader.GROUP_APP,
				ConfigReader.ITEM_FREE_ROUND_COUNT);
		int cycles = ConfigReader.getIntItem (ConfigReader.GROUP_APP,
				ConfigReader.ITEM_MEASURE);
		
		SessionStateHolder sessionState = new SessionStateHolder (SessionState.RETURN_OK);
		
		// Execute the warmup workload.
		for (int i = 0; i < warmup; i++) {
			sessionState.state = SessionState.RETURN_OK; 
			measuredWork (sessionState);
		}

		// Discard the warmup measurements.
		clearMeasurements ();

		// Collect the isolated measurements
		for (int i = 0; i < cycles; i++) {
			sessionState.state = SessionState.RETURN_OK; 
			measuredWork (sessionState);
		}
		
		// Print the isolated measurements.
		printMeasurements (ConfigReader.OUTPUT_CONTEXT_ISOLATED);
	}

	/* (non-Javadoc)
	 * @see rpg.Module#printMeasurements(java.lang.String)
	 */
	public void printMeasurements (String context) {}

	/* (non-Javadoc)
	 * @see rpg.Module#clearMeasurements()
	 */
	public void clearMeasurements () {}
	
	@Override
	public void printConfiguration () {
		ConfigReader.output (name, ConfigReader.OUTPUT_CONTEXT_CONFIG, ConfigReader.OUTPUT_ITEM_SYNCHRONIZED,
				isSynchronized() ? 1 : 0);		
	}
}