package rpg;

import java.util.Vector;
/**
 * A wrapper class for a JNI interface for reading the generated config file.
 */
public class ConfigReader {
	/* namespaces */
	/** Core group in the configuration file. */
	public static final String GROUP_CORE =					"core";
	/** Application group in the configuration file. */
	public static final String GROUP_APP =					"app";

	/* values */
	/** Random seed configuration item. */
	public static final String ITEM_RANDOM_SEED =			"random-seed";
	/** Random configuration item. */
	public static final String ITEM_RANDOM =				"random";
	/** Module path configuration item. */
	public static final String ITEM_MODULE_PATH =			"module-path";
	/** Path separator configuration item. */
	public static final String ITEM_PATH_SEPARATOR =		"path-separator";
	/** Modules configuration item. */
	public static final String ITEM_MODULES =				"modules";
	/** Slot number configuration item. */
	public static final String ITEM_SLOTS =  				"slots";
	/** Probabilitz configuration item. */
	public static final String ITEM_PROBABILITY =    		"probability";
	/** Architecture probability configuration item. */
	public static final String ITEM_ARCH_PROB =				"arch-prob";
	/** Architecture probability generator configuration item. */
	public static final String ITEM_ARCH_PROB_GEN =			"arch-prob-gen";
	/** Generation path configuration item. */
	public static final String ITEM_GEN_PATH =				"output-path";
	/** Parameters configuration item. */
	public static final String ITEM_PARAMS =				"params";
	/** Real class name of the module. */
	public static final String ITEM_CLASS_NAME =			"classname";



	/** Minimum cycles configuration item. */
	public static final String ITEM_MIN_CYCLES =			"min-cycles";
	/** Minimum time configuration item. */
	public static final String ITEM_MIN_TIME =				"min-time";
	/** Number of rounds of module works before module time measure start. */
	public static final String ITEM_FREE_ROUND_COUNT =		"measure-warmup";
	/** Number of measure cycles. */
	public static final String ITEM_MEASURE =				"measure-count";
	/** Number of threads of application. */
	public static final String ITEM_THREAD_COUNT =			"thread-count";
	/** Client wait time expected value. */
	public static final String ITEM_CLIENT_WAIT_TIME_EX =
													"client-wait-time-usec-ex";
	/** Number of clients. */
	public static final String ITEM_CLIENT_COUNT =			"client-count";
	/** Time limit of the running application. */
	public static final String ITEM_TIME_LIMIT =			"time-limit";
	/** Maximum size of measured time storage */
	public static final String ITEM_VALUES_STORAGE_MAX_SIZE =
													"values-storage-max-size";

	/* output stuff */
	public static final String OUTPUT_SEPARATOR = ";";
	public static final String OUTPUT_CONTEXT_ISOLATED = "isolated";
	public static final String OUTPUT_CONTEXT_SHARED = "shared";
	public static final String OUTPUT_CONTEXT_CONFIG = "config";
	public static final String OUTPUT_MONOTONIC_MEASURE = "monotonic";
	public static final String OUTPUT_THREADTIME_MEASURE = "threadtime";
	public static final String OUTPUT_ITEM_SYNCHRONIZED = "synchronized";
	
	static
	{
		// load ConfigReader JNI implementation library
		System.loadLibrary( "ConfigReader" );
	}

	/**
	 * Load the config file in the memory. Needs to be run before retrieving
	 * any values by the other methods.
	 * @param config_file a path to the config file
	 */
	public static native void loadConfig(String config_file);

	/**
	 * Retrieves an integer entry from the config file.
	 * @param group a group identifier of the parameter
	 * @param item a name of the parameter
	 * @return the integer value of the entry
	 */
	public static native int getIntItem(String group, String item);

	/**
	 * Retrieves a string entry from the config file.
	 * @param group a group identifier of the entry
	 * @param item a name of the entry
	 * @return the String value of the entry
	 */
	public static native String getStringItem(String group, String item);

	/**
	 * Retrieves a value at specified index in a list of integers entry from
	 * the config file.
	 * @param group a group identifier of the entry
	 * @param item a name of the entry
	 * @param index a position of the value in the list entry
	 * @return the retrieved value
	 */
	public static native int getIntListItemEntry(String group, String item,
			int index);

	/**
	 * Retrieves a value at specified index in a list of strings entry from
	 * the config file.
	 * @param group a group identifier of the entry
	 * @param item a name of the entry
	 * @param index a position of the value in the list entry
	 * @return the retrieved value
	 */
	public static native String getStringListItemEntry(String group,
			String item, int index);

	/**
	 * Retrieves the number values in a list entry in the config file.
	 * @param group a group identifier of the list
	 * @param item a name of the list
	 * @return the number of values
	 */
	public static native int getListItemLength(String group, String item);

	/**
	 * Checks if the specified entry is defined in the config file.
	 * @param group a group identifier of the entry
	 * @param item a name of the entry
	 * @return true if it is not defined, false otherwise
	 */
	public static native boolean isItemVoid(String group, String item);

	/**
	 * Retrieves a list of integers entry from the config file.
	 * @param group a group identifier of the parameter
	 * @param item a name of the parameter
	 * @return the retrieved list of values
	 */
	public static Vector<Integer> getIntListItem(
			String group, String item) {
		Vector<Integer> intList = new Vector<Integer>();
		int listLength = getListItemLength(group, item);

		for(int i = 0; i < listLength; i++) {
			int valueToAdd = getIntListItemEntry(group, item, i);
			intList.add(valueToAdd);
		}
		intList.trimToSize();

		return intList;
	}

	/**
	 * Retrieves a list of strings entry from the config file.
	 * @param group a group identifier of the parameter
	 * @param item a name of the parameter
	 * @return the retrieved list of values
	 */
	public static Vector<String> getStringListItem(
			String group, String item) {
	Vector<String> stringList = new Vector<String>();
		int listLength = getListItemLength(group, item);

		for(int i = 0; i < listLength; i++) {
			String valueToAdd = getStringListItemEntry(group, item, i);
			stringList.add(valueToAdd);
		}
		stringList.trimToSize();

		return stringList;
	}
	
	/** A method for unified application output syntax  
	 * 
	 * TODO: rename ConfigReader, this is not exactly ConfigReader functionality
	 */
	public static void output (String group, String context, String item, long value) {
		System.out.println(group + OUTPUT_SEPARATOR + context + OUTPUT_SEPARATOR + item + OUTPUT_SEPARATOR + value);
	}
	
	/** Create prefix for faster printing of many values with single prefix
	 * 
	 * Doing this through API allows easy finding places in code to update when the format changes  
	 */
	public static String outputCreatePrefix (String group, String context, String item) {
		return group + OUTPUT_SEPARATOR + context + OUTPUT_SEPARATOR + item + OUTPUT_SEPARATOR;
	}

	/** Output with prefix previously created by outputCreatePrefix() */
	public static void outputWithPrefix (String prefix, long value) {
		System.out.print (prefix);
		System.out.println (value);		
	}
}
