#include "common.h"
#include "config.h"
#include "module.h"

#include <stdlib.h>
#include <sys/types.h>

#include <sys/times.h>

#include <memory>
#include <iostream>

using namespace std;
using namespace rpg;

/// Global mutex for locking threads.
/// Used for module initialization, deinitialization and isolated measurements.
static pthread_mutex_t exclusive_work_mutex;

/// Mutex for serialized printing
static pthread_mutex_t print_mutex;

/// Global barrier for synchronizing threads.
static pthread_barrier_t barrier;

unsigned int max_measured_values = 0;

/// Provider of mutexes for module - all instances of a module share a mutex
class mutex_provider_t {
private:
	/// Map of pointers to mutexes by module name
	std::map <rpg_str_t, pthread_mutex_t *> mutexes;
public:
	/// Get an initialized mutex for module of given name. Created on demand.
	/// Assumes module initialization mutex is locked
	pthread_mutex_t * get_mutex (const rpg_str_t & module_name) {
		// try to find existing mutex
		std::map <rpg_str_t, pthread_mutex_t *> :: iterator it = mutexes.find (module_name);
		if (it != mutexes.end ()) {
			return it->second;
		}
		
		// create and initialize a new mutex
		pthread_mutex_t * mutex = (pthread_mutex_t *) malloc (sizeof (pthread_mutex_t));
		if (mutex == NULL) throw std::bad_alloc ();

		if (pthread_mutex_init (mutex, NULL) != 0) {
			perror ("pthread_mutex_init");
			exit (EXIT_FAILURE);
		}

		mutexes [module_name] = mutex;
		return mutex;
	}

	~mutex_provider_t () {
		// destroy all created mutexes
		for (std::map <rpg_str_t, pthread_mutex_t *> :: iterator
				it = mutexes.begin (); it != mutexes.end (); it++) {

			pthread_mutex_t * & mutex = it->second;
			if (pthread_mutex_destroy (mutex) == 0) {
				free (mutex);
			} else {
				perror ("pthread_mutex_destroy");
				// we are already exiting, cannot do much else
			}
		}
		// let the map itself destroy at once
	}
};

/// static singleton instance of a mutex provider
static mutex_provider_t mutex_provider;

//--------------------------------------------------------------------------
// Helper Functions

int wait_barrier(void) {
	return pthread_barrier_wait(&barrier);
}

void lock_exclusive_mutex(void) {
	pthread_mutex_lock(&exclusive_work_mutex);
}

void unlock_exclusive_mutex(void) {
	pthread_mutex_unlock(&exclusive_work_mutex);
}


//--------------------------------------------------------------------------
// Timer Methods

void monotonic_timer::start(void) {
	start_time = getMonotonicTime();
}

measure_time_t monotonic_timer::stop(void) {
	measure_time_t end_time = getMonotonicTime();
	// A simple overflow check.
	if (end_time > start_time) {
		return end_time - start_time;
	} else {
		return start_time - end_time;
	}
}

void thread_cputime_timer::start(void) {
	start_time = getThreadCpuTime();
}

measure_time_t thread_cputime_timer::stop(void) {
	measure_time_t end_time = getThreadCpuTime();
	// A simple overflow check.
	if (end_time > start_time) {
		return end_time - start_time;
	} else {
		return start_time - end_time;
	}
}


//--------------------------------------------------------------------------
// Module Methods

/** Module constructor.
 */
module_base::module_base (const std::string &name)
: work_mutex (NULL), name (name)
{
}

/** Module destructor.
 *
 * The virtual destructor is provided in case the children need it.
 */
module_base::~module_base ()
{
}

void module_base::set_synchronized () {
	if (work_mutex == NULL) {
		work_mutex = mutex_provider.get_mutex (name);
	} else {
		error ("module_base::set_synchronized called twice");
		throw INTERNAL_ERROR;
	}
}

/** Initialize the module.
 *
 * This method is meant to connect the module to its children,
 * and to perform any other module-specific initialization.
 * The default implementation handles the common case when the
 * module has no children.
 */
void module_base::init (int count, ...)
{
	if (count != 0)
	{
		std::cerr << "Unexpected children passed to module initializer." << std::endl;
		exit (EXIT_FAILURE);
	}
}

/** Deinitialize the module.
 *
 * This method is meant to safely deinitialize the module. Calls should be
 * protected by the exclusive mutex. Protecting destructors would be less straightforward.
 * The default implementation does nothing.
 */
void module_base::deinit () {
}

/** Internal wrapper for workload that collects measurements.
 *
 * This is the method that should be called from outside the module to have it work.
 * Serialized modules are protected by their mutex.
 */
void module_base::measured_work (int &session_state)
{
	if (is_synchronized ()) {
		pthread_mutex_lock (work_mutex);
	}

	// Execute whatever callbacks we have around the work.
	bool bail_out = before_internal_work (session_state);
	if (!bail_out) internal_work (session_state);
	after_internal_work (session_state);

	if (is_synchronized ()) {
		pthread_mutex_unlock (work_mutex);
	}
}

/** A callback before internal work. */
bool module_base::before_internal_work (int &session_state)
{
    // Bail out if terminating.
    // Makes terminating faster.
    return (rpg_terminate);
};

/** A callback after internal work. */
void module_base::after_internal_work (int &session_state)
{
};

/** Measures the workload in an internal loop.
 *
 * This method should be invoked for the isolated module measurement,
 * exclusive_work_mutex should be locked when multiple threads are used.
 *
 * To catch situations where incorrect setup causes undue overhead,
 * the method also collects the total time spent in the user mode,
 * in the kernel mode, and waiting, as reported by the operating
 * system.
 */
void module_base::measure_isolated ()
{
	// Query the duration of the warmup workload and the isolated workload from the configuration.
	rpg_data_source_t swarmup (sprovider.createSource(GROUP_APP, ITEM_FREE_ROUND_COUNT));
	rpg_data_source_t scycles (sprovider.createSource(GROUP_APP, ITEM_MEASURE));
	unsigned int warmup = swarmup->getIntValue ();
	unsigned int cycles = scycles->getIntValue ();

	// Execute the warmup workload.
	for (unsigned int i = 0 ; i < warmup ; i++)
	{
	    int session_state = RETURN_OK;
	    measured_work (session_state);
	}

	// Discard the warmup measurements.
	clear_measurements ();

	// Collect the isolated measurements together with kernel timing.
	struct tms tm_start;
	clock_t real_start = times (&tm_start);
	for (unsigned int i = 0 ; i < cycles ; i++)
    {
	    int session_state = RETURN_OK;
	    measured_work (session_state);
    }
	struct tms tm_end;
	clock_t real_end = times (&tm_end);

	// Print the isolated measurements.
	print_measurements (OUTPUT_CONTEXT_ISOLATED);

	output (cout, name, OUTPUT_CONTEXT_ISOLATED, "tms_real", real_end - real_start);
	output (cout, name, OUTPUT_CONTEXT_ISOLATED, "tms_user", tm_end.tms_utime - tm_start.tms_utime);
	output (cout, name, OUTPUT_CONTEXT_ISOLATED, "tms_kern", tm_end.tms_stime - tm_start.tms_stime);
}

/** Clears the collected measurements. */
void module_base::clear_measurements () { };

/** Print the collected measurements. */
void module_base::print_measurements (const std::string &context) { };

/** Print module implementation-specific information useful for model transformations,
 * but not known at architecture generation time.
 *
 * This method is currently not virtual (to avoid redefinitions and symbol renaming)
 * since the only information printed is whether the module is synchronized, which
 * is known to the base class.
 */
void module_base::print_configuration () {
	output (cout, name, OUTPUT_CONTEXT_CONFIG, OUTPUT_SYNCHRONIZED, is_synchronized ());
}

//--------------------------------------------------------------------------
// Timed Module Methods

/** Module constructor.
 *
 * Reserves buffers for measurements.
 */
module_timed::module_timed (const std::string &name)
: module_base (name)
{
	// Allocate space for samples to avoid distortion during measurement.
	measured_times.reserve (max_measured_values);
}


/** Callback before internal work.
 *
 * This callback starts the timers.
 */
bool module_timed::before_internal_work (int &session_state)
{
	// Start the timers for the duration of the work.
	mt.start ();
	tt.start ();

	// Nest the callbacks properly.
	return (module_base::before_internal_work (session_state));
}


/** Callback after internal work.
 *
 * This callback stops the timers and records the sample.
 */
void module_timed::after_internal_work (int &session_state)
{
	// Nest the callbacks properly.
	module_base::after_internal_work (session_state);

	// Collect the measurements.
	measure_time_t mtime = mt.stop();
	measure_time_t ttime = tt.stop();

    // Measurements are not collected when buffers are full or when shutdown is in progress.
	// Results are printed after shutdown, collecting during shutdown could therefore
	// distort the results.
	if (!rpg_terminate && measured_times.size() < max_measured_values)
	{
		measured_times.push_back (pair<measure_time_t, measure_time_t>(mtime, ttime));
	}
}


void module_timed::clear_measurements ()
{
    // Nest the methods properly.
    module_base::clear_measurements ();

    measured_times.clear ();
}


void module_timed::print_measurements (const std::string &context)
{
	// Nest the methods properly.
	module_base::print_measurements (context);

	// Printing should be serialized.
	pthread_mutex_lock (&print_mutex);

	std::vector <pair <measure_time_t, measure_time_t> > :: iterator iter;
	for (iter = measured_times.begin () ; iter != measured_times.end () ; iter ++)
	{
		output (cout, name, context, OUTPUT_MONOTONIC_MEASURE, iter->first);
		output (cout, name, context, OUTPUT_THREADTIME_MEASURE, iter->second);
	}

	pthread_mutex_unlock (&print_mutex);
}


//--------------------------------------------------------------------------
// Reliable Module Methods

/** Module constructor.
 */
module_reliable::module_reliable (const std::string &name)
: module_timed (name)
{
    // Read the reliability related configuration.
	bail_on_failure.reset (sprovider.createSource (name, CONFIG_BAIL_ON_FAILURE));
}


/** Callback before internal work.
 *
 * This callback takes care of bailing out on failure.
 */
bool module_reliable::before_internal_work (int &session_state)
{
    bool bailing_due_to_failure = bail_on_failure->getBoolValue () && (session_state == RETURN_ERROR_EFFECTIVE);
    bool bailing_due_to_parent = module_timed::before_internal_work (session_state);
    return (bailing_due_to_failure || bailing_due_to_parent);
}


/** Callback after internal work.
 *
 * This callback updates the counters based on the return value.
 */
void module_reliable::after_internal_work (int &session_state)
{
	// Nest the callbacks properly.
	module_timed::after_internal_work (session_state);

	// Count the return values. Counts are not collected when shutdown is in progress.
	// Results are printed after shutdown, collecting during shutdown could therefore
	// distort the results.

	// Note that it is the session state that gets counted, session state only
	// matches the properties of the module when the module is run in isolation.

	if (!rpg_terminate)
	{
		switch (session_state)
		{
		    case RETURN_OK: counted_result_ok ++; break;
		    case RETURN_ERROR_LATENT: counted_result_error_latent ++; break;
		    case RETURN_ERROR_EFFECTIVE: counted_result_error_effective ++; break;
		}
	}
}


void module_reliable::clear_measurements ()
{
    // Nest the methods properly.
    module_timed::clear_measurements ();

    counted_result_ok = 0;
    counted_result_error_latent = 0;
    counted_result_error_effective = 0;
}


void module_reliable::print_measurements (const std::string &context)
{
	// Nest the methods properly.
	module_timed::print_measurements (context);

	// Printing should be serialized.
	pthread_mutex_lock (&print_mutex);

	output (cout, name, context, OUTPUT_RESULT_OK_COUNT, counted_result_ok);
	output (cout, name, context, OUTPUT_RESULT_ERROR_LATENT_COUNT, counted_result_error_latent);
    output (cout, name, context, OUTPUT_RESULT_ERROR_EFFECTIVE_COUNT, counted_result_error_effective);

	pthread_mutex_unlock (&print_mutex);
}


//--------------------------------------------------------------------------
// Unreliable Module Methods

/** Module constructor.
 */
module_unreliable::module_unreliable (const std::string &name)
: module_reliable (name)
{
    // Read the unreliability related configuration.

	rpg_data_source_t internal_fault_source (sprovider.createSource (name, CONFIG_FAULT_PROBABILITY));
    internal_fault = internal_fault_source->getBoolValue ();

    error_internal_source.reset (sprovider.createSource (name, CONFIG_ERROR_INTERNAL_PROBABILITY));
    error_invocation_source.reset (sprovider.createSource (name, CONFIG_ERROR_INVOCATION_PROBABILITY));
    error_termination_source.reset (sprovider.createSource (name, CONFIG_ERROR_TERMINATION_PROBABILITY));

    failure_internal_source.reset (sprovider.createSource (name, CONFIG_FAILURE_INTERNAL_PROBABILITY));
    failure_invocation_source.reset (sprovider.createSource (name, CONFIG_FAILURE_INVOCATION_PROBABILITY));
    failure_termination_source.reset (sprovider.createSource (name, CONFIG_FAILURE_TERMINATION_PROBABILITY));

    // Initialize the rest.

    internal_error = false;
}


/** Callback before internal work.
 *
 * This callback takes care of generating errors and converting errors into failures.
 */
bool module_unreliable::before_internal_work (int &session_state)
{
    // Trigger a session error randomly.
    if (internal_fault && error_invocation_source->getBoolValue ())
        if (session_state < RETURN_ERROR_LATENT)
            session_state = RETURN_ERROR_LATENT;

    // Convert error to failure randomly.
    if (session_state == RETURN_ERROR_LATENT)
        if (failure_invocation_source->getBoolValue ())
            session_state = RETURN_ERROR_EFFECTIVE;

    return (module_reliable::before_internal_work (session_state));
}


/** Callback after internal work.
 *
 * This callback takes care of generating errors and converting errors into failures.
 */
void module_unreliable::after_internal_work (int &session_state)
{
    // Trigger a session error randomly.
    if (internal_fault && error_termination_source->getBoolValue ())
        if (session_state < RETURN_ERROR_LATENT)
            session_state = RETURN_ERROR_LATENT;

    // Trigger an internal error randomly.
    if (internal_fault && error_internal_source->getBoolValue ())
    {
        internal_error = true;
    }

    // Convert internal error to failure randomly.
    if (internal_error)
    {
        if (failure_internal_source->getBoolValue ())
        {
            session_state = RETURN_ERROR_EFFECTIVE;
            internal_error = false;
        }
    }

    // Convert session error to failure randomly.
    if (session_state == RETURN_ERROR_LATENT)
        if (failure_termination_source->getBoolValue ())
            session_state = RETURN_ERROR_EFFECTIVE;

    // Nest the callbacks properly.
    module_reliable::after_internal_work (session_state);
}


//--------------------------------------------------------------------------
// Static Functions

void modules_init(unsigned int threadcount) {
	if (pthread_barrier_init(&barrier, NULL, threadcount) != 0) {
		perror("Cannot initialize barrier");
		exit(EXIT_FAILURE);
	};

	if (pthread_mutex_init(&exclusive_work_mutex, NULL) != 0) {
		perror("Cannot initialize mutex");
		exit(EXIT_FAILURE);
	};

	if (pthread_mutex_init(&print_mutex, NULL) != 0) {
		perror("Cannot initialize mutex");
		exit(EXIT_FAILURE);
	};
}

void modules_destroy(void) {
	pthread_barrier_destroy(&barrier);
	pthread_mutex_destroy(&exclusive_work_mutex);
	pthread_mutex_destroy(&print_mutex);
}
