#include "common.h"
#include "config.h"
#include "gener.h"
#include "parser.h"
#include "rand.h"

#include <cassert>
#include <iostream>
#include <sstream>
#include <limits>
#include <algorithm>
#include <numeric>

#include <math.h>

/// typedefs used only in this file
namespace rpg {
	/** Map str-str. */
	typedef std::map<rpg_str_t, rpg_str_t> rpg_name_str_map;
	/** Map str-int. */
	typedef std::map<rpg_str_t, rpg_int_t> rpg_name_int_map;
	/** Map str-float. */
	typedef std::map<rpg_str_t, rpg_float_t> rpg_name_float_map;
	/** Map str-string list */
	typedef std::map<rpg_str_t, rpg_list_str_t> rpg_name_list_str_map;
	/** Map str-int list */
	typedef std::map<rpg_str_t, rpg_list_int_t> rpg_name_list_int_map;
	/** Map str-float list */
	typedef std::map<rpg_str_t, rpg_list_float_t> rpg_name_list_float_map;
}

using namespace rpg;
using namespace std;

/// Random generator lock
volatile int rnd_lock = 0;

/** Class provider singleton. */
dataClassProvider cprovider;
/** Data source provider singleton. */
dataSourceProvider sprovider (&cprovider);

//--------------------------------------------------------------------------
// Templates used for generating configuration and XML from data sources

/** Print parameter value to config file, depending on its type */
template <typename param_t>
static void printParamValue (std::ostream& con, param_t & param) {
	con << param;
}

/** String parameters are printed with quotes */
template <>
void printParamValue (std::ostream& con, const rpg_str_t & param) {
	con << GEN_QUOTE(param);
}

template <>
void printParamValue (std::ostream& con, rpg_str_t & param) {
	con << GEN_QUOTE(param);
}

/** Generate XML and config of simple valued parameters */
template <typename param_t>
static void printSimpleParam (std::ostream& con, std::ostream& xml,
		const char * className, const param_t & value) {

	con << className << GEN_PARAMBEGIN << CONFIG_COMMON_VALUE << GEN_ASSIGN;
	printParamValue (con, value);
	con << GEN_PARAMEND << endl;

	xml << XML_ATTR("classname", className) << ">";
	xml << "<attribute" << XML_ATTR("name", CONFIG_COMMON_VALUE) << ">";
	xml << XML_ITEM(value) << "</attribute>";
}

/** Generate XML and config of simple parameters (integers, strings, floats) */
template <typename param_t>
static void printParameterMap (std::map <rpg_str_t, param_t> & map,
		std::ostream& con, std::ostream & xml, const char * sep) {

	for (typename std::map <rpg_str_t, param_t> :: iterator
			iter = map.begin(); iter != map.end (); iter++) {

		con << sep << iter->first << GEN_ASSIGN;
		printParamValue (con, iter->second);
		sep = GEN_SEP;

		xml << "<attribute" << XML_ATTR("name", iter->first) << ">";
		xml << XML_ITEM(iter->second) << "</attribute>";
	}
}

/** Generate XML and config of list parameters (lists of integers, strings, floats) */
template <typename param_t>
static void printParameterMap (std::map <rpg_str_t, std::vector <param_t> > & map,
		std::ostream& con, std::ostream & xml, const char * sep) {

	for (typename std::map <rpg_str_t, std::vector <param_t> > :: iterator
			iter = map.begin(); iter != map.end(); iter++) {

		xml << "<attribute" << XML_ATTR ("name", iter->first) << ">";
		con << sep << iter->first << GEN_ASSIGN GEN_LISTBEGIN;
		sep = GEN_SEP;

		/* print all of the items of the list */
		std::vector <param_t> & list = iter->second;

		const char * sep2 = "";

		for (typename std::vector <param_t> :: iterator
				iter2 = list.begin(); iter2 != list.end(); iter2++) {

			xml << XML_ITEM(*iter2);
			con << sep2;
			printParamValue (con, *iter2);
			sep2 = GEN_SEP;
		}

		xml << "</attribute>";
		con << GEN_LISTEND;
	}
}

//--------------------------------------------------------------------------
// Random Geometric Distribution Data Source

/** Random geometric distribution data source holder class. */
class randomGeomDataSourceHolder : public dataSourceHolder {
private:
	friend class randomGeomDataSource;
	/** expected value of the geometric distribution*/
	rpg_int_t ex;
public:
	randomGeomDataSourceHolder (dataClassProvider * _provider) :
		dataSourceHolder (_provider), ex (-1) {
	}

	virtual dataSourceHolder* create () {
		return new randomGeomDataSourceHolder (provider);
	}

	/* Setting of parameter */
	virtual void setParam(const rpg_str_t& name, const rpg_int_t& value_) {
		if (name == CONFIG_COMMON_EX) {
			if (value_ > 0) {
				ex = value_;
			} else {
				error("Mean value must be greater than zero", line_number);
			}
		} else {
			/* error occurred - we are setting invalid parameter */
			rpg_str_t err_message = "Trying to set INT parameter '" + name + "' in source '" + getName() + "'.";
			error(err_message, line_number);
		}
	}

	/* Name redefinition */
	virtual const char* getName(void) const {
		return CLASS_RND_GEOM_NAME;
	}

	/* Data source constructor */
	virtual dataSource* createSource(void); 

	/* Check, if list contains at least one element */
	virtual bool checkData(void) {
		if (ex <= 0) {
			error("expected value of the geometric distribution must be positive!", line_number);
			return false;
		} else {
			return true;
		}
	}
};

/** Random geometric distribution data source class. */
class randomGeomDataSource : public dataSource {
private:
	/** expected value of the geometric distribution*/
	rpg_int_t ex;
	/** p parameter of the geometric distribution*/
	long double p;
public:
	/** Constructor.
	  * @param holder Holder, from which we want to create this data source.
	  */
	randomGeomDataSource(randomGeomDataSourceHolder* holder) : dataSource(holder) {
		/* set values */
		ex = holder->ex;
		p = 1.0 / ((long double)(ex + 1));
	}

	/* Get an integer value */
	virtual rpg_int_t getIntValue() {
		/* p must be greater than zero */
		assert (p > 0);

		return (rpg_int_t)rand_geom(p);
	}

	virtual rpg_int_t getIntMin () {
		return 0;
	}

	virtual rpg_int_t getIntMax () {
		return std::numeric_limits <rpg_int_t>::max();
	}

	virtual rpg_int_t getIntMean () {
		return ex;
	}

	/* some function redefinitions */
	virtual bool isConstant(void)    { return false; }
	virtual bool isInteger(void)	 { return true;  }
	virtual bool isVoid(void)        { return false; }

	/* printer */
	virtual rpg_str_t toString(void) {
		stringstream print_buffer;
		print_buffer << "<random number from geometric distribution [ p=" << p
			<< " ]>";
		return print_buffer.str();
	}
};

dataSource* randomGeomDataSourceHolder::createSource(void) {
	return new randomGeomDataSource(this);
}


//--------------------------------------------------------------------------
// Random Interval Data Source

/** Data holder for random interval. Represents random interval data type. */
class randomIntervalDataSourceHolder : public dataSourceHolder {
private:
	friend class randomIntervalDataSource;
	/** List of interval's lower bounds. */
	rpg_list_int_t mins;
	/** List of interval's upper boun. */
	rpg_list_int_t maxs;
public:
	randomIntervalDataSourceHolder (dataClassProvider * _provider) :
		dataSourceHolder (_provider) {
	}

	virtual dataSourceHolder * create () {
		return new randomIntervalDataSourceHolder (provider);
	}

	/* Setting of parameter */
	virtual void setParam(const rpg_str_t& name, const rpg_list_int_t& value_) {
		if (name == CONFIG_COMMON_MIN) {
			mins = value_;
		} else {
			if (name == CONFIG_COMMON_MAX) {
				maxs = value_;
			} else {
				/* error occurred - we are setting invalid parameter */
				rpg_str_t err_message = "Trying to set INT ARRAY parameter '" + name + "' in source '" + getName() + "'.";
				error(err_message, line_number);
			}
		}
	}

	/* Name redefinition */
	virtual const char* getName(void) const {
		return CLASS_RANDOM_INTERVAL_NAME;
	}

	/* Data source constructor */
	virtual dataSource* createSource(void);

	/* Check, if list contains at least one element */
	virtual bool checkData(void) {
		if (mins.size() == 0) {
			error("Lower bounds of intervals must have at least one element!", line_number);
			return false;
		} 

		if (maxs.size() == 0) {
			error("Upper bounds of intervals must have at least one element!", line_number);
			return false;
		} 

		if (mins.size() != maxs.size()) {
			error("Lower and upper bounds of intervals must have same size!", line_number);
			return false;
		}

		return true;
	}
};

/** Data source for the random interval. */
class randomIntervalDataSource : public dataSource {
private:
	/** List of interval's lower bounds. */
	rpg_list_int_t mins;
	/** List of interval's upper bounds. */
	rpg_list_int_t maxs;
public:
	/** Constructor.
	  * @param holder Holder, from which we want to create this data source.
	  */
	randomIntervalDataSource(randomIntervalDataSourceHolder* holder) : dataSource(holder) {
		/* set values */
		mins = holder->mins;
		maxs = holder->maxs;
	}

	/* Get integer interval (TODO: is this needed anywhere?) */
	virtual const rpg_interval_t getIntIterval () {
		/* list gained from holder will always have at least one element,
		 if holder is properly used */
		assert (mins.size() >= 1);
		assert (maxs.size() >= 1);

		/* if list contains only one item, return it, otherwise choose
		 random item */
		if (mins.size() != 1) {
			rpg_uint_t __max = mins.size() - 1;
			rpg_uint_t __min = 0;
			rpg_uint_t __index = (__builtin_abs(rand_l()) % (__max-__min + 1)) + __min;
			return rpg_interval_t(mins[__index], maxs[__index]);
		} else {
			return rpg_interval_t(mins[0], maxs[0]);
		}
	}

	virtual const rpg_list_int_t& access_interval_min_list(void) {
		return mins;
	};

	virtual const rpg_list_int_t& access_interval_max_list(void) {
		return maxs;
	};

	/* some function redefinitions */
	virtual bool isConstant(void)    { return false; }
	virtual bool isInterval(void)	 { return true;  }
	virtual bool isVoid(void)        { return false; }

	/* printer */
	virtual rpg_str_t toString(void) {
		stringstream print_buffer;
		print_buffer << "<random interval from interval list>";
		return print_buffer.str();
	}
};

dataSource* randomIntervalDataSourceHolder::createSource(void) {
	return new randomIntervalDataSource(this);
}



//--------------------------------------------------------------------------
// Random String from List Data Source

/** Random string from list data holder. */
class randomStrListDataSourceHolder : public dataSourceHolder {
private:
	friend class randomStrListDataSource;
	/** String values. */
	rpg_list_str_t values;

public:
	randomStrListDataSourceHolder (dataClassProvider * _provider) :
		dataSourceHolder (_provider) {
	}

	virtual dataSourceHolder * create () {
		return new randomStrListDataSourceHolder (provider);
	}

	/* Setting of parameter */
	virtual void setParam(const rpg_str_t& name, const rpg_list_str_t& value_) {
		if (name == CONFIG_COMMON_VALUES) {
			values = value_;
		} else {
			/* error occurred - we are setting invalid parameter */
			rpg_str_t err_message = "Trying to set STRING ARRAY parameter '" + name + "' in source '" + getName() + "'.";
			error(err_message, line_number);
		}
	}

	/* Name redefinition */
	virtual const char* getName(void) const {
		return CLASS_RANDOM_STR_LIST_NAME;
	}

	/* Data source constructor */
	virtual dataSource* createSource(void); 

	/* Check, if list contains at least one element */
	virtual bool checkData(void) {
		if (values.size() == 0) {
			error("List must have at least one element!", line_number);
			return false;
		} else {
			return true;
		}
	}
};

/** Random string data class. */
class randomStrListDataSource : public dataSource {
private:
	rpg_list_str_t values;
public:
	/** Constructor.
	  * @param holder Holder, from which we want to create this data source.
	  */
	randomStrListDataSource(randomStrListDataSourceHolder* holder) : dataSource(holder) {
		/* set values */
		values = holder->values;
	}

	/* Get a string value */
	virtual const rpg_str_t& getStringValue () {
		/* list gained from holder will always have at least one element,
		 if holder is properly used */
		assert (values.size() >= 1);

		/* if list contains only one item, return it, otherwise choose
		 random item */
		if (values.size() != 1) {
			rpg_uint_t max = values.size() - 1;
			rpg_uint_t min = 0;
			return values[(__builtin_abs(rand_l()) % (max-min + 1)) + min]; 	
		} else {
			return values[0];
		}
	}

	/* some function redefinitions */
	virtual bool isConstant(void)    { return false; }
	virtual bool isString(void)		 { return true;  }
	virtual bool isVoid(void)        { return false; }

	/* printer */
	virtual rpg_str_t toString(void) {
		stringstream print_buffer;
		print_buffer << "<random item from string list [";
		for (rpg_list_str_t :: iterator iter = values.begin(); iter != values.end(); iter++) {
			print_buffer << *iter << ", ";
		}
		print_buffer << "#]>";
		return print_buffer.str();
	}
};

dataSource* randomStrListDataSourceHolder::createSource(void) {
	return new randomStrListDataSource(this);
}


//--------------------------------------------------------------------------
// Random Integer from List Data Source

/** Random integer from integer list data source holder class. */
class randomIntListDataSourceHolder : public dataSourceHolder {
private:
	friend class randomIntListDataSource;
	/** List of integers, from which is random integer generated. */
	rpg_list_int_t values;

public:
	randomIntListDataSourceHolder (dataClassProvider * _provider) :
		dataSourceHolder (_provider) {
	}

	virtual dataSourceHolder * create(void) {
		return new randomIntListDataSourceHolder (provider);
	}

	/* Setting of parameter */
	virtual void setParam(const rpg_str_t& name, const rpg_list_int_t& value_) {
		if (name == CONFIG_COMMON_VALUES) {
			values = value_;
		} else {
			/* error occurred - we are setting invalid parameter */
			rpg_str_t err_message = "Trying to set INT ARRAY parameter '" + name + "' in source '" + getName() + "'.";
			error(err_message, line_number);
		}
	}

	/* Name redefinition */
	virtual const char* getName(void) const {
		return CLASS_RANDOM_INT_LIST_NAME;
	}

	/* Data source constructor */
	virtual dataSource* createSource(void); 

	/* Check, if list contains at least one element */
	virtual bool checkData(void) {
		if (values.size() == 0) {
			error("List must have at least one element!", line_number);
			return false;
		} else {
			return true;
		}
	}
};

/** Random integer from integer list data source class. */
class randomIntListDataSource : public dataSource {
private:
	rpg_list_int_t values;
public:
	/** Constructor.
	  * @param holder Holder, from which we want to create this data source.
	  */
	randomIntListDataSource(randomIntListDataSourceHolder* holder) : dataSource(holder) {
		/* set values */
		values = holder->values;
	}

	/* Get an integer value */
	virtual rpg_int_t getIntValue() {
		/* list gained from holder will always have at least one element,
		 if holder is properly used */
		assert (values.size() >= 1);

		/* if list contains only one item, return it, otherwise choose
		 random item */
		if (values.size() != 1) {
			rpg_uint_t max = values.size() - 1;
			rpg_uint_t min = 0;
			return values[(__builtin_abs(rand_l()) % (max-min + 1)) + min]; 	
		} else {
			return values[0];
		}
	}

	virtual rpg_int_t getIntMin () {
		return *min_element(values.begin(), values.end());
	}

	virtual rpg_int_t getIntMax () {
		return *max_element(values.begin(), values.end());
	}

	virtual rpg_int_t getIntMean () {
		// FIXME: does this check for overflows?
		rpg_int_t sum = accumulate (values.begin (), values.end (), 0);
		return sum / values.size();
	}

	/* some function redefinitions */
	virtual bool isConstant(void)    { return false; }
	virtual bool isInteger(void)	 { return true;  }
	virtual bool isVoid(void)        { return false; }

	/* printer */
	virtual rpg_str_t toString(void) {
		stringstream print_buffer;
		print_buffer << "<random item from integer list [";
		for (rpg_list_int_t :: iterator iter = values.begin(); iter != values.end(); iter++) {
			print_buffer << *iter << ", ";
		}
		print_buffer << "#]>";
		return print_buffer.str();
	}
};

dataSource* randomIntListDataSourceHolder::createSource(void) {
	return new randomIntListDataSource(this);
}


//--------------------------------------------------------------------------
// Random Range Data Source

/** Random integer from interval data source holder class. */
class randomRangeDataSourceHolder : public dataSourceHolder {
private:
	friend class randomRangeDataSource;
	/** Lower bound of the interval. */
	rpg_int_t min;
	/** Upper bound of the interval. */
	rpg_int_t max;

public:
	randomRangeDataSourceHolder (dataClassProvider * _provider) :
		dataSourceHolder (_provider), min (-1), max (-1) {
	}

	virtual dataSourceHolder * create () {
		return new randomRangeDataSourceHolder (provider);
	}

	/// Set data source parameters by name.
	virtual void setParam(const rpg_str_t& name, const rpg_int_t& value_) {
		if (name == CONFIG_COMMON_MIN) {
			min = value_;
		}
		else if (name == CONFIG_COMMON_MAX) {
            max = value_;
		}
		else {
            /* error occurred - we are setting invalid parameter */
            rpg_str_t err_message = "Trying to set INT parameter '" + name + "' in source '" + getName() + "'.";
            error(err_message, line_number);
        }
	}

	/// Data source name getter.
	virtual const char* getName(void) const {
		return CLASS_RANDOM_RANGE_NAME;
	}

	virtual dataSource* createSource(void); 

	/// Check whether the data source is configured properly.
	virtual bool checkData(void) {
		/* max must be non-negative */
		if (max < 0) {
			error("[max] parameter must be non-negative. ", line_number);
			return false;
		} 

		/* min must be non-negative */
		if (min < 0) {
			error("[min] parameter must be non-negative. ", line_number);
			return false;
		}

		/* we are selecting at least from two elements */
		if (min >= max) {
			error("Min must be strictly lesser than max parameter!", line_number);
			return false;
		}

		return true;
	}
};

/** Random integer from interval data source class. */
class randomRangeDataSource : public dataSource {
private:
	rpg_int_t min;
	rpg_int_t max;
public:
	/** Constructor.
	  * @param holder Holder, from which we want to create this data source.
	  */
	randomRangeDataSource(randomRangeDataSourceHolder* holder) : dataSource(holder) {
		/* set min and max */
		min = holder->min;
		max = holder->max;
	}

	/* Get an integer value */
	virtual rpg_int_t getIntValue () {
		return (__builtin_abs(rand_l()) % (max-min + 1)) + min; 	
	}

	virtual rpg_int_t getIntMin () {
		return min;
	}

	virtual rpg_int_t getIntMax () {
		return max;
	}

	virtual rpg_int_t getIntMean () {
		return min + (max - min) / 2;
	}

	/* some function redefinitions */
	virtual bool isConstant(void)    { return false; }
	virtual bool isInteger(void)	 { return true;  }
	virtual bool isVoid(void)        { return false; }

	/* printer */
	virtual rpg_str_t toString(void) {
		stringstream print_buffer;
		print_buffer << "<random-range generator [" << min << "," << max << "]>";
		return print_buffer.str();
	}
};

dataSource* randomRangeDataSourceHolder::createSource(void) {
	return new randomRangeDataSource(this);
}


//--------------------------------------------------------------------------
// Random Coin Toss Data Source

/// Random coin toss data source holder class.
class randomCoinTossDataSourceHolder : public dataSourceHolder
{
private:
    friend class randomCoinTossDataSource;
    /// Probability of TRUE result.
    rpg_float_t split_chance;

public:
    randomCoinTossDataSourceHolder (dataClassProvider *_provider)
    : dataSourceHolder (_provider), split_chance (-1)
    {
        split_chance = 0.5;
    }

    virtual dataSourceHolder * create () {
        return new randomCoinTossDataSourceHolder (provider);
    }

    /// Set data source parameters by name.
    virtual void setParam (const rpg_str_t &name, const rpg_float_t &value_)
    {
        if (name == CONFIG_COMMON_PROBABILITY) {
            split_chance = value_;
        }
        else {
            rpg_str_t err_message = "Trying to set FLOAT parameter '" + name + "' in source '" + getName() + "'.";
            error(err_message, line_number);
        }
    }

    /// Data source name getter.
    virtual const char* getName (void) const
    {
        return CLASS_RANDOM_COIN_TOSS_NAME;
    }

    virtual dataSource* createSource (void);

    /// Check whether the data source is configured properly.
    virtual bool checkData (void)
    {
        if ((split_chance < 0.0) || (split_chance > 1.0))
        {
            error("Coin toss probability has to be between 0 and 1. ", line_number);
            return false;
        }

        return true;
    }
};

/// Random coin toss data source class.
class randomCoinTossDataSource : public dataSource {
private:
    /// Probability of TRUE result.
    rpg_float_t chance_true;
public:
    /** Constructor.
      * @param holder Holder, from which we want to create this data source.
      */
    randomCoinTossDataSource(randomCoinTossDataSourceHolder* holder) : dataSource(holder)
    {
        chance_true = holder->split_chance;
    }

    virtual bool getBoolValue ()
    {
        return (rand_ld () < chance_true);
    }

    virtual bool isConstant(void)    { return false; }
    virtual bool isVoid(void)        { return false; }

    virtual rpg_str_t toString (void)
    {
        stringstream print_buffer;
        print_buffer << "<random-coin-toss generator [" << chance_true << "]>";
        return print_buffer.str();
    }
};

dataSource* randomCoinTossDataSourceHolder::createSource (void) {
    return new randomCoinTossDataSource(this);
}


//--------------------------------------------------------------------------
// Random Integer Data Source

/** Random integer data source holder class. */
class randomDataSourceHolder : public dataSourceHolder {
public:

	randomDataSourceHolder (dataClassProvider * _provider) :
		dataSourceHolder (_provider) {
	}

	virtual dataSourceHolder * create () {
		return new randomDataSourceHolder (provider);
	}

	/* Name redefinition */
	virtual const char* getName(void) const {
		return CLASS_RANDOM_NAME;
	}

	/* Data source constructor */
	virtual dataSource* createSource(void); 
};

/** Random integer data source class. */
class randomDataSource : public dataSource {
public:
	/** Constructor.
	  * @param holder Holder, from which we want to create this data source.
	  */
	randomDataSource(randomDataSourceHolder* holder) : dataSource(holder) {
	}

	/* Get an integer value */
	virtual rpg_int_t getIntValue() {
		return (rpg_uint_t)rand_l(); 	
	}

	virtual rpg_int_t getIntMin () {
		return 0;
	}

	virtual rpg_int_t getIntMax () {
		return std::numeric_limits <rpg_int_t>::max();
	}

	virtual rpg_int_t getIntMean () {
		return std::numeric_limits <rpg_int_t>::max() / 2;
	}

	/* some function redefinitions */
	virtual bool isConstant(void)    { return false; }
	virtual bool isInteger(void)	 { return true;  }
	virtual bool isVoid(void)        { return false; }

	/* printer */
	virtual rpg_str_t toString(void) {
		return "<random generator>";
	}
};

dataSource* randomDataSourceHolder::createSource(void) {
	return new randomDataSource(this);
}

//--------------------------------------------------------------------------
// Current Time Data Source

/** Time data source holder class. */
class timeDataSourceHolder : public dataSourceHolder {
public:

	timeDataSourceHolder (dataClassProvider * _provider) :
		dataSourceHolder (_provider) {
	}

	virtual dataSourceHolder * create () {
		return new timeDataSourceHolder (provider);
	}

	/* Name redefinition */
	virtual const char* getName(void) const {
		return CLASS_TIME_NAME;
	}

	/* Data source constructor */
	virtual dataSource* createSource(void); 
};

/** Time data source class. */
class timeDataSource : public dataSource {
private:
	rpg_int_t value;
public:
	/** Constructor.
	  * @param holder Holder, from which we want to create this data source.
	  */
	timeDataSource(timeDataSourceHolder* holder) : dataSource(holder) {
	}

	/* Get an integer value */
	virtual rpg_int_t getIntValue() {
		value = time(NULL);
		return value; 	
	}

	/* some function redefinitions */
	virtual bool isConstant(void)    { return false;  }
	virtual bool isInteger(void)	 { return true;  }
	virtual bool isVoid(void)        { return false; }

	/* printer */
	virtual rpg_str_t toString(void) {
		stringstream print_buffer;
		print_buffer << this->getIntValue ();
		return print_buffer.str();
	}
};

dataSource* timeDataSourceHolder::createSource(void) {
	return new timeDataSource(this);
}


//--------------------------------------------------------------------------
// Start Time Data Source

/** Start time data source holder class. */
class starttimeDataSourceHolder : public dataSourceHolder {
private:
	friend class starttimeDataSource;
	/** Start time value. */
	rpg_int_t value;

public:
	/** Constructor. */
	starttimeDataSourceHolder (dataClassProvider * _provider)
		: dataSourceHolder (_provider) {
		value = time(NULL);
	}

	/* This is the only exception where create is in fact clone, as there is just one startup time */
	virtual dataSourceHolder * create () {
		starttimeDataSourceHolder * holder = new starttimeDataSourceHolder (provider);
		holder->value = value;
		return holder;
	}

	/* Name redefinition */
	virtual const char* getName(void) const {
		return CLASS_STARTTIME_NAME;
	}

	/* Data source constructor */
	virtual dataSource* createSource(void); 
};

/** Start time data source class. */
class starttimeDataSource : public dataSource {
private:
	rpg_int_t value;
public:
	/** Constructor.
	  * @param holder Holder, from which we want to create this data source.
	  */
	starttimeDataSource(starttimeDataSourceHolder* holder) : dataSource(holder) {
		value = holder->value;
	}

	/* Get an integer value */
	virtual rpg_int_t getIntValue() {
		return value;	
	}

	/* some function redefinitions */
	virtual bool isConstant(void)    { return true;  }
	virtual bool isInteger(void)	 { return true;  }
	virtual bool isVoid(void)        { return false; }

	/* printer */
	virtual rpg_str_t toString(void) {
		stringstream print_buffer;
		print_buffer << value;
		return print_buffer.str();
	}
};

dataSource* starttimeDataSourceHolder::createSource(void) {
	return new starttimeDataSource(this);
}

template<typename param_t>
class simpleDataSource;

/** Integer data source class. */
template<typename param_t>
class simpleDataSourceHolder : public dataSourceHolder {
protected:
	friend class simpleDataSource<param_t>;

	/** Value of this data source. */
	param_t value;
	/** For checking that value was actually initialized */
	bool value_initialized;
public:
	simpleDataSourceHolder (dataClassProvider * _provider) :
		dataSourceHolder (_provider), value_initialized (false) {
	}

	virtual simpleDataSourceHolder * create () {
		return new simpleDataSourceHolder<param_t> (provider);
	}

	/* Setting of parameter */
	virtual void setParam(const rpg_str_t& name, const param_t& value_) {
		if (name == CONFIG_COMMON_VALUE) {
			value = value_;
			value_initialized = true;
		} else {
			/* error occurred - we are setting invalid parameter */
			rpg_str_t err_message = "Trying to set wrong type of parameter '" + name + "' in source '" + getName() + "'.";
			error(err_message, line_number);
		}
	}

	//TODO: change this to something that's compile-time error
	const char * getName () const { return "error"; };

	virtual bool checkData () {
		if (! value_initialized) {
			error("missing value parameter", line_number);
			return false;
		} else {
			return true;
		}
	}

	virtual simpleDataSource<param_t> * createSource ();
};

template<typename param_t>
class simpleDataSource : public dataSource {
protected:
	param_t value;
public:
	/** Constructor.
	  * @param holder Holder, from which we want to create this data source.
	  */
	simpleDataSource (simpleDataSourceHolder<param_t> * holder) : dataSource(holder) {
		value = holder->value;
	}

	/*
	 * We have to redefine everything that needs to be specialized by value type,
	 * by default just call the base class.
	 */
	virtual bool isConstant(void)    { return true;  }
	virtual bool isVoid(void)        { return false; }

	virtual bool isInteger ()		{ return false; }
	virtual bool isFloat ()			{ return false; }
	virtual bool isString ()		{ return false; }

	virtual bool isIntegerArray ()	{ return false; }
	virtual bool isFloatArray ()	{ return false; }
	virtual bool isStringArray ()	{ return false; }

	virtual rpg_int_t getIntValue() {	return dataSource::getIntValue (); }
	virtual rpg_int_t getIntMin () {	return dataSource::getIntMin (); }
	virtual rpg_int_t getIntMax () {	return dataSource::getIntMax (); }
	virtual rpg_int_t getIntMean () {	return dataSource::getIntMean (); }

	virtual rpg_float_t getFloatValue() {	return dataSource::getFloatValue (); }
	virtual rpg_float_t getFloatMin () {	return dataSource::getFloatMin (); }
	virtual rpg_float_t getFloatMax () {	return dataSource::getFloatMax (); }
	virtual rpg_float_t getFloatMean () {	return dataSource::getFloatMean (); }

	virtual const rpg_str_t & getStringValue ()	{ return dataSource::getStringValue (); }

	virtual bool getBoolValue ()	{ return dataSource::getBoolValue (); }

	virtual const rpg_list_int_t & access_int_list ()	{ return dataSource::access_int_list (); }
	virtual const rpg_list_float_t & access_float_list ()	{ return dataSource::access_float_list (); }
virtual const rpg_list_str_t & access_str_list ()	{ return dataSource::access_str_list (); }

	virtual const rpg_list_int_t & access_interval_min_list () { return dataSource::access_interval_min_list(); }
	virtual const rpg_list_int_t & access_interval_max_list () { return dataSource::access_interval_max_list(); }

	virtual rpg_uint_t array_size () { return dataSource::array_size (); }
	virtual const rpg_str_t & operator[] (const rpg_uint_t index) { return dataSource::operator[] (index); };

	/* printer */
	virtual rpg_str_t toString (void) {
		return rpg::toString (value);
	}
};

template<typename param_t>
simpleDataSource<param_t> * simpleDataSourceHolder<param_t>::createSource() {
	return new simpleDataSource<param_t> (this);
}

//--------------------------------------------------------------------------
// Integer Data Source

template <>
const char * simpleDataSourceHolder<rpg_int_t>::getName () const {
	return CLASS_INTEGER_NAME;
}

/* Get an integer value */
template <>
rpg_int_t simpleDataSource<rpg_int_t>::getIntValue() {
	return value;
}

template <>
rpg_int_t simpleDataSource<rpg_int_t>::getIntMin () {
	return value;
}

template <>
rpg_int_t simpleDataSource<rpg_int_t>::getIntMax () {
	return value;
}

template <>
rpg_int_t simpleDataSource<rpg_int_t>::getIntMean () {
	return value;
}

template <>
bool simpleDataSource<rpg_int_t>::isInteger () {
	return true;
}

//--------------------------------------------------------------------------
// Float Data Source

template <>
const char * simpleDataSourceHolder<rpg_float_t>::getName () const {
	return CLASS_FLOAT_NAME;
}

template<>
rpg_float_t simpleDataSource<rpg_float_t>::getFloatValue() {
	return value;
}

template<>
rpg_float_t simpleDataSource<rpg_float_t>::getFloatMin () {
	return value;
}

template<>
rpg_float_t simpleDataSource<rpg_float_t>::getFloatMax () {
	return value;
}

template<>
rpg_float_t simpleDataSource<rpg_float_t>::getFloatMean () {
	return value;
}

template<>
bool simpleDataSource<rpg_float_t>::isFloat () {
	return true;
}

//--------------------------------------------------------------------------
// Boolean Data Source

// TODO: having boolean constants in the grammar would make this easier
class booleanDataSourceHolder : public simpleDataSourceHolder<bool> {
public:
	booleanDataSourceHolder (dataClassProvider * provider)
		: simpleDataSourceHolder<bool> (provider) {
	}

	virtual booleanDataSourceHolder * create () {
		return new booleanDataSourceHolder (provider);
	}

	virtual const char * getName () const {
		return CLASS_BOOLEAN_NAME;
	}

	/* Setting of parameter */
	virtual void setParam (const rpg_str_t& name, const rpg_int_t& value_) {
		if (value_ != 0 && value_ != 1) {
			/* error occurred - we are setting invalid parameter */
			rpg_str_t err_message = "Setting wrong value (not 0 or 1 ) for boolean parameter '" + name + "' in source '" + getName() + "'.";
			error(err_message, line_number);
		}
		if (name == CONFIG_COMMON_VALUE) {
			value = value_;
			value_initialized = true;
		} else {
			/* error occurred - we are setting invalid parameter */
			rpg_str_t err_message = "Trying to set wrong type of parameter '" + name + "' in source '" + getName() + "'.";
			error(err_message, line_number);
		}
	}
};

template <>
bool simpleDataSource<bool>::getBoolValue () {
	return value;
}

//--------------------------------------------------------------------------
// String Data Source

template<>
const char* simpleDataSourceHolder<rpg_str_t>::getName () const {
	return CLASS_STRING_NAME;
}

template<>
const rpg_str_t& simpleDataSource<rpg_str_t>::getStringValue () {
	return value;
}

template<>
bool simpleDataSource<rpg_str_t>::isString () {
	return true;
}

template<>
rpg_str_t simpleDataSource<rpg_str_t>::toString () {
	return value;
}

//--------------------------------------------------------------------------
// Integer List Data Source

template<>
const char* simpleDataSourceHolder<rpg_list_int_t>::getName () const {
	return CLASS_INTEGERLIST_NAME;
}

template<>
bool simpleDataSource<rpg_list_int_t>::isIntegerArray () {
	return true;
}

template<>
const rpg_list_int_t& simpleDataSource<rpg_list_int_t>::access_int_list () {
	return value;
}
	
template<>
rpg_uint_t simpleDataSource<rpg_list_int_t>::array_size () {
	return value.size();
}

//--------------------------------------------------------------------------
// Float List Data Source

template<>
const char* simpleDataSourceHolder<rpg_list_float_t>::getName () const {
	return CLASS_FLOATLIST_NAME;
}

template<>
bool simpleDataSource<rpg_list_float_t>::isFloatArray() {
	return true;
}

template<>
const rpg_list_float_t & simpleDataSource<rpg_list_float_t>::access_float_list () {
	return value;
}
	
template<>
rpg_uint_t simpleDataSource<rpg_list_float_t>::array_size () {
	return value.size();
}

//--------------------------------------------------------------------------
// String List Data Source

template<>
const char* simpleDataSourceHolder<rpg_list_str_t>::getName(void) const {
	return CLASS_STRINGLIST_NAME;
}

template<>
bool simpleDataSource<rpg_list_str_t>::isStringArray () {
	return true;
}

template<>
const rpg_list_str_t & simpleDataSource<rpg_list_str_t>::access_str_list () {
	return value;
}
	
template<>
rpg_uint_t simpleDataSource<rpg_list_str_t>::array_size () {
	return value.size();
}

template<>
const rpg_str_t & simpleDataSource<rpg_list_str_t>::operator[] (const rpg_uint_t index) {
	return value[index];
}

//--------------------------------------------------------------------------
// Parameter data source

/** Parameter holder for module parameters */
class parameterDataSourceHolder : public dataSourceHolder {
private:

	/** Class name of the "nested" parameter */
	rpg_str_t nestedClassName;
	/** Int parameter map. */
	rpg_name_int_map  pint;
	/** Float parameter map */
	rpg_name_float_map pfloat;
	/** String parameter map. */
	rpg_name_str_map  pstr;
	/** Int list parameter map. */
	rpg_name_list_int_map plint;
	/** Float list parameter map. */
	rpg_name_list_float_map plfloat;
	/** String list parameter map. */
	rpg_name_list_str_map plstr;
	/** Holder of the "nested" source that this parameterDataSource describes */
	dataSourceHolder * nestedSourceHolder;

	/** Predicate for find_if used in applyParams() */
	static bool paramIsClassName (const rpg_param_value_t & param) {
		return (param.param_name == CONFIG_COMMON_CLASSNAME);
	}
public:

	parameterDataSourceHolder (dataClassProvider * _provider)
		: dataSourceHolder (_provider), nestedSourceHolder (NULL) {};

	~parameterDataSourceHolder() {}

	virtual dataSourceHolder * create () {
		return new parameterDataSourceHolder (provider);
	}

	/* Name redefinition */
	virtual const char* getName(void) const {
		return CLASS_PARAMETER;
	}

	/* Check data delegated to nested data source holder */
	virtual bool checkData(void) {
		return nestedSourceHolder->checkData ();
	}

	/* Data source factory */
	virtual dataSource* createSource ();

	virtual void applyParams(const rpg_params_t& params) {
		rpg_params_t :: const_iterator it_classname = find_if (params.begin (), params.end (), paramIsClassName);
		if (it_classname == params.end ()) {
			// TODO: unify error reporting
			error ("classname missing in parameter");
			throw INVALID_SOURCE_ACCESS;
		}

		nestedClassName = it_classname->v_string;
		nestedSourceHolder = provider->create (nestedClassName);

		rpg_params_t :: const_iterator iter;
		for (iter = params.begin(); iter != params.end(); iter++) {
			if (iter != it_classname) {
				applyParam (*iter);
				nestedSourceHolder->applyParam (*iter);
			}
		}
	}

	virtual void setParam(const rpg_str_t& name, const rpg_str_t& value) {
		pstr [name] = value;
	}

	virtual void setParam(const rpg_str_t& name, const rpg_int_t& value) {
		pint [name] = value;
	}

	virtual void setParam(const rpg_str_t& name, const rpg_float_t& value) {
	    pfloat [name] = value;
	}

	virtual void setParam(const rpg_str_t& name, const rpg_list_str_t& value) {
		plstr [name] = value;
	}

	virtual void setParam(const rpg_str_t& name, const rpg_list_int_t& value) {
		plint [name] = value;
	}

	virtual void setParam(const rpg_str_t& name, const rpg_list_float_t& value) {
	    plfloat [name] = value;
	}

	friend class parameterDataSource;
};

/** Parameter data source class. */
class parameterDataSource : public dataSource {
private:
	/** "Nested" data source described by this parameter data source */
	dataSource * nestedDataSource;
	/** Int parameter map. */
	rpg_name_int_map  pint;
	/** Float parameter map */
	rpg_name_float_map pfloat;
	/** String parameter map. */
	rpg_name_str_map  pstr;
	/** Int list parameter map. */
	rpg_name_list_int_map plint;
	/** Float list parameter map. */
	rpg_name_list_float_map plfloat;
	/** String list parameter map. */
	rpg_name_list_str_map plstr;

public:
	/** Constructor.
	  * @param holder Holder, from which we want to create this data source.
	  */
	parameterDataSource(parameterDataSourceHolder* holder) : dataSource(holder) {
		/* set values */
		pstr = holder->pstr;
		pint = holder->pint;
		pfloat = holder->pfloat;
		plstr = holder->plstr;
		plint = holder->plint;
		plfloat = holder->plfloat;

		nestedDataSource = holder->nestedSourceHolder->createSource ();

		nestedDataSource->setIdentifiers (group, item);
	}

	virtual ~parameterDataSource() {};

	/* delegate the queries to the nested datasource */
	virtual bool isVoid ()        { return nestedDataSource->isVoid(); }
	virtual bool isConstant ()    { return nestedDataSource->isConstant (); }

	virtual bool isInteger ()     { return nestedDataSource->isInteger (); }
	virtual bool isFloat ()       { return nestedDataSource->isFloat (); }
	virtual bool isString ()      { return nestedDataSource->isString (); }

	virtual bool isIntegerArray ()  { return nestedDataSource->isIntegerArray (); }
	virtual bool isFloatArray ()    { return nestedDataSource->isFloatArray (); }
	virtual bool isStringArray ()   { return nestedDataSource->isStringArray (); }

	virtual bool isInterval ()      { return nestedDataSource->isInterval (); }

	virtual rpg_int_t getIntMin ()   { return nestedDataSource->getIntMin (); }
	virtual rpg_int_t getIntMax ()   { return nestedDataSource->getIntMax (); }
	virtual rpg_int_t getIntMean ()  { return nestedDataSource->getIntMean (); }

	virtual rpg_float_t getFloatMin ()   { return nestedDataSource->getFloatMin (); }
	virtual rpg_float_t getFloatMax ()   { return nestedDataSource->getFloatMax (); }
	virtual rpg_float_t getFloatMean ()  { return nestedDataSource->getFloatMean (); }

	virtual rpg_str_t toString(void) {
		stringstream print_buffer;
		print_buffer << "<parameter>";
		return print_buffer.str();
	}

	/** Generate configuration line and xml description of a parameter data source.
	 */
	virtual void generateConfigAndXML (std::ostream& con, std::ostream & xml) {

		const char * nestedClassName = nestedDataSource->getName();

		xml << XML_ATTR("classname", nestedClassName) << ">";
		con << nestedClassName << GEN_PARAMBEGIN;

		const char * sep = "";

		printParameterMap (pstr, con, xml, sep);
		printParameterMap (pint, con, xml, sep);
		printParameterMap (pfloat, con, xml, sep);

		printParameterMap (plstr, con, xml, sep);
		printParameterMap (plint, con, xml, sep);
		printParameterMap (plfloat, con, xml, sep);

		con << GEN_PARAMEND << endl;
	}
};

dataSource* parameterDataSourceHolder::createSource () {
	return new parameterDataSource(this);
}

/*     
  ---            dataClassProvider implementation                       ---
																		*/

/** Initializator of class provider. Registers all classes,
  * which are resources. */
dataClassProvider::dataClassProvider() {
	/* Register all holder classes */
	registerClass (new simpleDataSourceHolder<rpg_int_t> (this));
    registerClass (new simpleDataSourceHolder<rpg_list_int_t> (this));
    registerClass (new simpleDataSourceHolder<rpg_float_t> (this));
   	registerClass (new simpleDataSourceHolder<rpg_list_float_t> (this));
   	registerClass (new booleanDataSourceHolder (this));
   	registerClass (new simpleDataSourceHolder<rpg_str_t> (this));
	registerClass (new simpleDataSourceHolder<rpg_list_str_t> (this));

	registerClass (new starttimeDataSourceHolder (this));
	registerClass (new timeDataSourceHolder (this));

	registerClass (new randomDataSourceHolder (this));
	registerClass (new randomRangeDataSourceHolder (this));
	registerClass (new randomIntListDataSourceHolder (this));
	registerClass (new randomStrListDataSourceHolder (this));
	registerClass (new randomIntervalDataSourceHolder (this));
	registerClass (new randomCoinTossDataSourceHolder (this));
	registerClass (new randomGeomDataSourceHolder (this));

	registerClass (new parameterDataSourceHolder (this));
}


/** Destructor of class provider. Clears resource class 
  * storage. */
dataClassProvider::~dataClassProvider() {
	/* erase all class providers */
	rpg_holder_map_t :: iterator iter;
	for (iter = classes.begin(); iter != classes.end(); iter++) {
		delete iter->second;
	}
	classes.clear();
}

/** Registers class of the sources to the class provider. 
  * @param className Class name of class, which has to be registered.
  * @param holder Holder (one instance) of this class.
  */
void dataClassProvider::registerClass(const rpg_str_t& className, 
									  dataSourceHolder* holder) {
	assert (classes.find(className) == classes.end());
	assert (holder != NULL);
	classes[className] = holder;
}

/** Registers class of the sources to the class provider. ClassName
  * is retrieved from holder instance.
  * @param holder Holder (one instance) of this class.
  */
void dataClassProvider::registerClass(dataSourceHolder* holder) {
	registerClass(holder->getName(), holder);
}

/** Returns new data source holder of class name className.
  * If class is not registered or does not exists, returns NULL.
  * @param className Resource class, of which instance has to be returned.
  * @return Instance of the resource class, or NULL, if it is not registered,
  * or does not exists, or there is not enough memory to allocate it.
  */
dataSourceHolder* dataClassProvider::create(const rpg_str_t& className) {
	if (classes.find(className) != classes.end()) {
		return classes[className]->create ();
	} else {
		return NULL;
	}
}

/*     
  ---            dataSourceProvider implementation                       ---
																		 */

/** Source provider destructor. Destroys all resource holders. 
  * After call of the destructor, it is invalid to access any
  * source retrieved by this class.
  */
dataSourceProvider::~dataSourceProvider() {
	/* erase all source providers */
	rpg_source_group_map_t::iterator iterGroups;
	for (iterGroups = sources.begin(); iterGroups != sources.end(); iterGroups++) {
		rpg_source_item_map_t * items = iterGroups->second;
		rpg_source_item_map_t::iterator iterItems;
		for (iterItems = items->begin(); iterItems != items->end(); iterItems++) {
			delete iterItems->second;
		}
		items->clear();
		delete items;
	}
	sources.clear();
}

/** Locates a source holder from group identifier and item identifier.
 *
 * Used internally to create source.
 *
 * @return Data source holder for the specified group::item or NULL.
 */
dataSourceHolder *dataSourceProvider::locateSourceHolder (const rpg_str_t &group, const rpg_str_t &item)
{
    rpg_source_group_map_t::const_iterator iterGroups;
    rpg_source_item_map_t::const_iterator iterItems;

    iterGroups = sources.find (group);
    if (iterGroups != sources.end ())
    {
        rpg_source_item_map_t *items = iterGroups->second;

        iterItems = items->find (item);
        if (iterItems != items->end ())
        {
            return (iterItems->second);
        }
    }

    return (NULL);
}


/** Creates source from group identifier and item identifier, with optional default substitute.
 *
 * With group and item, the order of evaluation of the default substitutes is:
 * 1. group::item
 * 2. default-group::item
 * 3. default::item
 *
 * @param group Group of the data source.
 * @param item Item of the data source in the group.
 * @param checkConstant Warn if this source is not constant.
 *
 * @return Data source created by the specified group::item holder.
 */
dataSource *dataSourceProvider::createSource (
		const rpg_str_t &group,
		const rpg_str_t &item,
		const bool checkConstant)
{
    // Just try to locate the direct holder or the holder of the default substitutes.
    dataSourceHolder *dsh = locateSourceHolder (group, item);
    if (dsh == NULL) dsh = locateSourceHolder (GROUP_DEFAULT "-" + group, item);
    if (dsh == NULL) dsh = locateSourceHolder (GROUP_DEFAULT, item);

    if (dsh == NULL)
    {
        cerr << "Error: Unable to locate source for " << group << "::" << item << "." << endl;
        throw INVALID_SOURCE_ACCESS;
    }

    // Create the source and remember where it came from.
    dataSource *ds = dsh->createSource ();
    ds->setIdentifiers (group, item);

    // Print a warning if the source should be constant but is not.
	if (checkConstant && !ds->isVoid () && !ds->isConstant ())
	{
		cerr << "Warning: Parameter " << group << "::" << item << " should be constant but is not." << endl;
	}

	return (ds);
}


/** Registers new data source holder in the source provider. 
  * @param group Group of the data source.
  * @param item Item of the data source in the group.
  * @param holder Instance of the holder with specified data.
  */
void dataSourceProvider::registerSource(const rpg_str_t& group, 
										const rpg_str_t& item,
										dataSourceHolder* holder) {
    assert (holder != NULL);

    /* if source group exists, reuse it */
    rpg_source_group_map_t::iterator iterGroups =  sources.find(group);
    rpg_source_item_map_t * mItems;
    if (iterGroups != sources.end()) {
    	mItems = iterGroups->second;
    } else {
    	mItems = new rpg_source_item_map_t();
    	sources[group] = mItems;
    }

	/* if source item exists, delete it */
	rpg_source_item_map_t::iterator iterItems = mItems->find(item);

	if (iterItems != mItems->end()) {
		delete iterItems->second;
	}

	/* set new holder */
	(*mItems)[item] = holder;
}

/** Registers simple source. If source exists, it is overwritten.
  * @param group Group of the data source.
  * @param item Item of the data source in the group.
  * @param source Value of the data source.
  */
template<typename param_t>
void dataSourceProvider::registerSource(const rpg_str_t& group, const rpg_str_t& item, const param_t & source) {
	dataSourceHolder* holder = new simpleDataSourceHolder<param_t> (classProvider);
	holder->setParam(CONFIG_COMMON_VALUE, source);
	registerSource(group, item, holder);
}

/*
 * Explicit template instantiation. Either we need this, or move simpleDataSourceHolder to config.h
 * Arguably this is less troublesome.
 */
template void dataSourceProvider::registerSource<rpg_int_t>
	(const rpg_str_t& group, const rpg_str_t& item, const rpg_int_t & source);
template void dataSourceProvider::registerSource<rpg_float_t>
	(const rpg_str_t& group, const rpg_str_t& item, const rpg_float_t & source);
template void dataSourceProvider::registerSource<rpg_str_t>
	(const rpg_str_t& group, const rpg_str_t& item, const rpg_str_t & source);
template void dataSourceProvider::registerSource<rpg_list_int_t>
	(const rpg_str_t& group, const rpg_str_t& item, const rpg_list_int_t & source);
template void dataSourceProvider::registerSource<rpg_list_float_t>
	(const rpg_str_t& group, const rpg_str_t& item, const rpg_list_float_t & source);
template void dataSourceProvider::registerSource<rpg_list_str_t>
	(const rpg_str_t& group, const rpg_str_t& item, const rpg_list_str_t & source);

/** Print list of resources on the standard output. */
void dataSourceProvider::dumpSources(void) {
	for (rpg_source_group_map_t::iterator iterGroups = sources.begin();
		 iterGroups != sources.end(); iterGroups++) {

		const rpg_str_t sGroup = iterGroups->first;
		rpg_source_item_map_t & group = *(iterGroups->second);
		for (rpg_source_item_map_t::iterator iterItems = group.begin();
			 iterItems != group.end(); iterItems++) {

			const rpg_str_t sItem = iterItems->first;
			cout << sGroup << "::" << sItem << " [";

			/* print content of the source */
			dataSource* source = iterItems->second->createSource();
			if (source->isConstant())		{ cout << "C"; }
			if (source->isInteger())		{ cout << "I"; }
			if (source->isString())			{ cout << "S"; }
			if (source->isStringArray())	{ cout << "SA"; }
			if (source->isIntegerArray())	{ cout << "IA"; }
			if (source->isVoid())			{ cout << "V"; }
			cout << "] = " << source->toString() << endl;

			delete source;
		}
	}
}
rpg_source_item_map_t *
dataSourceProvider::getItemMap (const rpg_str_t & group) {
	rpg_source_group_map_t::iterator iter = sources.find(group);
	if (iter != sources.end()) {
		return iter->second;
	} else {
		return NULL;
	}
}

/** Constructor */
dataSourceProvider::dataSourceProvider (dataClassProvider * _classProvider)
	: classProvider (_classProvider) {
	/* register random source */
	rpg_int_t seed = 0;
	registerSource(GROUP_CORE, ITEM_RANDOM_SEED, seed);
}

/*     
  ---            dataSourceHolder implementation                       ---
																		 */

/** Constructor */
dataSourceHolder::dataSourceHolder (dataClassProvider * _provider)
	: provider (_provider) {
}

/** Sets the named integer parameter to this resource. Some resources
  * do not support this function, error message is printed instead.
  * @param name Name of the resource parameter.
  * @param value Value of the resource parameter.
  */
void dataSourceHolder::setParam(const rpg_str_t& name, const rpg_int_t& value) {
	rpg_str_t err_message = "Trying to set INT parameter '" + name + "' in source '" + getName() + "'.";
	error(err_message, line_number);
}

/** Sets the named float parameter to this resource. Some resources
  * do not support this function, error message is printed instead.
  * @param name Name of the resource parameter.
  * @param value Value of the resource parameter.
  */
void dataSourceHolder::setParam(const rpg_str_t& name, const rpg_float_t& value) {
	rpg_str_t err_message = "Trying to set FLOAT parameter '" + name + "' in source '" + getName() + "'.";
	error(err_message, line_number);
}

/** Sets the named string parameter to this resource. Some resources
  * do not support this function, error message is printed instead.
  * @param name Name of the resource parameter.
  * @param value Value of the resource parameter.
  */
void dataSourceHolder::setParam(const rpg_str_t& name, const rpg_str_t& value) {
	rpg_str_t err_message = "Trying to set STRING parameter '" + name + "' in source '" + getName() + "'.";
	error(err_message, line_number);
}

/** Sets the named int array parameter to this resource. Some resources
  * do not support this function, error message is printed instead.
  * @param name Name of the resource parameter.
  * @param value Value of the resource parameter.
  */
void dataSourceHolder::setParam(const rpg_str_t& name, const rpg_list_int_t& value) {
	rpg_str_t err_message = "Trying to set INT ARRAY parameter '" + name + "' in source '" + getName() + "'.";
	error(err_message, line_number);
}

/** Sets the named int array parameter to this resource. Some resources
  * do not support this function, error message is printed instead.
  * @param name Name of the resource parameter.
  * @param value Value of the resource parameter.
  */
void dataSourceHolder::setParam(const rpg_str_t& name, const rpg_list_float_t& value) {
	rpg_str_t err_message = "Trying to set FLOAT ARRAY parameter '" + name + "' in source '" + getName() + "'.";
	error(err_message, line_number);
}

/** Sets the named string array parameter to this resource. Some resources
  * do not support this function, error message is printed instead.
  * @param name Name of the resource parameter.
  * @param value Value of the resource parameter.
  */
void dataSourceHolder::setParam(const rpg_str_t& name, const rpg_list_str_t& value) {
	rpg_str_t err_message = "Trying to set STRING ARRAY parameter '" + name + "' in source '" + getName() + "'.";
	error(err_message, line_number);
}

/** Retrieves class name of this holder.
  * @return Name of the resource class.
  */
const char* dataSourceHolder::getName(void) const {
	return CLASS_BASE_NAME;
}

/** Creates resource from the resource holder. 
  * Data source is then independent on the resource holder.
  * @return New instance of the data source class
  */
dataSource* dataSourceHolder::createSource(void) {
	return new dataSource(this);
}

void dataSourceHolder::applyParam (const rpg_param_value_t & param) {
	switch (param.type) {
		case INT:            { setParam(param.param_name, param.v_int);          break; }
		case FLOAT:			 { setParam(param.param_name, param.v_float); 		 break; }
		case STRING:         { setParam(param.param_name, param.v_string);       break; }
		case INTLIST:        { setParam(param.param_name, param.v_int_list);     break; }
		case FLOATLIST:      { setParam(param.param_name, param.v_float_list);   break; }
		case STRINGLIST:     { setParam(param.param_name, param.v_string_list);  break; }
		default:
			assert (false);
	}
}

/** Applies parameters params to this holder.
  * @param params Parameters to be applied. */
void dataSourceHolder::applyParams(const rpg_params_t& params) {
	rpg_params_t :: const_iterator iter;
	for (iter = params.begin(); iter != params.end(); iter++) {
		applyParam (*iter);
	}
}

/** Checks internal data integrity of the resource. */
bool dataSourceHolder::checkData(void) {
	return true;
}

dataSourceHolder::~dataSourceHolder() {
}

/*     
  ---                  dataSource implementation                        ---
																		 */

/** Initializes data source based on the data source holder.
  * @param holder_ Holder, which is creating this data source.
  */
dataSource::dataSource(const dataSourceHolder* holder_) {
	/* only register a name */
	name = holder_->getName();
}

/** Set the group::item configuration identifier of the data source.
 * This is used for more descriptive error messages.
 */
void
dataSource::setIdentifiers (const rpg_str_t & _group, const rpg_str_t & _item) {
	group = _group;
	item = _item;
}

rpg_exception_t
dataSource::common_error (const char * cause, rpg_exception_t exc) {
	if (group.size () == 0 || item.size () == 0) {
		error (rpg_str_t (cause) + " from unknown config of type " + getName());
	} else {
		error (rpg_str_t (cause) + " from config " + group + "::" + item + " of type " + getName());
	}
	throw exc;
}

/** Gets an integer value from the data source. Some data sources
  * cannot provide integers, so an error is printed and exception raised.
  * @return Reference to integer value.
  */
rpg_int_t
dataSource::getIntValue() {
	throw common_error ("Trying to access invalid IntValue resource", INVALID_SOURCE_ACCESS);
}

rpg_int_t
dataSource::getIntMin() {
	throw common_error ("Trying to access invalid IntMin resource", INVALID_SOURCE_ACCESS);
}

rpg_int_t
dataSource::getIntMax() {
	throw common_error ("Trying to access invalid IntMax resource", INVALID_SOURCE_ACCESS);
}

rpg_int_t
dataSource::getIntMean() {
	throw common_error ("Trying to access invalid IntMean resource", INVALID_SOURCE_ACCESS);
}

/** Gets a float value from the data source. Some data sources
  * cannot provide floats, so an error is printed and exception raised.
  * @return Reference to float value.
  */
rpg_float_t
dataSource::getFloatValue() {
	if (isInteger ()) {
		return (rpg_float_t) getIntValue ();
	}
	throw common_error ("Trying to access invalid FloatValue resource", INVALID_SOURCE_ACCESS);
}

rpg_float_t
dataSource::getFloatMin() {
	if (isInteger ()) {
		return (rpg_float_t) getIntMin ();
	}
	throw common_error ("Trying to access invalid FloatMin resource", INVALID_SOURCE_ACCESS);
}

rpg_float_t
dataSource::getFloatMax() {
	if (isInteger ()) {
		return (rpg_float_t) getIntMax ();
	}
	throw common_error ("Trying to access invalid FloatMax resource", INVALID_SOURCE_ACCESS);
}

rpg_float_t
dataSource::getFloatMean() {
	if (isInteger ()) {
		return (rpg_float_t) getIntMean ();
	}
	throw common_error ("Trying to access invalid FloatMean resource", INVALID_SOURCE_ACCESS);
}

/** Gets an bool value from the data source. Some data sources
  * cannot provide bools, so an error is printed and exception raised.
  * @return Reference to integer value.
  */
bool
dataSource::getBoolValue() {
	if (isInteger ()) {
		return (bool) getBoolValue ();
	}
	throw common_error ("Trying to access invalid BoolValue resource", INVALID_SOURCE_ACCESS);
}

/** Gets a string value from the data source. Some data sources
  * cannot provide strings, so an error is printed and exception raised.
  * @return Reference to string value.
  */
const rpg_str_t&
dataSource::getStringValue () {
	throw common_error ("Trying to access invalid STRING resource", INVALID_SOURCE_ACCESS);
}

/** Gets min list from the interval data type.
  * @return Reference to the list.
  */
const rpg_list_int_t& dataSource::access_interval_min_list(void) {
	throw common_error ("Trying to access invalid INTERVAL resource", INVALID_SOURCE_ACCESS);
}

/** Gets min list from the interval data type.
  * @return Reference to the list.
  */
const rpg_list_int_t& dataSource::access_interval_max_list(void) {
	throw common_error ("Trying to access invalid INTERVAL resource", INVALID_SOURCE_ACCESS);
}


/** For array resources, this function returns array size.
  * @return Array size.
  */
rpg_uint_t dataSource::array_size(void) {
	throw common_error ("Trying to access invalid array size resource", INVALID_SOURCE_ACCESS);
}

/** Access to the string array of this resource.
  * @param index Index to the string array, must be in interval [0,array_size-1]
  * @return Reference to the string on the position 
  */
const rpg_str_t& dataSource::operator[] (const rpg_uint_t index) {
	throw common_error ("Trying to access invalid STRING ARRAY resource", INVALID_SOURCE_ACCESS);
}

/** Checks if type is constant. Constants are for example
  * integers, strings. Non-constants are for example random
  * generators, timers etc.
  * @return True, if data source is constant, false otherwise.
  */
bool dataSource::isConstant(void) {
	return false;
}

/** Checks if type is integral. Integrals are simple integer types.
  * @return True, if data source is integral, false otherwise.
  */
bool dataSource::isInteger(void) {
	return false;
}

/** Checks if type is float. */
bool dataSource::isFloat(void) {
	return false;
}

/** Checks if type is interval. Intervals are pairs of integers.
  * @return True, if data source is interval, false otherwise.
  */
bool dataSource::isInterval(void) {
	return false;
}

/** Checks if type is string. Strings are simple string types.
  * @return True, if data source is string, false otherwise.
  */
bool dataSource::isString(void) {
	return false;
}

/** Checks if type is string list (array). String arrays can be accessed trough
  * access_str_list function, or [] operator.
  * @return True, if data source is string list, false otherwise.
  */
bool dataSource::isStringArray(void) {
	return false;
}

/** Checks if type is integer list (array). Integer arrays can be accessed trough
  * access_int_list function.
  * @return True, if data source is integral list, false otherwise.
  */
bool dataSource::isIntegerArray(void) {
	return false;
}

/** Checks if type is float list (array). Float arrays can be accessed through
 *  access_float_list function.
 */
bool dataSource::isFloatArray () {
	return false;
}

/** Checks if data source is void. Void data sources are empty and any call
  * to them, which want to access data, cause error and exception raised.
  * @return True, if data source is void, false otherwise.
  */
bool dataSource::isVoid(void) {
	return rpg_str_t(getName()) == rpg_str_t(CLASS_BASE_NAME);
}

/** Returns data source name.
  * @return Source name.
  */
const char* dataSource::getName(void) {
	return name;
}

/** Access to the integer array, if the data source is integer array.
  * If the data source is not integer array, error is reported and
  * exception raised.
  * @return Reference to the integer array.
  */
const rpg_list_int_t & dataSource::access_int_list(void) {
	throw common_error ("Trying to access invalid INT ARRAY resource", INVALID_SOURCE_ACCESS);
}

/** Access to the float array, if the data source is float array.
  * If the data source is not float array, error is reported and
  * exception raised.
  * @return Reference to the integer array.
  */
const rpg_list_float_t & dataSource::access_float_list(void) {
	throw common_error ("Trying to access invalid FLOAT ARRAY resource", INVALID_SOURCE_ACCESS);
}

/** Access to the string array, if the data source is string array.
  * If the data source is not string array, error is reported and
  * exception raised.
  * @return Reference to the string array.
  */
const rpg_list_str_t & dataSource::access_str_list(void) {
	throw common_error ("Trying to access invalid STRING ARRAY resource", INVALID_SOURCE_ACCESS);
}

/** Converts data source's content to the string representation 
  * @return String representation of the data source value. */
rpg_str_t dataSource::toString(void) {
	return "(void)";
}

/* Convert data source's content into a configuration and XML definition and print it in the given streams
 * Used by generator for module parameters.
 */
void dataSource::generateConfigAndXML (std::ostream& con, std::ostream& xml) {
	/* it is not a parameter class, which means we'll take just one value, depending on its type
	 * if it's a random distribution, it means the randomization of the module parameter is done
	 * when generating the architecture, not run-time
	 * if it's a constant, it should work the same
	 */

	if (isInteger ()) {
		rpg_int_t value = getIntValue ();
		printSimpleParam (con, xml, CLASS_INTEGER_NAME, value);
	} else if (isFloat ()) {
		rpg_float_t value = getFloatValue ();
		printSimpleParam (con, xml, CLASS_FLOAT_NAME, value);
	} else if (isString ()) {
		const rpg_str_t & value = getStringValue ();
		printSimpleParam (con, xml, CLASS_STRING_NAME, value);
	} else {
		throw common_error ("Trying to generate configuration/XML of invalid resource", INVALID_SOURCE_ACCESS);
	}
}

/** Destructors */
dataSource::~dataSource(){}

/*
  ---                  rpg_context_t implementation                     ---
																		 */

/** Constructor of parser context. Sets class and source providers.
  * @param sp Source provider.
  * @param cp Class provider.
  */
rpg_context_t::rpg_context_t(dataSourceProvider&  sp, dataClassProvider& cp) 
    : check_group_name (false), sprovider (sp), cprovider (cp) {
}

/** Checks, if group of the configuration is correct.
  * Group name is correct, if it is equal to the set name
  * trough setGroupName() function, or it is not specified - function
  * clearGroupName() was used.
  * @param name Name to be checked.
  * @return True, if group name is correct, false otherwise 
  */
rpg_bool_t rpg_context_t::checkGroupName(const rpg_str_t& name) {
	return ( (! check_group_name) || (name == group_name) );
}

/** Clears group name, so it is not used in checking. */
void rpg_context_t::clearGroupName() {
	check_group_name = false;	
}

/** Sets group name, so it is used in checking of group names.
  * @param name Reference name, which is used in comparison to
  * name specified in configuration.
  */
void rpg_context_t::setGroupName(const rpg_str_t& name) {
	check_group_name = true;
	group_name = name;
}

/** Interval constructor.
  * @param _min Lower bound of interval.
  * @param _max Upper bound of interval.
  */
rpg_interval_t::rpg_interval_t(rpg_uint_t _min, rpg_uint_t _max) :
	min (_min), max (_max) {
}
