
# Plotting functions for RPG

source ("common.R")

require ("vioplot")

############################################################################
# Plotting helper functions
############################################################################

# Plot standard deviation (or confidence internval) as arrows 
# @param asymmetric: when true, vMeans is considered lower bound, vSds upper bound
plotSdArrows <- function (vMeans, vSds, vX, asymmetric=FALSE) {
  if (asymmetric == TRUE) {
    vLower <- vMeans
    vUpper <- vSds
  } else {
    vLower <- vMeans - vSds
    vUpper <- vMeans + vSds
  }
  arrows (vX, vLower, vX, vUpper, code = 3, col = "pink", angle = 75, length = .1, lwd=2)
}

# Combine boxplot with mean+sd plot
boxplotWithMeanSd <- function (lGroups, ...) {
  boxplot (lGroups, ...)
  
  vMeans <- vector (mode = "numeric", length = 0)
  vSds <- vector (mode = "numeric", length = 0)
  for (group in lGroups) {
    if (is.vector (group)) {
      vMeans = c (vMeans, mean (group))
      vSds = c (vSds, sd (group))
    } else {
      vMeans = c (vMeans, group)
      vSds = c (vSds, 0)
    }
  }

  vX <- 0.2 + seq (length (lGroups))
  points (vX, vMeans, col = "orange", pch = 8, lwd=2, cex=2)
  plotSdArrows (vMeans, vSds, vX)
}

multihist <- function (lDataSets, vCols, ...) {

  iNumSets <- length (lDataSets)

  lHists <- lapply (lDataSets, hist, plot = FALSE)
  vXlim <- range (unlist (lDataSets))
  fMaxDensity <- max (sapply (lapply (lHists, "[[", "density"), max))

  plot (lHists[[1]], xlim = vXlim, ylim = c(0, fMaxDensity), col = vCols[1], freq = FALSE, ...)
  for (iSet in 2:iNumSets) {
    plot (lHists[[iSet]], add = TRUE, col = vCols[iSet], freq=FALSE)
  }
}

# simpler interface to plot multiple vioplots, accepting list of named groups of values
# instead of (x, ...)
viogroup <- function (lData, xlab, ylab) {
  # discard data groups with less than 10 samples
  lData <- filterSmallGroups (lData, 10)

  # save the names to be used later
  vNames <- names (lData)
  # rename the first item because vioplot expects 'x')
  names(lData)[1] <- "x"

  # since vioplot does not accept usual parameters such as xlab/ylab, prepare the plot manually
  iGroups <- length (vNames)
  vRange <- range (unlist(lData))
  plot (NULL, xlim = c (1, iGroups), ylim = vRange, xaxt="n", xlab=xlab, ylab=ylab)
  axis (1, at = 1:iGroups, labels = vNames)

  # call vioplot
  do.call ("vioplot", c (lData, list (col="gray", names=vNames, add=TRUE)))
}

############################################################################
# Plots of summaries across multiple architectures
############################################################################

ePcmApp <- 1
eQpnApp <- 2
ePcmQpn <- 3

getYlab <- function (eY, bResp=FALSE) {

  if (bResp) {
    sWhat <- "resp. time"
  } else {
    sWhat <- "tput"
  }

  if (eY == ePcmApp) {
    sYlab <- "Diff. in SAM-predicted/measured mean"
  } else if (eY == eQpnApp) {
    sYlab <- "Diff. in QPN-predicted/measured mean"
  } else if (eY == ePcmQpn) {
    sYlab <- "Diff. in SAM/QPN predicted mean"
  }

  sYlab <- paste (sYlab, sWhat, "[%]")

  return (sYlab)
}

getYvec <- function (tStats, eY, bResp=FALSE) {
  if (bResp) {
    sWhat <- "Resp"
  } else {
    sWhat <- "Tput"
  }

  sPcm <- paste ("pcm", sWhat, "Mean", sep="")
  sQpn <- paste ("sim", sWhat, "Mean", sep="")
  sApp <- paste ("app", sWhat, "Mean", sep="")

  if (eY == ePcmApp) {
    vY <- tStats[[sPcm]] / tStats[[sApp]]
  } else if (eY == eQpnApp) {
    vY <- tStats[[sQpn]] / tStats[[sApp]]
  } else if (eY == ePcmQpn) {
    vY <- tStats[[sPcm]] / tStats[[sQpn]]
  }
  vY <- vY * 100 - 100

  return (vY)
}

# filters out groups of data (created with split()) that have less than iThreshold members
filterSmallGroups <- function (lData, iThreshold) {
  lData <- lData [sapply (lData, function(x) { return (length (x) >= iThreshold) })]
  return (lData)
}

# print table with correlations between predicted/measured data depending on number of threads
printCorTable <- function (tStats, fOut) {
  
  printHeader <- function (vNums) {
    iCols <- length (vNums)
    cat (sprintf ("\\begin{tabular}{l|*{%d}{c|}l}\n", iCols), file=fOut)

    cat (sprintf ("\\cline{2-%d}\n", iCols + 1), file=fOut)
    cat (sprintf ("& \\multicolumn{%d}{|c|}{\\textbf{%s}}\\\\\n", iCols, "Number of threads"), file=fOut)
    cat (sprintf ("\\hline\n", iCols), file=fOut)
    
    cat (sprintf ("\\multicolumn{1}{|l|}{\\textbf{Correlation between}}"), file=fOut)
    for (sNum in vNums) {
      cat (sprintf ("& %s ", sNum), file=fOut)
    }
    cat (sprintf ("\\\\\n\\hline\n"), file=fOut)
  }

  printValues <- function (vValues, sRowName) {
    cat (sprintf ("\\multicolumn{1}{|l|}{%s}", sRowName), file=fOut)
    for (fValue in vValues) {
      cat (sprintf ("& %0.2f ", fValue, digits=2), file=fOut)
    }
    cat (sprintf ("\\\\\n"), file=fOut)
  }

  lSplit <- split(tStats, tStats$archThreadCount)

  printHeader (names (lSplit))

  vValues <- sapply (lSplit, function(x) { cor(x$pcmTputMean, x$appTputMean) })

  printValues (vValues, "SAM prediction / measured ")

  vValues <- sapply (lSplit, function(x) { cor(x$pcmTputMean, x$simTputMean) })

  printValues (vValues, "SAM / QPN prediction ")

  vValues <- sapply (lSplit, function(x) { cor(x$simTputMean, x$appTputMean) })

  printValues (vValues, "QPN prediction / measured ")
}

# determine for how many pairs of architectures the prediction decides correctly which one has faster throughput
# @param fThreshold is used to ignore pairs where the ratio (higher of the pair to lower of the pair) of both prediction and measurement is lower than the threshold
calculatePredPairMatch <- function (tGroup, fThreshold, sValue1="pcmTputMean", sValue2="appTputMean") {

  isAboveThreshold <- function (fValue1, fValue2) {
    return (min (fValue1, fValue2) * fThreshold < max (fValue1, fValue2))
  }

  determineMatch <- function (tArch1, tArch2) {
    
    fArch1Value1 <- tArch1 [[sValue1]]
    fArch1Value2 <- tArch1 [[sValue2]]
    fArch2Value1 <- tArch2 [[sValue1]]
    fArch2Value2 <- tArch2 [[sValue2]]
    
    if (!isAboveThreshold (fArch1Value1, fArch2Value1) & !isAboveThreshold (fArch1Value2, fArch2Value2)) {
      # not above threshold in both values
      return (0)
    } else {
      if ((fArch1Value1 < fArch2Value1) == (fArch1Value2 < fArch2Value2)) {
	# prediction and measurement have the same conclusion
	return (1)
      } else {
	# prediction and measurement differ in conclusion
	return (2)
      }
    }
  }

  iRows <- nrow (tGroup)
  vCombs <- combs (1:iRows, 2)

  vRes <- vector (mode = "integer", length = nrow (vCombs))
  
  # TODO: figure out how to vectorize this ?
  for (iComb in 1:nrow (vCombs)) {
    tArch1 <- tGroup [vCombs [iComb,1],]
    tArch2 <- tGroup [vCombs [iComb,2],]

    vRes [iComb] <- determineMatch (tArch1, tArch2)
  }

  return (vRes)
}

calculatePredPairMatches <- function (tStats, fThreshold, sValue1="pcmTputMean", sValue2="appTputMean") {
  lSplit <- split(tStats, tStats$archThreadCount)
  lMatches <- mclapply (lSplit, calculatePredPairMatch, fThreshold=fThreshold, sValue1=sValue1, sValue2=sValue2)

  return (lMatches)
}

processPredPairMatches <- function (lMatches) {
  process <- function (x) {
    iHits <- length (x [x == 1])
    iMisses <- length (x [x == 2])
    iTotal <- iHits + iMisses
    return (c (hits = iHits / iTotal, misses = iMisses / iTotal))
  }

  sapply (lMatches, process)
}

plotPredPairMatches <- function (lMatches, sFile) {
  postscript (sFile, width=8.27, height=5.43, horizontal=FALSE)
  barplot (processPredPairMatches (lMatches), xlab="Threads", ylab="Ratio of hits and misses predicting alternatives with higher throughput")
  dev.off ()
}

calcAll <- function (tStats) {
  lMatches1 <- calculatePredPairMatches (tStats, 1.001)
  save (lMatches1, file="lMatches1.Rdata")
  lMatches2 <- calculatePredPairMatches (tStats, 1.01)
  save (lMatches2, file="lMatches2.Rdata")
  lMatches3 <- calculatePredPairMatches (tStats, 1.1)
  save (lMatches3, file="lMatches3.Rdata")
  lMatches4 <- calculatePredPairMatches (tStats, 1.5)
  save (lMatches4, file="lMatches4.Rdata")
}

# scatterplot - service time slowdown depending on number of modules in the architecture
plotScatterModulesCountVsSvcTimeSlowdown <- function (lStats, bPCM=FALSE) {
  tStats <- lStats$main
  
  if (bPCM) {
    vSim <- tStats$pcmRespMean
    sMain <- "Mean response time prediction accuracy with PCM depending on number of modules"
  } else {
    vSim <- tStats$simRespMean
    sMain <- "Mean response time prediction accuracy with SimQPN depending on number of modules"
  }

  plot (tStats$allModCount, tStats$appRespMean / vSim, xlab="Number of modules in the architecture",
    ylab="Ratio of measured/predicted service times", main=sMain, panel.first = grid(NA, NULL))
  axis(4)
}

# Simple scatterplot of SimQPN throughput prediction precision
plotScatterSimTputPrecision <- function (lStats) {
  tStats <- lStats$main
  plot (tStats$simTputMean / tStats$appTputMean * 100 - 100,
    ylab="Difference in pred./measured mean throughput [%]", xlab="Generated system")
}

# Simple scatterplot of PCM throughput prediction precision
plotScatterPCMTputPrecision <- function (lStats, sMain="Precision of mean throughput prediction") {
  tStats <- lStats$main
  plot (tStats$pcmTputMean / tStats$appTputMean * 100 - 100,
    main=sMain, ylab="Diff. in predicted/measured mean throughput [%]", xlab="Generated system")
}

# Simple scatterplot of PCM throughput prediction precision
plotScatterPCMRespPrecision <- function (lStats, sMain="Precision of mean response time prediction") {
  tStats <- lStats$main
  plot (tStats$pcmRespMean / tStats$appRespMean * 100 - 100,
    main=sMain, ylab="Diff. in predicted/measured mean resp. time [%]", xlab="Generated system")
}

# violin plots of PCM resp precision depending on given factor
plotVioPCMRespPrecision <- function (tStats, vFactor, xlab) {
  lSplit <- split (tStats$pcmRespMean / tStats$appRespMean * 100 - 100, vFactor)
  viogroup (lSplit, xlab=xlab, ylab="Diff. in predicted/measured mean resp. time [%]")
}

# Scatterplot of PCM throughput prediction precision depending on given X vector
plotScatterPCMTputPrecisionX <- function (vX, sXlab, lStats, sMain="Precision of mean throughput prediction", ...) {
  tStats <- lStats$main
  plot (vX, tStats$pcmTputMean / tStats$appTputMean * 100 - 100,
    main=sMain, ylab="Diff. in predicted/measured mean throughput [%]", xlab=sXlab, ...)
}

plotVioPredictionPrecision <- function (vFactor, eY, sXlab, lStats, bResp=FALSE, sMain="Precision of mean throughput prediction", ...) {
  tStats <- lStats$main
  vY <- getYvec (tStats, eY, bResp)
  sYlab <- getYlab (eY, bResp)

  lSplit <- split (vY, vFactor)
  viogroup (lSplit, xlab=sXlab, ylab=sYlab, ...)
}

plotScatterPredictionPrecision <- function (vX, eY, sXlab, lStats, bResp=FALSE, sMain="Precision of mean throughput prediction", ...) {
  tStats <- lStats$main
  vY <- getYvec (tStats, eY, bResp)
  sYlab <- getYlab (eY, bResp)

  plot (vX, vY, main=sMain, xlab=sXlab, ylab=sYlab, ...)
}

# Scatterplot of prediction precision on given parameter, with all combinations of pcm/qpn/measured 
plotScatterAllPrecision <- function (sFilePrefix, lStats, vX, sXlab, bResp=FALSE, ...) {
  postscript (paste (sFilePrefix, "-pcm_app.ps", sep=""), width=8.27, height=5.43, horizontal=FALSE)
 
  plotScatterPredictionPrecision (vX, ePcmApp, sXlab, lStats, bResp=bResp, sMain="", ...)

  dev.off ()

  postscript (paste (sFilePrefix, "-qpn_app.ps", sep=""), width=8.27, height=5.43, horizontal=FALSE)
 
  plotScatterPredictionPrecision (vX, eQpnApp, sXlab, lStats, bResp=bResp, sMain="", ...)

  dev.off ();

  postscript (paste (sFilePrefix, "-pcm_qpn.ps", sep=""), width=8.27, height=5.43, horizontal=FALSE)
 
  plotScatterPredictionPrecision (vX, ePcmQpn, sXlab, lStats, bResp=bResp, sMain="", ...)

  dev.off ();
}

# Vioplots of prediction precision on given parameter, with all combinations of pcm/qpn/measured 
plotVioAllPrecision <- function (sFilePrefix, lStats, vFactor, sXlab, bResp=FALSE, ...) {
  postscript (paste (sFilePrefix, "-pcm_app.ps", sep=""), width=8.27, height=5.43, horizontal=FALSE)
 
  plotVioPredictionPrecision (vFactor, ePcmApp, sXlab, lStats, bResp=bResp, sMain="", ...)

  dev.off ()

  postscript (paste (sFilePrefix, "-qpn_app.ps", sep=""), width=8.27, height=5.43, horizontal=FALSE)
 
  plotVioPredictionPrecision (vFactor, eQpnApp, sXlab, lStats, bResp=bResp, sMain="", ...)

  dev.off ();

  postscript (paste (sFilePrefix, "-pcm_qpn.ps", sep=""), width=8.27, height=5.43, horizontal=FALSE)
 
  plotVioPredictionPrecision (vFactor, ePcmQpn, sXlab, lStats, bResp=bResp, sMain="", ...)

  dev.off ();
}

# Simple scatterplot of PCM vs QPN throughput prediction precision
plotScatterPCMVsQPNTputPrecision <- function (lStats, sMain="Precision of mean throughput prediction compared to SimQPN") {
  tStats <- lStats$main
  plot (tStats$pcmTputMean / tStats$simTputMean * 100 - 100,
    main=sMain, ylab="Diff. in predicted mean throughput compared to SimQPN [%]", xlab="Generated system")
}

# Scatterplot of SimQPN throughput prediction precision, depending on predited CPU Util
plotScatterCPUUtilVsSimTputPrecision <- function (lStats) {
  tStats <- lStats$main
  plot (tStats$simCpuUtil, tStats$simTputMean / tStats$appTputMean * 100 - 100,
    ylab="Difference in pred./measured mean throughput [%]", xlab="Predicted CPU utilization [cores]")
}

# Scatterplot of SimQPN throughput prediction precision, depending on predited CPU Util, including simulation with exp. distr. (red color)
plotScatterCPUUtilVsSimTputPrecisionWithExp <- function (lStats) {
  tStats <- lStats$main

  vNorPrecision <- tStats$simTputMean / tStats$appTputMean * 100 - 100
  vExpPrecision <- tStats$simExpTputMean / tStats$appTputMean * 100 - 100
  
  vRange <- range (c (vNorPrecision, vExpPrecision))

  plot (tStats$simCpuUtil, vNorPrecision, ylim=vRange,
    ylab="Difference in pred./measured mean throughput [%]", xlab="Predicted CPU utilization [cores]")
  points (tStats$simCpuUtil, vExpPrecision, col="red")
}

# Scatterplot of SimQPN throughput prediction precision, depending on predited CPU Util, including simulation with shared timings (red color)
plotScatterCPUUtilVsSimTputPrecisionWithShr <- function (lStats) {
  tStats <- lStats$main

  vNorPrecision <- tStats$simTputMean / tStats$appTputMean * 100 - 100
  vShrPrecision <- tStats$simShrTputMean / tStats$appTputMean * 100 - 100
  
  vRange <- range (c (vNorPrecision, vShrPrecision))

  plot (vNorPrecision, ylim=vRange,
   ylab="Difference in pred./measured mean throughput [%]", xlab="Predicted CPU utilization [cores]")
  points (vShrPrecision, col="red")
}

plotScatterCPUUtilVsSimTputPrecisionExpNor <- function (lStats) {
  tStats <- lStats$main
  plot(tStats$simCpuUtil, tStats$simExpTputMean / tStats$simTputMean * 100 - 100,
    xlab="Predicted CPU utilization [cores]",
    ylab="Difference in predicted mean throughput\nusing exp/norm CPU service times [%]")
}

# boxplot of service time slowdown depending on the presence of module classes in the architecture
plotBoxplotModuleClassPresenceVsSvcTimeSlowdown <- function (lStats) {
  tData <- data.frame()
  for (iDir in seq (along=lStats$modcount)) {
    tDir <- expand.grid (slowdown=lStats$nor[iDir], module=lStats$modclasses[[iDir]])
    tData <- rbind (tData, tDir)
  }

  plot(slowdown ~ module, data=tData, las=3)
}

plotScatterCPUUtilVsSvcTimeSlowdown <- function (lStats) {
  plot (lStats$simcpuutil, lStats$nor, xlab="Predicted mean CPU utilization [cores]",
    ylab="Ratio of measured/predicted service times", panel.first = grid(NA, NULL))
}

plotScatterCPUUtilVsSvcTimeSlowdownExp <- function (lStats) {
  plot (lStats$simcpuutil, lStats$exp, xlab="Predicted mean CPU utilization [cores]",
    ylab="Ratio of measured/predicted service times", panel.first = grid(NA, NULL))
}

plotSeparateModulesScatterCPUUtilVsSvcTimeSlowdown <- function (lStats, vModules=NULL, bPostscript=FALSE) {
  if (is.null (vModules)) {
    vModules = lStats$allmodclasses
  }
  for (sMod in vModules) {
    mResults <- vector ()
    for (iDir in seq (along=lStats$modcount)) {
      if (sMod %in% lStats$modclasses[iDir][[1]]) {
	fModCPUUtil <- as.numeric (lStats$modcpuutils[iDir][[1]][sMod])
	fAllCPUUtil <- lStats$simcpuutil[iDir]
	if (! is.na (fModCPUUtil)) {
	  vResults <- c (absmodutil=fModCPUUtil, relmodutil=fModCPUUtil / fAllCPUUtil, nor=lStats$nor[iDir])
	  mResults <- rbind(mResults, vResults, deparse.level=0)
	}
      }
    }
    dResults <- data.frame (mResults)
    if (length (mResults) > 0) {
      if (bPostscript == FALSE) {
	png (paste ("scatter-util-abs-svctimes-",sMod,".png",sep=""), width=1024, height=768)
	plot (dResults$absmodutil, dResults$nor, xlab=sMod)
	dev.off()
      }
      if (bPostscript) {
	postscript (paste ("scatter-module-impact-", sMod, ".ps", sep=""), width=8.27, height=5.83, horizontal=FALSE)
      plot (dResults$relmodutil, dResults$nor,
	  xlab=paste ("CPU utilization by", sMod),
	  ylab="Measured/predicted service times", cex.axis=1.5, cex.lab=1.5, cex.main=1.5)
      } else {
#	png (paste ("scatter-util-rel-svctimes-",sMod,".png",sep=""), width=400, height=400, bg="transparent", antialias="none")
	png (paste ("scatter-util-rel-svctimes-",sMod,".png",sep=""), width=1024, height=768)
      plot (dResults$relmodutil, dResults$nor,
	  xlab=paste ("CPU utilization by", sMod),
	  ylab="Measured/predicted service times")
	}
      dev.off()
    }
  }
}

plotBarplotModuleCounts <- function (lStats) {
  vOnes = rep (1, times=length (lStats$modcount))
  png ("barplot-modcount.png", width=1024, height=768)
  barplot (tapply (vOnes, INDEX=lStats$modallcount, FUN=sum))
  dev.off()
  png ("barplot-modcpucount.png", width=1024, height=768)
  barplot (tapply (vOnes, INDEX=lStats$modcount, FUN=sum))
  dev.off()
}

plotHistPCMTputPrecision <- function (lStats) {
  tStats <- lStats$main
  hist (abs(tStats$pcmTputMean / tStats$appTputMean - 1) * 100, main="Prediction error of mean throughput",
    xlab="Difference between SAM-predicted and measured mean throughput [%]", ylab="Number of architectures")
}

plotHistSimTputPrecision <- function (lStats) {
  tStats <- lStats$main
  hist (abs(tStats$simTputMean / tStats$appTputMean - 1) * 100, main="Prediction error of mean throughput",
    xlab="Difference between predicted and measured mean throughput [abs. %]", ylab="Number of architectures")
}

plotHistPCMRespPrecision <- function (lStats) {
  tStats <- lStats$main
  hist (abs(tStats$pcmRespMean / tStats$appRespMean - 1) * 100, main="Prediction error of mean response time",
    xlab="Difference between SAM-predicted and measured mean response times [%]", ylab="Number of architectures")
}

plotHistPCMQPNTputPrecision <- function (lStats) {
  tStats <- lStats$main
  hist (abs(tStats$pcmTputMean / tStats$simTputMean - 1) * 100, main="Prediction of mean throughput compared to SimQPN",
    xlab="Difference between SAM- and SimQPN-predicted mean throughput [%]", ylab="Number of architectures")
}


############################################################################
# Plotting trends with changing number of clients
############################################################################

# read the vector of client numbers from clients.list
readClients <- function (sDir=".") {
  return (sort (scan (paste (sDir, "clients.list", sep="/"), quiet=TRUE)))
}

# Plot a matrix with both measured and simulated mean response times for client numbers
plotResponseTimeTrends <- function (vClients) {

  mResults <- matrix (ncol=length (vClients), nrow=4)
  vMeasuredCIlo <- vector (mode="numeric", length=length (vClients));
  vMeasuredCIhi <- vector (mode="numeric", length=length (vClients));

  iCol = 1
  for (iClient in vClients) {
    tAppOutput <- readAppOutput (paste ("main-clients-", iClient, ".out", sep=""))
    tSimExpOutput <- readSimQPNOutput (paste ("sim-clients-", iClient, "-exp.out", sep=""))
    tSimNorOutput <- readSimQPNOutput (paste ("sim-clients-", iClient, "-nor.out", sep=""))

    fSimExpResp <- getSimMeanResponse (tSimExpOutput, iClient)

    fSimNorResp <- getSimMeanResponse (tSimNorOutput, iClient)

    tAppOutput <- filterAppMeasurements (tAppOutput)

    vResponseTimes <- tAppOutput[tAppOutput$measurementType == "monotonic", ]$measuredValue / 1000000000

    vResponseTimesThr <- tAppOutput[tAppOutput$measurementType == "threadtime", ]$measuredValue / 1000000000

    mResults [1, iCol] <- fSimExpResp
    mResults [2, iCol] <- fSimNorResp
    mResults [3, iCol] <- mean (vResponseTimes)
    # 95% Confidence Interval
    vCI <- t.test (vResponseTimes, conf.level=0.95)$conf.int
    vMeasuredCIlo [iCol] <- vCI [1]
    vMeasuredCIhi [iCol] <- vCI [2]

# TODO: might consider plotting this too?
#     bootMean <- boot (vResponseTimes, function(x,i) { mean(x[i]) }, R=1000)
#     vCI <- boot.ci (bootMean, type="perc", conf=0.95)$percent
#     vMeasuredCIlo [iCol] <- vCI [4]
#     vMeasuredCIhi [iCol] <- vCI [5]


    if (length (vResponseTimesThr) > 0) {
      mResults [4, iCol] <- mean (vResponseTimesThr)
    }
    
    iCol <- iCol + 1
  }

#   mClients <- mResults
#   save (mClients, file="mClients.Rdata")

  iLines <- 4
  vLegends <- c ("Simulated (exp)", "Simulated (nor)", "Measured (monotonic) 95% CI")
  if (TRUE %in% is.na (mResults [4,])) {
    mResults <- mResults [seq(1,3),]
    iLines <- 3
  } else {
    vLegends <- c (vLegends, "Measured (threadtime)")
  }

  clrs <- rich.colors (iLines)

  matplot (vClients, t(mResults), type="b", lty=1, lwd=2, pch=c(19,21,8,20,22:25), col=clrs,
    xlab="Number of clients", ylab="Response time [s]")
  plotSdArrows (vMeasuredCIlo, vMeasuredCIhi, vClients, TRUE)
  smartlegend (x="right", y="bottom", vLegends, fill=clrs, title="Result type")
}

# Plot a matrix with both measured and simulated throughputs for client numbers
plotThrputTrends <- function (vClients, sDir=".", bSim=TRUE, bPCM=TRUE, sPCMSubDir=".") {

  iLines <- 1
  vLegends <- c ("Measured")

  if (bSim == TRUE) {
    iLines <- iLines + 1
    iSimRow <- iLines
    vLegends <- c (vLegends, "Simulated (QPN)")
  }

  if (bPCM == TRUE) {
    iLines <- iLines + 1
    iPCMRow <- iLines
    vLegends <- c (vLegends, "Simulated (PCM)")
  }

  mResults <- matrix (ncol=length (vClients), nrow=iLines)
  
  iCol = 1
  for (iClient in vClients) {
    sOutputSuffix <- paste ("-clients-", iClient, sep="")
    tAppOutput <- readAppOutput (paste (sDir, "/main", sOutputSuffix, ".out", sep=""))

    tAppTimes <- getAppTimes (tAppOutput)

    fAppTput <- meanThrput (tAppTimes$client_end)

    mResults [1, iCol] <- fAppTput

    if (bSim) {
      tSimOutput <- readSimQPNOutput (paste (sDir, "/sim", sOutputSuffix, ".out", sep=""))
      fSimTput <- getSimMeanThroughput (tSimOutput)
      mResults [iSimRow, iCol] <- fSimTput
    }

    if (bPCM) {
      sPCMFile = paste ("main", sOutputSuffix, ".csv", sep="")
      tPCM <- readPCMOutput (paste (sDir, sPCMSubDir, sPCMFile, sep="/"))
      fPCMTputMean <- meanThrput (tPCM$client_end)
      mResults [iPCMRow, iCol] <- fPCMTputMean
    }

    iCol <- iCol + 1
  }

#  mClients <- mResults
#  save (mClients, file="mClients.Rdata")

  clrs <- rich.colors (iLines)

  matplot (vClients, t(mResults), type="l", lty=1, lwd=2, col=clrs, xlab="Number of clients", ylab="Throughput [1/s]")
  smartlegend (x="right", y="bottom", vLegends, fill=clrs, title="Result type")
}

############################################################################
# Main plotting functions
############################################################################

# Plot normalized module measurements
plotModuleMeasurements <- function (tAppOutput, bMonotonic) {
  tAppOutput <- filterModuleMeasurements (tAppOutput, bMonotonic)
  tAppOutput <- normalizeModuleTimes (tAppOutput)

  
  if (bMonotonic) {
    border = c ("black", "blue", "red")
  } else {
    border = c ("black", "red")
  }
  
  boxplot (tAppOutput[, "measuredValue"] ~ factor(tAppOutput[["moduleName"]]:factor(tAppOutput[["measurementType"]])), las = 2, border = border)
}

# Plot measured client response times
# @param tAppTimes - table of app times from getAppTimes()
plotAppResponseTimes <- function (tAppTimes) {

  plot (tAppTimes$client_time, main="Measured response time for individual clients", 
    sub="Time counted from client arrival to finished processing", xlab="Client number (order by arrival)", ylab="Response time [s]")
}

# Plot PCM-predicted client response times
# @param tPCM - table of PCM output from readPCMOutput()
plotPCMResponseTimes <- function (tPCM) {

  plot (tPCM$client_time, main="Predicted (PCM) response time for individual clients", 
    sub="Time counted from client arrival to finished processing", xlab="Client number (order by arrival)", ylab="Response time [s]")
}


# Plot measured thread response times
# @param tAppTimes - table of app times from getAppTimes()
plotAppThreadResponseTimes <- function (tAppTimes) {

  plot (tAppTimes$thread_time, main="Measured thread response time for individual clients", 
    sub="Time counted from client acquiring a thread to finished processing", xlab="Client number (order by arrival)", ylab="Thread response time [s]")
}

# Plot mean app throughput using a sliding window
# @param tAppTimes - table of app times from getAppTimes()
plotAppThrput <- function (tAppTimes) {
  
  # not sure what's the best value
  iWidth <- 30

  vRollTput <- rollapply (as.ts (tAppTimes$client_end), iWidth, meanThrput, na.pad = FALSE)
  
  plot (vRollTput, main="Measured mean client throughput using a sliding window", sub="Using a sliding window of 30 observations", ylab="Mean throughput [1/s]")
}

# Plot measured and PCM-simulated client response times together using density plot
plotAppPCMResponseTimesDensity <- function (tAppOutput, tPCM) {
  tAppTimes <- getAppTimes (tAppOutput)

  vApp <- tAppTimes$client_time
  vPCM <- tPCM$client_time

  vRange <- range (c (vApp, vPCM))
  plot( density(vApp), col="black", xlim=vRange, main="Measured and PCM-predicted client response times", xlab="")
  lines(density(vPCM), col="red")

  rug(vApp, col="black", ticksize=0.01, line=2.5)
  rug(vPCM, col="red", ticksize=0.01, line=3.0)

  smartlegend (x="right", y="top", c("Measured", "Pred. (PCM)"), fill=c("black", "red"), title="Client resp. times [s]")
}

# Plot measured and PCM-simulated client response times together using superimposed histograms
plotAppPCMResponseTimesHist <- function (tAppOutput, tPCM) {
  tAppTimes <- getAppTimes (tAppOutput)

  vApp <- tAppTimes$client_time
  vPCM <- tPCM$client_time

  iNumSets = 2
  vCols <- hcl (h = seq (30, by = 360 / iNumSets, length = iNumSets), l = 65, alpha = 0.5)

  multihist (list (vApp, vPCM), vCols, main="Measured and PCM-predicted client response times", xlab="Client resp. times [s]")

  smartlegend (x="right", y="top", c("Measured", "Pred. (PCM)"), fill=vCols, title="Client resp. times [s]")
}

# Plot times between consecutive processing finish times
# @param tAppTimes - table of app times from getAppTimes()
plotAppFinishTimes <- function (tAppTimes) {
  # use rollapply with width 2 to get differences between consecutive times in a simple way
  vEndDelta <- rollapply (as.ts (tAppTimes$client_end), 2, function (vT) { return (vT[2] - vT[1]) }, na.pad = FALSE)
  plot (vEndDelta, main="Time between consecutive finished client requests", ylab="Time between two consecutive finished requests [s]")
}

# Plot mean clientresponse times in app and simulation side-by-side
plotMeanResponseTimes <- function (tAppOutput, tSimExpOutput, tSimNorOutput, tPCMOutput) {
  
  tAppTimes <- getAppTimes (tAppOutput)

  iClients <- getAppClients (tAppOutput)

  fSimExpResp <- getSimMeanResponse (tSimExpOutput, iClients)

  if (! is.null (tSimNorOutput)) {
    fSimNorResp <- getSimMeanResponse (tSimNorOutput, iClients)
  }	

  # TODO: reinstate if needed
  #vResponseTimesThr <- tAppOutput[tAppOutput$measurementType == "threadtime", ]$measuredValue / 1000000000
  vResponseTimesThr <- c ()

  # build up parameters from what data is available  
  values = list (tAppTimes$client_time)
  border = c ("black")
  names = c ("Measured")

  if (length (vResponseTimesThr) > 0) {
    values = c (values, list (vResponseTimesThr))
    border = c (border, "green")
    names = c (names, "measured_thr")
  }

  if (! is.null (tSimNorOutput)) {
    values = c (values, list (fSimNorResp, fSimExpResp))
    border = c (border, "red", "blue")
    names = c (names, "SimQPN (nor)", "SimQPN (exp)")
  } else {
    values = c (values, list (fSimExpResp))
    border = c (border, "red")
    names = c (names, "SimQPN")
  }

  if (! is.null (tPCMOutput)) {
    values = c (values, list (tPCMOutput$client_time))
    border = c (border, "green")
    names = c (names, "PCM")
  }

  boxplotWithMeanSd (values, border=border, names=names, main="Measured and predicted client response time summary", xlab="Source of data", ylab="Client response time [s]")
}

# Plot mean throughputs in app and simulation side-by-side
plotMeanThrputs <- function (tAppOutput, tSimExpOutput, tSimNorOutput, tPCMOutput) {

  tAppTimes <- getAppTimes (tAppOutput)

  fSimExpTput <- getSimMeanThroughput (tSimExpOutput)
  if (! is.null (tSimNorOutput)) {
    fSimNorTput <- getSimMeanThroughput (tSimNorOutput)
  }	

  # build up parameters from what data is available
  fAppTput <- meanThrput (tAppTimes$client_end)
  values = c (fAppTput)
  border = c ("black")
  names = c ("Measured")

  if (! is.null (tSimNorOutput)) {
    values = c (values, fSimNorTput, fSimExpTput)
    border = c (border, "red", "blue")
    names = c (names, "SimQPN (nor)", "SimQPN (exp)")
  } else {
    values = c (values, fSimExpTput)
    border = c (border, "red")
    names = c (names, "SimQPN")
  }

  if (! is.null (tPCMOutput)) {
    fPCMTput <- meanThrput (tPCMOutput$client_end)
    values = c (values, fPCMTput)
    border = c (border, "green")
    names = c (names, "PCM")
  }

  # how to summarize measured throughput variability?
  barplot (values, col=border, names.arg=names, main="Measured and predicted client throughput summary", xlab="Source of data", ylab="Mean throughput [1/s]")

}
