/*******************************************************************************
 * Copyright (c) 2008,2009 Q-ImPrESS consortium
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * This work was funded in the context of the Q-ImPrESS research project  
 * (FP7-215013) by the European Union under the Information and  
 * Communication Technologies priority of the Seventh Research Framework  
 * Programme.
 *
 * Contributors:
 *     itemis 	- initial API and implementation
 *******************************************************************************/
package de.itemis.qimpress.showcase.simulation_manager.jsf.beans;

import java.util.Timer;
import java.util.TimerTask;

import javax.naming.NamingException;

import org.apache.log4j.Logger;

import com.icesoft.faces.async.render.RenderManager;

import de.itemis.qimpress.showcase.simulation_manager.be.service.ServiceLocator;
import de.itemis.qimpress.showcase.simulation_manager.be.service.SimulationManager;

/**
 * @author Claudius Haecker
 * @author Wladimir Safonov
 *
 */
public class StatusPoller {

    private static final Logger LOG = Logger.getLogger(StatusPoller.class);

    // interval in milliseconds at which the status is updated
    private static final long POLLING_INTERVAL = 2000;

    // must be accessed synchronized
    private boolean isOrderSimulatorRunning = false;
    private boolean isShipmentSimulatorRunning = false;

    private SimulationManager simulationManager;
    private RenderManager renderManager;

    /**
     * 
     */
    public StatusPoller() {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">>  StatusPoller");
        }
        this.simulationManager = getSimulationManager();
        // Timer(true) means: thread will not prevent tomcat from shutting down.
        new Timer(true).schedule(new TimerTask() {

            public void run() {
                boolean newOrderSimulatorStatus = false, newShipmentSimulatorStatus = false;
                try {
                    newOrderSimulatorStatus = simulationManager.isOrderSimulatorRunning();
                    newShipmentSimulatorStatus = simulationManager.isShipmentSimulatorRunning();
                } catch (Exception e) {
                    LOG.error("Failed to get simulators status, simulators may still be running.", e);
                } finally {
                    //                    if (LOG.isDebugEnabled()) {
                    //                        LOG.debug("run(): Simulators are " + (areRunning ? "" : "NOT ") + "running.");
                    //                    }
                    // renderManager might be not set yet
                    if (renderManager != null
                            && updateSimulatorsStatus(newOrderSimulatorStatus, newShipmentSimulatorStatus)) {
                        if (LOG.isDebugEnabled()) {
                            LOG.debug("Status has changed. Requesting render.");
                        }
                        renderManager.getOnDemandRenderer("simulatorsStatus").requestRender();
                    }
                }
            }
        }, 0, POLLING_INTERVAL);
    }

    /**
     * Updates the local simulators status with the given values. Return true,
     * if at least one status value has changed.
     * 
     * @param newOrderSimulatorStatus new status value for Order Simulator to set
     * @param newShipmentSimulatorStatus new status value for Shipment Simulator to set
     * @return true, if at least one status has changed
     */
    private synchronized boolean updateSimulatorsStatus(boolean newOrderSimulatorStatus,
            boolean newShipmentSimulatorStatus) {
        if (newOrderSimulatorStatus == isOrderSimulatorRunning
                && newShipmentSimulatorStatus == isShipmentSimulatorRunning) {
            return false;
        } else {
            isOrderSimulatorRunning = newOrderSimulatorStatus;
            isShipmentSimulatorRunning = newShipmentSimulatorStatus;
            return true;
        }
    }

    private SimulationManager getSimulationManager() {
        SimulationManager localSimulationManager = ServiceLocator.getInstance().getSimulationManager();
        if (localSimulationManager == null) {
            LOG.error("simulationManager from ServiceLocater was null");
            throw new IllegalStateException("Could not configure simulationManager.");
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("simulationManager = " + localSimulationManager);
        }
        return localSimulationManager;
    }

    /**
     * @return true, if Order Simulator was running during the last check
     */
    public synchronized boolean isOrderSimulatorRunning() {
        if (LOG.isDebugEnabled()) {
            LOG.debug("isOrderSimulatorRunning(): OS is " + (isOrderSimulatorRunning ? "" : "NOT ") + "running.");
        }
        return isOrderSimulatorRunning;
    }

    /**
     * @return true, if Shipment Simulator was running during the last check
     */
    public synchronized boolean isShipmentSimulatorRunning() {
        if (LOG.isDebugEnabled()) {
            LOG.debug("isShipmentSimulatorRunning(): SS is " + (isShipmentSimulatorRunning ? "" : "NOT ") + "running.");
        }
        return isShipmentSimulatorRunning;
    }

    /**
     * @param renderManager the renderManager to set
     */
    public void setRenderManager(RenderManager renderManager) {
        this.renderManager = renderManager;
    }
}
