/*******************************************************************************
 * Copyright (c) 2008,2009 Q-ImPrESS consortium
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * This work was funded in the context of the Q-ImPrESS research project  
 * (FP7-215013) by the European Union under the Information and  
 * Communication Technologies priority of the Seventh Research Framework  
 * Programme.
 *
 * Contributors:
 *     itemis 	- initial API and implementation
 *******************************************************************************/
package de.itemis.qimpress.showcase.simulation_manager.be.service;


import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Properties;

import org.apache.log4j.Logger;
import org.jdom.Element;
import org.jdom.Namespace;
import org.jdom.transform.JDOMResult;
import org.jdom.transform.JDOMSource;
import org.springframework.ws.client.core.support.WebServiceGatewaySupport;

import de.itemis.qimpress.showcase.simulation_manager.be.dao.ParametersDao;
import de.itemis.qimpress.showcase.simulation_manager.be.domain.TimeParameters;
import de.itemis.qimpress.showcase.simulation_manager.be.exceptions.ApplicationException;
import de.itemis.qimpress.showcase.simulation_manager.be.exceptions.DaoException;
import de.itemis.qimpress.showcase.simulation_manager.be.exceptions.ApplicationException.ApplicationErrorCode;

/**
 * @author Wladimir Safonov
 *
 */
public class SimulationManagerImpl extends WebServiceGatewaySupport implements SimulationManager {

    private static final String OS_WS_URI = "os_ws_uri";
    private static final String SS_WS_URI = "ss_ws_uri";

    private static final Logger LOG = Logger.getLogger(SimulationManagerImpl.class);

    private static final String START_ORDER_SIMULATOR_REQUEST = "StartOrderSimulatorRequest";
    private static final String STOP_ORDER_SIMULATOR_REQUEST = "StopOrderSimulatorRequest";
    private static final String GET_ORDER_SIMULATOR_STATUS_REQUEST = "GetOrderSimulatorStatusRequest";
    private static final String START_SHIPMENT_SIMULATOR_REQUEST = "StartShipmentSimulatorRequest";
    private static final String STOP_SHIPMENT_SIMULATOR_REQUEST = "StopShipmentSimulatorRequest";
    private static final String GET_SHIPMENT_SIMULATOR_STATUS_REQUEST = "GetShipmentSimulatorStatusRequest";
    
    private static final String STATUS_RUNNING = "RUNNING";
    
    private static final DateFormat SDF = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
    
    private ParametersDao parametersDao;

    private Properties webServiceURIs;
    private Namespace orderSimulatorNS;
    private Namespace shipmentSimulatorNS;
    
    /**
     * standard constructor inits variables
     */
    public SimulationManagerImpl() {
        webServiceURIs = new Properties();
        try {
            webServiceURIs.load(this.getClass().getClassLoader().getResourceAsStream("webservice.properties"));
        } catch (Exception e) {
            LOG.error("Failed to load webservice.properties file", e);
        }    
        orderSimulatorNS = Namespace.getNamespace("os", "http://www.itemis.com/order_simulator/schemas");
        shipmentSimulatorNS = Namespace.getNamespace("ss", "http://www.itemis.com/shipment_simulator/schemas");
    }

    /**
     * @param parametersDao the parametersDao to set
     */
    public void setParametersDao(ParametersDao parametersDao) {
        this.parametersDao = parametersDao;
    }

    /* @Override */
    public boolean isOrderSimulatorRunning() throws ApplicationException {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">> isOrderSimulatorRunning()");
        }

        // prepare request source
        Element request = new Element(GET_ORDER_SIMULATOR_STATUS_REQUEST, orderSimulatorNS);
        JDOMSource source = new JDOMSource(request);
        
        // create response result
        JDOMResult result = new JDOMResult();
        
        // invoke webservice
        getWebServiceTemplate().sendSourceAndReceiveToResult(webServiceURIs.getProperty(OS_WS_URI), source, result);

        // parse result
        Element response = result.getDocument().getRootElement();
        if (!response.getNamespaceURI().equals(orderSimulatorNS.getURI())) {
            LOG.error("Received response element with wrong namespace! Expected: '" + orderSimulatorNS.getURI() + 
                    "', was: '" + response.getNamespaceURI() + "'");
            throw new ApplicationException(ApplicationErrorCode.TE);
        }
        Element statusResult = response.getChild("StatusResult", orderSimulatorNS);
        if (null == statusResult) {
            LOG.error("Received response element with no StatusResult child element!");
            throw new ApplicationException(ApplicationErrorCode.TE);
        }
        String statusString = statusResult.getTextTrim();
        if (null == statusString || "".equals(statusString)) {
            LOG.error("Received response element with empty StatusResult child element!");
            throw new ApplicationException(ApplicationErrorCode.TE);
        }
        
        return STATUS_RUNNING.equals(statusString);
    }

    /* @Override */
    public boolean isShipmentSimulatorRunning() throws ApplicationException {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">> isShipmentSimulatorRunning()");
        }
        
        // prepare request source
        Element request = new Element(GET_SHIPMENT_SIMULATOR_STATUS_REQUEST, shipmentSimulatorNS);
        JDOMSource source = new JDOMSource(request);
        
        // create response result
        JDOMResult result = new JDOMResult();
        
        // invoke webservice
        getWebServiceTemplate().sendSourceAndReceiveToResult(webServiceURIs.getProperty(SS_WS_URI), source, result);

        // parse result
        Element response = result.getDocument().getRootElement();
        if (!response.getNamespaceURI().equals(shipmentSimulatorNS.getURI())) {
            LOG.error("Received response element with wrong namespace! Expected: '" + shipmentSimulatorNS.getURI() + 
                    "', was: '" + response.getNamespaceURI() + "'");
            throw new ApplicationException(ApplicationErrorCode.TE);
        }
        Element statusResult = response.getChild("StatusResult", shipmentSimulatorNS);
        if (null == statusResult) {
            LOG.error("Received response element with no StatusResult child element!");
            throw new ApplicationException(ApplicationErrorCode.TE);
        }
        String statusString = statusResult.getTextTrim();
        if (null == statusString || "".equals(statusString)) {
            LOG.error("Received response element with empty StatusResult child element!");
            throw new ApplicationException(ApplicationErrorCode.TE);
        }
        
        return STATUS_RUNNING.equals(statusString);
    }

    /* @Override */
    public TimeParameters loadTimeParameters() throws ApplicationException {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">> loadTimeParameters()");
        }
        
        try {
            return parametersDao.loadTimeParameters();
        } catch (DaoException e) {
            throw new ApplicationException(ApplicationErrorCode.TE);
        }
    }

    /* @Override */
    public boolean saveTimeParameters(TimeParameters timeParameters) throws ApplicationException {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">> saveTimeParameters()");
        }
        
        try {
            parametersDao.storeTimeParameters(timeParameters);
        } catch (DaoException e) {
            throw new ApplicationException(ApplicationErrorCode.TE);
        }
        return true;
    }

    /* @Override */
    public boolean startSimulators(TimeParameters timeParameters) throws ApplicationException {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">> startSimulators()");
        }
        
        boolean orderSimulatorStarted = startOrderSimulator(timeParameters);
        boolean shipmentSimulatorStarted = startShipmentSimulator(timeParameters);

        return orderSimulatorStarted && shipmentSimulatorStarted;
    }

    /* @Override */
    public boolean stopSimulators() throws ApplicationException {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">> stopSimulators()");
        }
        
        boolean orderSimulatorStopped = stopOrderSimulator();
        boolean shipmentSimulatorStopped = stopShipmentSimulator();
        
        return orderSimulatorStopped && shipmentSimulatorStopped;
    }
    
    protected boolean startOrderSimulator(TimeParameters timeParameters) throws ApplicationException {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">> startOrderSimulator()");
        }
        
        // prepare request source
        Element request = createStartOrderSimulatorRequest(timeParameters);
        JDOMSource source = new JDOMSource(request);
        
        // create response result
        JDOMResult result = new JDOMResult();
        
        // invoke webservice
        getWebServiceTemplate().sendSourceAndReceiveToResult(webServiceURIs.getProperty(OS_WS_URI), source, result);

        // parse result
        Element response = result.getDocument().getRootElement();
        if (!response.getNamespaceURI().equals(orderSimulatorNS.getURI())) {
            LOG.error("Received response element with wrong namespace! Expected: '" + orderSimulatorNS.getURI() + 
                    "', was: '" + response.getNamespaceURI() + "'");
            throw new ApplicationException(ApplicationErrorCode.TE);
        }
        Element startResult = response.getChild("StartResult", orderSimulatorNS);
        if (null == startResult) {
            LOG.error("Received response element with no StartResult child element!");
            throw new ApplicationException(ApplicationErrorCode.TE);
        }
        String startResultString = startResult.getTextTrim();
        if (null == startResultString || "".equals(startResultString)) {
            LOG.error("Received response element with empty StartResult child element!");
            throw new ApplicationException(ApplicationErrorCode.TE);
        }
        
        return Boolean.parseBoolean(startResultString);
    }
    
    protected boolean startShipmentSimulator(TimeParameters timeParameters) throws ApplicationException {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">> startShipmentSimulator()");
        }
        
        // prepare request source
        Element request = createStartShipmentSimulatorRequest(timeParameters);
        JDOMSource source = new JDOMSource(request);
        
        // create response result
        JDOMResult result = new JDOMResult();
        
        // invoke webservice
        getWebServiceTemplate().sendSourceAndReceiveToResult(webServiceURIs.getProperty(SS_WS_URI), source, result);
        
        // parse result
        Element response = result.getDocument().getRootElement();
        if (!response.getNamespaceURI().equals(shipmentSimulatorNS.getURI())) {
            LOG.error("Received response element with wrong namespace! Expected: '" + shipmentSimulatorNS.getURI() + 
                    "', was: '" + response.getNamespaceURI() + "'");
            throw new ApplicationException(ApplicationErrorCode.TE);
        }
        Element startResult = response.getChild("StartResult", shipmentSimulatorNS);
        if (null == startResult) {
            LOG.error("Received response element with no StartResult child element!");
            throw new ApplicationException(ApplicationErrorCode.TE);
        }
        String startResultString = startResult.getTextTrim();
        if (null == startResultString || "".equals(startResultString)) {
            LOG.error("Received response element with empty StartResult child element!");
            throw new ApplicationException(ApplicationErrorCode.TE);
        }
        
        return Boolean.parseBoolean(startResultString);
    }
    
    protected boolean stopOrderSimulator() throws ApplicationException {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">> stopOrderSimulator()");
        }
        
        // prepare request source
        Element request = new Element(STOP_ORDER_SIMULATOR_REQUEST, orderSimulatorNS);
        JDOMSource source = new JDOMSource(request);
        
        // create response result
        JDOMResult result = new JDOMResult();
        
        // invoke webservice
        getWebServiceTemplate().sendSourceAndReceiveToResult(webServiceURIs.getProperty(OS_WS_URI), source, result);

        // parse result
        Element response = result.getDocument().getRootElement();
        if (!response.getNamespaceURI().equals(orderSimulatorNS.getURI())) {
            LOG.error("Received response element with wrong namespace! Expected: '" + orderSimulatorNS.getURI() + 
                    "', was: '" + response.getNamespaceURI() + "'");
            throw new ApplicationException(ApplicationErrorCode.TE);
        }
        Element stopResult = response.getChild("StopResult", orderSimulatorNS);
        if (null == stopResult) {
            LOG.error("Received response element with no StopResult child element!");
            throw new ApplicationException(ApplicationErrorCode.TE);
        }
        String stopResultString = stopResult.getTextTrim();
        if (null == stopResultString || "".equals(stopResultString)) {
            LOG.error("Received response element with empty StopResult child element!");
            throw new ApplicationException(ApplicationErrorCode.TE);
        }
        
        return Boolean.parseBoolean(stopResultString);
    }
    
    protected boolean stopShipmentSimulator() throws ApplicationException {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">> stopShipmentSimulator()");
        }
        
        // prepare request source
        Element request = new Element(STOP_SHIPMENT_SIMULATOR_REQUEST, shipmentSimulatorNS);
        JDOMSource source = new JDOMSource(request);
        
        // create response result
        JDOMResult result = new JDOMResult();
        
        // invoke webservice
        getWebServiceTemplate().sendSourceAndReceiveToResult(webServiceURIs.getProperty(SS_WS_URI), source, result);
        
        // parse result
        Element response = result.getDocument().getRootElement();
        if (!response.getNamespaceURI().equals(shipmentSimulatorNS.getURI())) {
            LOG.error("Received response element with wrong namespace! Expected: '" + shipmentSimulatorNS.getURI() + 
                    "', was: '" + response.getNamespaceURI() + "'");
            throw new ApplicationException(ApplicationErrorCode.TE);
        }
        Element stopResult = response.getChild("StopResult", shipmentSimulatorNS);
        if (null == stopResult) {
            LOG.error("Received response element with no StopResult child element!");
            throw new ApplicationException(ApplicationErrorCode.TE);
        }
        String stopResultString = stopResult.getTextTrim();
        if (null == stopResultString || "".equals(stopResultString)) {
            LOG.error("Received response element with empty StopResult child element!");
            throw new ApplicationException(ApplicationErrorCode.TE);
        }
        
        return Boolean.parseBoolean(stopResultString);
    }
    
    protected Element createStartOrderSimulatorRequest(TimeParameters timeParameters) {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">> createStartOrderSimulatorRequest()");
        }
        
        Element request = new Element(START_ORDER_SIMULATOR_REQUEST, orderSimulatorNS);
        request.addContent(new Element("GeneratorTimeMode", orderSimulatorNS)
                    .setText(timeParameters.getGeneratorTimeMode().toString()));
        request.addContent(new Element("StartTimeForOrders", orderSimulatorNS)
                    .setText(SDF.format(timeParameters.getStartTimeForOrders())));
        request.addContent(new Element("TimeScaleFactor", orderSimulatorNS)
                    .setText(Float.toString(timeParameters.getTimeScaleFactor())));
        
        return request;
    }
    
    protected Element createStartShipmentSimulatorRequest(TimeParameters timeParameters) {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">> createStartShipmentSimulatorRequest()");
        }
        
        Element request = new Element(START_SHIPMENT_SIMULATOR_REQUEST, shipmentSimulatorNS);
        request.addContent(new Element("GeneratorTimeMode", shipmentSimulatorNS)
                    .setText(timeParameters.getGeneratorTimeMode().toString()));
        request.addContent(new Element("StartTimeForOrders", shipmentSimulatorNS)
                    .setText(SDF.format(timeParameters.getStartTimeForOrders())));
        request.addContent(new Element("TimeScaleFactor", shipmentSimulatorNS)
                    .setText(Float.toString(timeParameters.getTimeScaleFactor())));
        
        return request;
    }
    
}
