/*******************************************************************************
 * Copyright (c) 2008,2009 Q-ImPrESS consortium
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * This work was funded in the context of the Q-ImPrESS research project  
 * (FP7-215013) by the European Union under the Information and  
 * Communication Technologies priority of the Seventh Research Framework  
 * Programme.
 *
 * Contributors:
 *     itemis 	- initial API and implementation
 *******************************************************************************/
package de.itemis.qimpress.showcase.simulation_manager.be.dao;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;

import org.apache.log4j.Logger;
import org.springframework.dao.DataAccessException;
import org.springframework.jdbc.core.ResultSetExtractor;
import org.springframework.jdbc.core.support.JdbcDaoSupport;

import de.itemis.qimpress.showcase.simulation_manager.be.domain.TimeParameters;
import de.itemis.qimpress.showcase.simulation_manager.be.domain.TimeParameters.GeneratorTimeMode;
import de.itemis.qimpress.showcase.simulation_manager.be.exceptions.DaoException;

/**
 * Implementation of the {@link ParametersDao} with plain JDBC (using <code>JdbcDaoSupport</code>).
 * 
 * @author Wladimir Safonov
 *
 */
public class ParametersDaoImpl extends JdbcDaoSupport implements ParametersDao {

    private static final Logger LOG = Logger.getLogger(ParametersDaoImpl.class);
    
    private static final String GENERATOR_TIME_MODE = "GENERATOR_TIME_MODE";
    private static final String ORDERS_STARTING_TIME = "ORDERS_STARTING_TIME";
    private static final String TIME_SCALE_FACTOR = "TIME_SCALE_FACTOR";
    
    private static final String SELECT_PARAMETERS = "SELECT * FROM simulation_parameters";
    private static final String UPDATE_PARAMETERS = "UPDATE simulation_parameters SET parameter_value=? WHERE parameter_name=?";
    
    private static final DateFormat DF = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
    
    private ResultSetExtractor timeParametersExtractor;
    
    /* (non-Javadoc)
     * @see de.itemis.qimpress.showcase.simulation_manager.be.dao.ParametersDao#loadTimeParameters()
     */
    public ParametersDaoImpl() {
        super();
        this.timeParametersExtractor = new ResultSetExtractor() {
        
            /* @Override */
            public Object extractData(ResultSet rs) throws SQLException, DataAccessException {
                
                TimeParameters timeParameters = new TimeParameters();
                while (rs.next()) {
                    String paramName = rs.getString("parameter_name");
                    String paramValue = rs.getString("parameter_value");
                    
                    try {
                        if (GENERATOR_TIME_MODE.equals(paramName)) {
                            timeParameters.setGeneratorTimeMode(GeneratorTimeMode.valueOf(paramValue));
                        } else if (ORDERS_STARTING_TIME.equals(paramName)) {
                            timeParameters.setStartTimeForOrders(DF.parse(paramValue));
                        } else if (TIME_SCALE_FACTOR.equals(paramName)) {
                            timeParameters.setTimeScaleFactor(Float.parseFloat(paramValue));
                        }
                    } catch (Exception e) {
                        LOG.error("Failed to parse time parameter '" + paramName + "', value='" + paramValue + "'", e);
                    }
                }
                return timeParameters;
            }
        };
    }

    /* @Override */
    public TimeParameters loadTimeParameters() throws DaoException {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">> loadTimeParameters()");
        }
        
        TimeParameters timeParameters = null;
        try {
            timeParameters = (TimeParameters) this.getJdbcTemplate().query(SELECT_PARAMETERS, this.timeParametersExtractor);
        } catch (DataAccessException e) {
            LOG.error("Failed to load TimeParameters", e);
            throw new DaoException();
        }
        
        return timeParameters;
    }

    /* (non-Javadoc)
     * @see de.itemis.qimpress.showcase.simulation_manager.be.dao.ParametersDao#storeTimeParameters()
     */
    /* @Override */
    public void storeTimeParameters(TimeParameters timeParameters) throws DaoException {
        
        this.getJdbcTemplate().update(UPDATE_PARAMETERS, 
                new Object[] {timeParameters.getGeneratorTimeMode().toString(), GENERATOR_TIME_MODE});
        this.getJdbcTemplate().update(UPDATE_PARAMETERS, 
                new Object[] {DF.format(timeParameters.getStartTimeForOrders()), ORDERS_STARTING_TIME});
        this.getJdbcTemplate().update(UPDATE_PARAMETERS, 
                new Object[] {timeParameters.getTimeScaleFactor(), TIME_SCALE_FACTOR});
    }
    
}
