/*******************************************************************************
 * Copyright (c) 2008,2009 Q-ImPrESS consortium
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * This work was funded in the context of the Q-ImPrESS research project  
 * (FP7-215013) by the European Union under the Information and  
 * Communication Technologies priority of the Seventh Research Framework  
 * Programme.
 *
 * Contributors:
 *     itemis 	- initial API and implementation
 *******************************************************************************/
package de.itemis.qimpress.showcase.shipment_simulator.webservice;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;

import javax.naming.NamingException;

import static org.apache.commons.lang.Validate.*;

import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import org.jdom.Element;
import org.jdom.Namespace;
import org.springframework.ws.server.endpoint.AbstractJDomPayloadEndpoint;

import de.itemis.qimpress.showcase.shipment_simulator.be.manager.ServiceLocator;
import de.itemis.qimpress.showcase.shipment_simulator.be.manager.ShipmentSimulatorManager;
import de.itemis.qimpress.showcase.shipment_simulator.be.manager.model.TimeParameters;
import de.itemis.qimpress.showcase.shipment_simulator.be.manager.model.TimeParameters.GeneratorTimeMode;


/**
 * @author Wladimir Safonov
 *
 */
public class ShipmentSimulatorJDomEndpoint extends AbstractJDomPayloadEndpoint {

    private static final Logger LOG = Logger.getLogger(ShipmentSimulatorJDomEndpoint.class);
    
    private static final String START_SHIPMENT_SIMULATOR_REQUEST = "StartShipmentSimulatorRequest";
    private static final String START_SHIPMENT_SIMULATOR_RESPONSE = "StartShipmentSimulatorResponse";
    private static final String STOP_SHIPMENT_SIMULATOR_REQUEST = "StopShipmentSimulatorRequest";
    private static final String STOP_SHIPMENT_SIMULATOR_RESPONSE = "StopShipmentSimulatorResponse";
    private static final String GET_SHIPMENT_SIMULATOR_STATUS_REQUEST = "GetShipmentSimulatorStatusRequest";
    private static final String GET_SHIPMENT_SIMULATOR_STATUS_RESPONSE = "GetShipmentSimulatorStatusResponse";
    
    private static final String STATUS_RUNNING = "RUNNING";
    private static final String STATUS_STOPPED = "STOPPED";
    
    private static final ShipmentSimulatorManager SHIPMENT_SIMULATOR_MANAGER = ServiceLocator.getInstance().getShipmentSimulatorManager();
    
    private Namespace namespace;
    private final DateFormat sdf1 = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
    private final DateFormat sdf2 = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSS");
    
    /**
     * Inits global variables
     */
    public ShipmentSimulatorJDomEndpoint() {
        super();
        
        namespace = Namespace.getNamespace("ss", "http://www.itemis.com/shipment_simulator/schemas");
    }

    @Override
    protected Element invokeInternal(Element request) throws Exception {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">> invokeInternal()");
        }
        
        String rootElement = request.getName();
        if (START_SHIPMENT_SIMULATOR_REQUEST.equals(rootElement)) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Detected root element: " + START_SHIPMENT_SIMULATOR_REQUEST);
            }
            
            return processStartShipmentSimulatorRequest(request);
        } else if (STOP_SHIPMENT_SIMULATOR_REQUEST.equals(rootElement)) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Detected root element: " + STOP_SHIPMENT_SIMULATOR_REQUEST);
            }
            
            return processStopShipmentSimulatorRequest(request);
        } else if (GET_SHIPMENT_SIMULATOR_STATUS_REQUEST.equals(rootElement)) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Detected root element: " + GET_SHIPMENT_SIMULATOR_STATUS_REQUEST);
            }
            
            return processGetShipmentSimulatorStatusRequest(request);
        }
        return null;
    }

    
    protected Element processStartShipmentSimulatorRequest(Element request) throws Exception {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">> processStartShipmentSimulatorRequest()");
        }
        
        // pre-validate parameters
        notEmpty(request.getChildTextTrim("GeneratorTimeMode", namespace), "'GeneratorTimeMode' parameter is empty");
        
        // unmarshall input parameters
        TimeParameters timeParameters = new TimeParameters();
        GeneratorTimeMode timeMode = null;
        String timeModeString = request.getChildTextTrim("GeneratorTimeMode", namespace);
        try {
            timeMode = GeneratorTimeMode.valueOf(timeModeString);
        } catch (Exception e) {
            String errMsg = "'GeneratorTimeMode' parameter has invalid value: " + timeModeString;
            LOG.error(errMsg);
            throw new IllegalArgumentException(errMsg);
        }
        Date startTimeForOrders = null;
        String startTimeString = request.getChildTextTrim("StartTimeForOrders", namespace);
        if (StringUtils.isNotEmpty(startTimeString)) {
            try {
                startTimeForOrders = sdf1.parse(startTimeString);
            } catch (ParseException e) {
                // try next format
                try {
                    startTimeForOrders = sdf2.parse(startTimeString);
                } catch (ParseException e1) {
                    String errMsg = "'StartTimeForOrders' parameter has wrong format: " + startTimeString;
                    LOG.error(errMsg);
                    throw new IllegalArgumentException(errMsg);
                }
            }
        }
        float timeScaleFactor = 0.0f;
        String timeScaleString = request.getChildTextTrim("TimeScaleFactor", namespace);
        if (StringUtils.isNotEmpty(timeScaleString)) {
            try {
                timeScaleFactor = Float.parseFloat(timeScaleString);
            } catch (NumberFormatException e) {
                String errMsg = "'TimeScaleFactor' parameter has invalid value: " + timeScaleString;
                LOG.error(errMsg);
                throw new IllegalArgumentException(errMsg);
            }
        }
        timeParameters.setGeneratorTimeMode(timeMode);
        timeParameters.setStartTimeForOrders(startTimeForOrders);
        timeParameters.setTimeScaleFactor(timeScaleFactor);
        
        // validate TimeParameters
        if (GeneratorTimeMode.SIMULATED_TIME == timeParameters.getGeneratorTimeMode()) {
            notNull(timeParameters.getStartTimeForOrders(), "'StartTimeForOrders' can not be empty for SIMULATED_TIME mode");
            isTrue(timeParameters.getTimeScaleFactor() != 0.0f, "'TimeScaleFactor' can not be empty for SIMULATED_TIME mode");
        }
        
        // start order generator
        boolean isStarted = SHIPMENT_SIMULATOR_MANAGER.startShipmentGenerator(timeParameters);
        
        // create response element
        Element response = new Element(START_SHIPMENT_SIMULATOR_RESPONSE, namespace);
        Element result = new Element("StartResult", namespace).setText(Boolean.toString(isStarted));
        response.addContent(result);
        
        return response;
    }
    
    protected Element processStopShipmentSimulatorRequest(Element request) throws Exception {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">> processStopShipmentSimulatorRequest()");
        }

        // no input parameters => process request directly
        
        // stop order generator
        boolean isStopped = SHIPMENT_SIMULATOR_MANAGER.stopShipmentGenerator();
        
        // create response element
        Element response = new Element(STOP_SHIPMENT_SIMULATOR_RESPONSE, namespace);
        Element result = new Element("StopResult", namespace).setText(Boolean.toString(isStopped));
        response.addContent(result);
        
        return response;
    }
    
    protected Element processGetShipmentSimulatorStatusRequest(Element request) throws Exception {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">> processGetShipmentSimulatorStatusRequest()");
        }
        
        // no input parameter => process request directly
        
        // check order generator status
        boolean isRunning = SHIPMENT_SIMULATOR_MANAGER.isShipmentGeneratorRunning();
        
        // create response element
        Element response = new Element(GET_SHIPMENT_SIMULATOR_STATUS_RESPONSE, namespace);
        Element result = new Element("StatusResult", namespace).setText(isRunning ? STATUS_RUNNING : STATUS_STOPPED);
        response.addContent(result);
        
        return response;
    }
}
