/*******************************************************************************
 * Copyright (c) 2008,2009 Q-ImPrESS consortium
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * This work was funded in the context of the Q-ImPrESS research project  
 * (FP7-215013) by the European Union under the Information and  
 * Communication Technologies priority of the Seventh Research Framework  
 * Programme.
 *
 * Contributors:
 *     itemis 	- initial API and implementation
 *******************************************************************************/
package de.itemis.qimpress.showcase.shipment_simulator.be;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.fail;

import java.util.Date;
import java.util.LinkedList;
import java.util.List;
import java.util.Random;

import org.apache.log4j.Logger;
import org.junit.Before;
import org.junit.Test;
import org.springframework.beans.BeansException;
import org.springframework.context.ApplicationContext;
import org.springframework.context.support.ClassPathXmlApplicationContext;
import org.springframework.transaction.annotation.Transactional;

import de.itemis.qimpress.showcase.shipment_simulator.be.dao.OrderReportDao;
import de.itemis.qimpress.showcase.shipment_simulator.be.domain.ItemShipping;
import de.itemis.qimpress.showcase.shipment_simulator.be.domain.OrderShipping;
import de.itemis.qimpress.showcase.shipment_simulator.be.domain.Shipment;
import de.itemis.qimpress.showcase.shipment_simulator.be.domain.ShipmentItem;
import de.itemis.qimpress.showcase.shipment_simulator.be.domain.ShippingState;
import de.itemis.qimpress.showcase.shipment_simulator.be.exceptions.DaoException;
import de.itemis.qimpress.showcase.shipment_simulator.be.generator.dao.OrderShippingDao;
import de.itemis.qimpress.showcase.shipment_simulator.be.generator.dao.ShipmentDao;

/**
 * @author Claudius Haecker
 * Tests implementaions of OrderReportDao<br/>
 * 
 * After running this test execute the following SQL commands, 
 * otherwise test data will remain in the database:<br/>
 * DELETE FROM item_shipping WHERE product_code like 'JUnit-test-PROD0%';
 * DELETE FROM shipment_item WHERE shipped_product_code like 'JUnit-test-PROD00%';
 * DELETE FROM shipment WHERE tracking_code like 'JUnit-test-TRACK%';
 * DELETE FROM order_shipping WHERE order_number like 'JUnit-test-ORDER%';
 */
public class OrderReportDaoTest {
    private static final Logger LOG = Logger.getLogger(OrderReportDaoTest.class);
    private static OrderReportDao ORDER_REPORT_DAO;
    private static OrderShippingDao ORDER_SHIPPING_DAO;
    private static ShipmentDao SHIPMENT_DAO;
    private OrderShipping savedOrderShipping;
    private Random random = new Random();
    private String trackingCode = "JUnit-test-TRACK0001" + getRandomString();
    private String orderNumber = "JUnit-test-ORDER" + getRandomString();
    private String customerNumber = "JUnit-test-CUSTOMER" + getRandomString();

    public String getRandomString() {
        return Long.toString(random.nextLong(), 36).toUpperCase();
    }

    /**
     * @throws java.lang.Exception is handled by JUnit
     */
    @Before
    public void setUp() throws Exception {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">> setUpBeforeClass()");
        }

        ApplicationContext ctxt = new ClassPathXmlApplicationContext("applicationContext.xml");
        assertNotNull("Failed to load context", ctxt);
        try {
            ORDER_REPORT_DAO = (OrderReportDao) ctxt.getBean("orderReportDao");
        } catch (BeansException e) {
            fail("Failed to load OrderReportDao from the application context");
        }

        try {
            ORDER_SHIPPING_DAO = (OrderShippingDao) ctxt.getBean("orderShippingDao");
        } catch (BeansException e) {
            fail("Failed to load OrderShippingDao from the application context");
        }

        try {
            SHIPMENT_DAO = (ShipmentDao) ctxt.getBean("shipmentDao");
        } catch (BeansException e) {
            fail("Failed to load ShipmentDao from the application context");
        }
        savedOrderShipping = setupForOrderShipping();

        setupForShipments(savedOrderShipping);
    }

    /**
     * 
     */
    private void setupForShipments(OrderShipping anOrderShipping) {
        Shipment newShipment = new Shipment();
        newShipment.setDispatchDate(new Date());
        newShipment.setTrackingCode(trackingCode);
        newShipment.setOrderShipping(anOrderShipping);

        List<ShipmentItem> shipmentItems = new LinkedList<ShipmentItem>();
        ShipmentItem shipmentItem = new ShipmentItem();
        shipmentItem.setShippedProductCode("JUnit-test-PROD001" + getRandomString());
        shipmentItem.setShippedQuantity(4);
        shipmentItems.add(shipmentItem);
        shipmentItem = new ShipmentItem();
        shipmentItem.setShippedProductCode("JUnit-test-PROD002" + getRandomString());
        shipmentItem.setShippedQuantity(3);
        shipmentItems.add(shipmentItem);

        newShipment.setShipmentItems(shipmentItems);

        try {
            SHIPMENT_DAO.save(newShipment);
        } catch (DaoException e) {
            e.printStackTrace();
            fail("Failed to store Shipment");
        }
    }

    /**
     * 
     */
    private OrderShipping setupForOrderShipping() {
        OrderShipping orderShipping = new OrderShipping();
        orderShipping.setOrderNumber(orderNumber);
        orderShipping.setCustomerNumber(customerNumber);
        orderShipping.setNextProcessingDate(new Date());
        orderShipping.setShippingState(new ShippingState(1L));

        List<ItemShipping> itemShippings = new LinkedList<ItemShipping>();
        ItemShipping itemShipping = new ItemShipping();
        itemShipping.setProductCode("JUnit-test-PROD01" + getRandomString());
        itemShipping.setLeftQuantity(5);
        itemShippings.add(itemShipping);
        itemShipping = new ItemShipping();
        itemShipping.setProductCode("JUnit-test-PROD02" + getRandomString());
        itemShipping.setLeftQuantity(10);
        itemShippings.add(itemShipping);
        orderShipping.setItemShippings(itemShippings);

        List<OrderShipping> orderShippings = new LinkedList<OrderShipping>();
        orderShippings.add(orderShipping);

        try {
            ORDER_SHIPPING_DAO.save(orderShippings);
        } catch (DaoException e) {
            e.printStackTrace();
            fail("Failed to save list of OrderShippings");
        }
        return orderShipping;
    }

    /**
     * Test method for
     * {@link de.itemis.qimpress.showcase.shipment_simulator.be.dao.OrderReportDao#getShipments(de.itemis.qimpress.showcase.shipment_simulator.be.domain.OrderShipping)}.
     */
    @Transactional
    @Test
    public void testGetShipments() {
        List<Shipment> shipments1 = null;
        try {
            shipments1 = ORDER_REPORT_DAO.getShipments(savedOrderShipping);
        } catch (DaoException e) {
            e.printStackTrace();
            fail();
        }
        assertEquals(trackingCode, shipments1.get(0).getTrackingCode());
    }

    /**
     * Test method for {@link de.itemis.qimpress.showcase.shipment_simulator.be.dao.OrderReportDao#getOrderShipping(java.lang.String)}.
     * saves an <code>OrderShipping</code> with two <code>ItemShipping</code>s and ensures that no <code>DaoException</code> is thrown.<br/>
     * Then tries to read that <code>OrderShipping</code>.<br/>
     */
    @Test
    public void testGetOrderShipping() {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">> testGetOrderShipping");
        }

        OrderShipping retrievedOrderShipping = null;
        try {
            retrievedOrderShipping = ORDER_REPORT_DAO.getOrderShipping(orderNumber);
        } catch (DaoException e) {
            e.printStackTrace();
            fail();
        }
        assertEquals(customerNumber, retrievedOrderShipping.getCustomerNumber());
    }
}
