/*******************************************************************************
 * Copyright (c) 2008,2009 Q-ImPrESS consortium
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * This work was funded in the context of the Q-ImPrESS research project  
 * (FP7-215013) by the European Union under the Information and  
 * Communication Technologies priority of the Seventh Research Framework  
 * Programme.
 *
 * Contributors:
 *     itemis 	- initial API and implementation
 *******************************************************************************/
package de.itemis.qimpress.showcase.shipment_simulator.be.remote;

import java.util.Properties;

import org.apache.log4j.Logger;
import org.jdom.Element;
import org.jdom.Namespace;
import org.jdom.transform.JDOMResult;
import org.jdom.transform.JDOMSource;
import org.springframework.ws.client.core.support.WebServiceGatewaySupport;

import de.itemis.qimpress.showcase.shipment_simulator.be.exceptions.ApplicationException;
import de.itemis.qimpress.showcase.shipment_simulator.be.exceptions.ApplicationException.ApplicationErrorCode;

/**
 * Implements {@link de.itemis.qimpress.showcase.shipment_simulator.be.remote.ServiceManager}
 * relying on Spring WS support for WebService clients.
 * 
 * @author Wladimir Safonov
 *
 */
public class ServiceManagerImpl extends WebServiceGatewaySupport implements ServiceManager {

    private static final Logger LOG = Logger.getLogger(ServiceManagerImpl.class);
    
    private static final String CHECK_PRODUCT_AVAILABILITY_REQUEST = "CheckProductAvailabilityRequest";
    private static final String CHECK_PRODUCT_AVAILABILITY_RESPONSE = "CheckProductAvailabilityResponse";
    
    private static final String IS_WS_URI = "is_ws_uri";
    
    private Namespace inventoryNamespace;
    
    private Properties webServiceURIs;
    
    
    public ServiceManagerImpl() {
        webServiceURIs = new Properties();
        try {
            webServiceURIs.load(this.getClass().getClassLoader().getResourceAsStream("webservice.properties"));
        } catch (Exception e) {
            LOG.error("Failed to load webservice.properties file", e);
        }
        inventoryNamespace = Namespace.getNamespace("is", "http://www.itemis.com/inventory_simulator/schemas");
    }

    /* @Override */
    public int checkProductAvailability(String productCode, int quantity) throws ApplicationException {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">> checkProductAvailability()");
        }
        
        // create request source
        Element requestElem = new Element(CHECK_PRODUCT_AVAILABILITY_REQUEST, inventoryNamespace);
        requestElem.addContent(new Element("ProductCode", inventoryNamespace).setText(productCode));
        requestElem.addContent(new Element("Quantity", inventoryNamespace).setText(Integer.toString(quantity)));
        JDOMSource requestSource = new JDOMSource(requestElem);
        
        // create response result
        JDOMResult responseResult = new JDOMResult();
        
        // invoke webservice
        getWebServiceTemplate().sendSourceAndReceiveToResult(
                webServiceURIs.getProperty(IS_WS_URI), 
                requestSource, 
                responseResult);
        
        // parse result
        Element responseElement = responseResult.getDocument().getRootElement();
        if (!responseElement.getNamespaceURI().equals(inventoryNamespace.getURI())) {
            LOG.error("Received response element with wrong namespace! Expected: '" + inventoryNamespace.getURI() + 
                    "', was: '" + responseElement.getNamespaceURI() + "'");
            throw new ApplicationException(ApplicationErrorCode.TE);
        }
        Element availableQuantityElement = responseElement.getChild("AvailableQuantity", inventoryNamespace);
        if (null == availableQuantityElement) {
            LOG.error("Received response element with no AvailableQuantity child element!");
            throw new ApplicationException(ApplicationErrorCode.TE);
        }
        int availableQuantity = Integer.parseInt(availableQuantityElement.getTextTrim());
        
        return availableQuantity;
    }
    
}
