/*******************************************************************************
 * Copyright (c) 2008,2009 Q-ImPrESS consortium
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * This work was funded in the context of the Q-ImPrESS research project  
 * (FP7-215013) by the European Union under the Information and  
 * Communication Technologies priority of the Seventh Research Framework  
 * Programme.
 *
 * Contributors:
 *     itemis 	- initial API and implementation
 *******************************************************************************/
package de.itemis.qimpress.showcase.shipment_simulator.be.manager.dao;

import java.sql.ResultSet;
import java.sql.SQLException;

import org.apache.log4j.Logger;
import org.springframework.dao.DataAccessException;
import org.springframework.jdbc.core.ResultSetExtractor;
import org.springframework.jdbc.core.support.JdbcDaoSupport;

import de.itemis.qimpress.showcase.shipment_simulator.be.exceptions.DaoException;
import de.itemis.qimpress.showcase.shipment_simulator.be.manager.model.GlobalParameters;

/**
 * @author Wladimir Safonov
 * @author Claudius Haecker
 *
 */
public class GlobalParametersDaoImpl extends JdbcDaoSupport implements GlobalParametersDao {

    private static final Logger LOG = Logger.getLogger(GlobalParametersDaoImpl.class);

    private static final String MIN_SHIPMENT_DURATION = "MIN_SHIPMENT_DURATION";
    private static final String MAX_SHIPMENT_DURATION = "MAX_SHIPMENT_DURATION";

    private static final String SELECT_PARAMETERS = "SELECT * FROM global_parameters";
    private static final String INSERT_PARAMETERS = "INSERT INTO global_parameters(parameter_name, parameter_value) VALUES(?, ?)";
    private static final String UPDATE_PARAMETERS = "UPDATE global_parameters SET parameter_value=? WHERE parameter_name=?";

    private ResultSetExtractor parametersExtractor;

    /* (non-Javadoc)
     * @see de.itemis.qimpress.showcase.shipment_simulator.be.manager.dao.GlobalParametersDao#getAll()
     */
    public GlobalParametersDaoImpl() {
        super();

        this.parametersExtractor = new ResultSetExtractor() {

            /* @Override */
            public Object extractData(ResultSet rs) throws SQLException, DataAccessException {

                GlobalParameters parameters = new GlobalParameters();
                while (rs.next()) {
                    String paramName = rs.getString("parameter_name");
                    String paramValue = rs.getString("parameter_value");

                    if (MIN_SHIPMENT_DURATION.equals(paramName)) {
                        parameters.setMinShipmentInterval(Integer.parseInt(paramValue));
                    } else if (MAX_SHIPMENT_DURATION.equals(paramName)) {
                        parameters.setMaxShipmentInterval(Integer.parseInt(paramValue));
                    }
                }
                return parameters;
            }
        };
    }

    /* @Override */
    public GlobalParameters load() throws DaoException {
        if (LOG.isDebugEnabled()) {
            LOG.debug("called load()");
        }

        GlobalParameters parameters;
        try {
            parameters = (GlobalParameters) getJdbcTemplate().query(SELECT_PARAMETERS, parametersExtractor);
        } catch (DataAccessException e) {
            LOG.error("Failed to load GlobalParameters", e);
            throw new DaoException(e);
        }
        return parameters;
    }

    /* @Override */
    public void store(GlobalParameters globalParameters) throws DaoException {
        if (LOG.isDebugEnabled()) {
            LOG.debug("called store()");
        }

        // try to update parameter first, if failed then insert
        // MIN_SHIPMENT_DURATION
        int insertedRows = 0;
        try {
            insertedRows = getJdbcTemplate().update(UPDATE_PARAMETERS,
                    new Object[] { globalParameters.getMinShipmentInterval(), MIN_SHIPMENT_DURATION });
            if (insertedRows == 0) {
                LOG.info("Existing GlobalParameters." + MIN_SHIPMENT_DURATION
                        + " is not found in the database. Inserting instead...");
                getJdbcTemplate().update(INSERT_PARAMETERS,
                        new Object[] { MIN_SHIPMENT_DURATION, globalParameters.getMinShipmentInterval() });
            }
        } catch (DataAccessException e) {
            LOG.error("Failed to store GlobalParameters." + MIN_SHIPMENT_DURATION, e);
            throw new DaoException(e);
        }

        // MAX_SHIPMENT_DURATION
        try {
            insertedRows = getJdbcTemplate().update(UPDATE_PARAMETERS,
                    new Object[] { globalParameters.getMaxShipmentInterval(), MAX_SHIPMENT_DURATION });
            if (insertedRows == 0) {
                LOG.info("Existing GlobalParameters." + MAX_SHIPMENT_DURATION
                        + " is not found in the database. Inserting instead...");
                getJdbcTemplate().update(INSERT_PARAMETERS,
                        new Object[] { MAX_SHIPMENT_DURATION, globalParameters.getMaxShipmentInterval() });
            }
        } catch (DataAccessException e) {
            LOG.error("Failed to store GlobalParameters." + MAX_SHIPMENT_DURATION, e);
            throw new DaoException(e);
        }

    }

}
