/*******************************************************************************
 * Copyright (c) 2008,2009 Q-ImPrESS consortium
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * This work was funded in the context of the Q-ImPrESS research project  
 * (FP7-215013) by the European Union under the Information and  
 * Communication Technologies priority of the Seventh Research Framework  
 * Programme.
 *
 * Contributors:
 *     itemis 	- initial API and implementation
 *******************************************************************************/
package de.itemis.qimpress.showcase.shipment_simulator.be.generator.dao;

import java.util.Date;
import java.util.List;

import de.itemis.qimpress.showcase.shipment_simulator.be.domain.OrderShipping;
import de.itemis.qimpress.showcase.shipment_simulator.be.domain.ShippingState;
import de.itemis.qimpress.showcase.shipment_simulator.be.exceptions.DaoException;

/**
 * DAO-operations interface for loading and storing
 * {@link de.itemis.qimpress.showcase.shipment_simulator.be.domain.OrderShipping OrderShippings}
 * objects.
 * 
 * @author Wladimir Safonov
 *
 */
public interface OrderShippingDao {

    /**
     * Loads a list of all
     * {@link de.itemis.qimpress.showcase.shipment_simulator.be.domain.OrderShipping OrderShippings}
     * with <code>OPEN</code> or <code>PARTIALLY_DELIVERED</code> shipping states, whose
     * {@link OrderShipping#getNextProcessingDate() next processing date} is earlier than
     * the given date.
     * 
     * @param untilProcessingDate upper limit for the 
     * {@link OrderShipping#getNextProcessingDate() next processing date}
     * @return list of
     * {@link de.itemis.qimpress.showcase.shipment_simulator.be.domain.OrderShipping OrderShippings}
     * with <code>OPEN</code> or <code>PARTIALLY_DELIVERED</code> shipping states
     * @throws DaoException in case of a database failure
     */
    List<OrderShipping> getPending(Date untilProcessingDate) throws DaoException;
    
    /**
     * Stores
     * {@link de.itemis.qimpress.showcase.shipment_simulator.be.domain.OrderShipping OrderShippings}
     * from the given list in the database. OrderShippings without a given
     * {@link de.itemis.qimpress.showcase.shipment_simulator.be.domain.OrderShipping#getId() ID}
     * will be interpreted as new (transient) and inserted to the database. Generated IDs will be
     * set after saving.
     * 
     * @param orderShippings list of 
     * {@link de.itemis.qimpress.showcase.shipment_simulator.be.domain.OrderShipping OrderShippings}
     * to store in the database
     * @throws DaoException in case of a database failure
     */
    void save(List<OrderShipping> orderShippings) throws DaoException;
    
    /**
     * Loads {@link ShippingState} for the given {@link ShippingState#getStateName() state name}.
     * Allows to use the returned instance while creating new (transient) 
     * {@link OrderShipping OrderShippings} in order to set a reference to a valid persistent
     * {@link ShippingState}.
     * 
     * @param stateName state name of the {@link ShippingState}
     * @return {@link ShippingState} for the given state name
     * @throws DaoException in case of database failure
     */
    ShippingState getShippingState(String stateName) throws DaoException;
}
