/*******************************************************************************
 * Copyright (c) 2008,2009 Q-ImPrESS consortium
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * This work was funded in the context of the Q-ImPrESS research project  
 * (FP7-215013) by the European Union under the Information and  
 * Communication Technologies priority of the Seventh Research Framework  
 * Programme.
 *
 * Contributors:
 *     itemis 	- initial API and implementation
 *******************************************************************************/
package de.itemis.qimpress.showcase.shipment_simulator.be.generator;

import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.List;
import java.util.Random;

import org.apache.log4j.Logger;

import de.itemis.qimpress.showcase.shipment_simulator.be.domain.ItemShipping;
import de.itemis.qimpress.showcase.shipment_simulator.be.domain.OrderShipping;
import de.itemis.qimpress.showcase.shipment_simulator.be.domain.Shipment;
import de.itemis.qimpress.showcase.shipment_simulator.be.domain.ShipmentItem;
import de.itemis.qimpress.showcase.shipment_simulator.be.domain.ShippingState;
import de.itemis.qimpress.showcase.shipment_simulator.be.exceptions.ApplicationException;
import de.itemis.qimpress.showcase.shipment_simulator.be.exceptions.DaoException;
import de.itemis.qimpress.showcase.shipment_simulator.be.generator.dao.OrderShippingDao;
import de.itemis.qimpress.showcase.shipment_simulator.be.generator.dao.ShipmentDao;
import de.itemis.qimpress.showcase.shipment_simulator.be.manager.model.GlobalParameters;
import de.itemis.qimpress.showcase.shipment_simulator.be.manager.model.TimeParameters;
import de.itemis.qimpress.showcase.shipment_simulator.be.remote.ServiceManager;

/**
 * Shipment generation job. 
 * @author Claudius Haecker
 */
public class ShipmentGeneratorJob {

    private static final Logger LOG = Logger.getLogger(ShipmentGeneratorJob.class);

    /** service manager for retrieving data from Order Simulator via JMS */
    //    private ServiceManager serviceManager;
    /** list of order processors to receive the generated order */
    //    private List<OrderProcessor> orderProcessors;
    /** service manager for accessing Inventory web service */
    private ServiceManager serviceManager;
    /** shipping information about the order. */
    private OrderShippingDao orderShippingDao;
    // random number generator
    private Random random = new Random();

    private Long count = 0L;

    private ShipmentDao shipmentDao;

    /**
     * @param serviceManager the serviceManager to set
     */
    //    public void setServiceManager(ServiceManager serviceManager) {
    //        this.serviceManager = serviceManager;
    //    }
    /**
     * @param orderProcessors the orderProcessors to set
     */
    //    public void setOrderProcessors(List<OrderProcessor> orderProcessors) {
    //        this.orderProcessors = orderProcessors;
    //    }
    /**
     * Generates a shipment and dispatches it to the shipment handler.
     * 
     * @param params parameters for this generator run
     * @throws DaoException If accessing the database produces an error
     */
    public void execute(GlobalParameters params) throws DaoException {
        // get the current time
        // take all orders from the database table of open orders that are
        //    * not completely delivered and 
        //    * whose next delivery time is due
        //    for each of those orders generate a shipment:
        //        if this shipment is open, it doesn't have a nextProcessingDate. So we use the current time.
        //        start a new shipment
        //        generate shipment number
        //        for each order item of this order that are not completely delivered yet:
        //            ask the inventory simulator how many pieces of this order item can be delivered
        //            add those pieces to the shipment
        //            if all pieces of this order item are delivered now
        //                mark the order item as delivered
        //            else
        //                diminish the number of delivered pieces for this order item
        //                determine time of next order generation
        //        if any pieces have been added to the shipment 
        //            add the shipment to the list of shipments
        //            if all order items of this order are delivered now
        //                mark this order as completely delivered
        //            if the order is completely delivered now, mark it as closed
        // execute the list of shipments
        if (LOG.isDebugEnabled()) {
            LOG.debug(">> execute()");
        }
        // get the current time
        Date nowVirtual = calculateCurrentSimulationTime(params);

        // take all orders from the database table of open orders that are
        //    * not completely delivered and 
        //    * whose next delivery time is due
        List<OrderShipping> oldPendingOrders;
        List<OrderShipping> newPendingOrders = new ArrayList<OrderShipping>();
        try {
            oldPendingOrders = orderShippingDao.getPending(nowVirtual);
        } catch (DaoException de) {
            LOG.error("Could not get pending orders", de);
            oldPendingOrders = new ArrayList<OrderShipping>();
        }
        //    for each of those orders generate a shipment:
        for (OrderShipping orderShipping : oldPendingOrders) {
            // if this shipment is open, it doesn't have a nextProcessingDate. So we use the current time.
            if (orderShipping.getShippingState().equals(orderShippingDao.getShippingState(ShippingState.OPEN))) {
                orderShipping.setNextProcessingDate(calculateCurrentSimulationTime(params));
            }
            //        start a new shipment
            Shipment shipment = new Shipment();
            shipment.setShipmentItems(new ArrayList<ShipmentItem>());
            shipment.setDispatchDate(orderShipping.getNextProcessingDate());
            //        for each order item of this order that are not completely delivered yet:
            boolean completelyDelivered = true;
            for (ItemShipping itemShipping : orderShipping.getItemShippings()) {
                int leftQuantity = itemShipping.getLeftQuantity();
                if (leftQuantity > 0) {
                    //            ask the inventory simulator how many pieces of this order item can be delivered
                    int shippable;
                    try {
                        shippable = serviceManager
                                .checkProductAvailability(itemShipping.getProductCode(), leftQuantity);
                    } catch (ApplicationException ae) {
                        LOG.error("Could not check product availability", ae);
                        shippable = 0;
                    }
                    //            add those pieces to the shipment
                    if (shippable > 0) {
                        ShipmentItem shipmentItem = new ShipmentItem();
                        shipmentItem.setShippedProductCode(itemShipping.getProductCode());
                        shipmentItem.setShippedQuantity(shippable);
                        shipment.getShipmentItems().add(shipmentItem);
                        //                mark the order item as PARTIALLY delivered
                        orderShipping.setShippingState(orderShippingDao
                                .getShippingState(ShippingState.PARTIALLY_DELIVERED));
                        newPendingOrders.add(orderShipping);
                        itemShipping.setLeftQuantity(leftQuantity - shippable);
                    }
                    completelyDelivered = completelyDelivered && (itemShipping.getLeftQuantity() == 0);
                }
            }
            //            create a tracking code
            shipment.setTrackingCode(generateTrackingCode());
            orderShipping.setNextProcessingDate(calculateNextProcessingDate(params, shipment.getDispatchDate()));
            if (completelyDelivered) {
                //                diminish the number of delivered pieces for this order item
                //            if all order items of this order are delivered now
                //                mark this order as completely delivered
                orderShipping.setShippingState(orderShippingDao.getShippingState(ShippingState.COMPLETELY_DELIVERED));
            }
            shipment.setOrderShipping(orderShipping);
            if (shipment.getShipmentItems().size() > 0) {
                shipmentDao.save(shipment);
            }
            //                determine time of next order generation
            orderShippingDao.save(newPendingOrders);
            //            if the order is completely delivered now, mark it as closed
            // execute the list of shipments
        }
    }

    /**
     * @param params parameters for this generator run
     * @param oldDispatchDate
     * @return the nextProcessingDate in virtual time
     */
    private Date calculateNextProcessingDate(GlobalParameters params, Date oldDispatchDate) {
        int differenceInDays = params.getMinShipmentInterval()
                + random.nextInt(params.getMaxShipmentInterval() - params.getMinShipmentInterval());
        GregorianCalendar nextDate = new GregorianCalendar();
        nextDate.setTime(oldDispatchDate);
        nextDate.add(Calendar.DAY_OF_MONTH, differenceInDays);
        return nextDate.getTime();
    }

    /**
     * @param params parameters for this generator run
     * @return the current virtual time if generatorTimeMode is SIMULATED_TIME, otherwise the current real time
     */
    static Date calculateCurrentSimulationTime(GlobalParameters params) {
        if (params.getTimeParameters().getGeneratorTimeMode() == TimeParameters.GeneratorTimeMode.REAL_TIME) {
            return new Date();
        }
        long realElapsedMilliSecsSinceGeneratorStart = (new Date()).getTime()
                - params.getTimeParameters().getStartTimeOfGenerator().getTime();
        Date nowVirtual = new Date(params.getTimeParameters().getStartTimeForOrders().getTime()
                + (long) (params.getTimeParameters().getTimeScaleFactor() * realElapsedMilliSecsSinceGeneratorStart));
        return nowVirtual;
    }

    String generateTrackingCode() {
        return new DecimalFormat("000000000").format(count++) + "_"
                + Long.toString(random.nextLong(), 36).toUpperCase();
    }

    /**
     * @param orderShippingDao the orderShippingDao to set
     */
    public void setOrderShippingDao(OrderShippingDao orderShippingDao) {
        this.orderShippingDao = orderShippingDao;
    }

    /**
     * @param serviceManager the serviceManager to set
     */
    public void setServiceManager(ServiceManager serviceManager) {
        this.serviceManager = serviceManager;
    }

    /**
     * @param shipmentDao the shipmentDao to set
     */
    public void setShipmentDao(ShipmentDao shipmentDao) {
        this.shipmentDao = shipmentDao;
    }

    /*
        private Order generateOrder(Customer customer, CustomerTypeParameters customerTypeParams)
                throws ApplicationException {

            // create order
            String orderNumber = orderNumberGenerator.generateNextOrderNumber();
            Order generatedOrder = new Order(orderNumber, new Date(), customer);

            // generate order items number
            int orderItemsNum = random.nextInt(customerTypeParams.getMaxItemsPerOrder()
                    - customerTypeParams.getMinItemsPerOrder() + 1)
                    + customerTypeParams.getMinItemsPerOrder();

            // fill order with items
            List<OrderItem> orderItems = new LinkedList<OrderItem>();
            for (int i = 0; i < orderItemsNum; i++) {

                // generate product quantity for this item
                int quantity = random.nextInt(customerTypeParams.getMaxQuantityPerItem()
                        - customerTypeParams.getMinQuantityPerItem() + 1)
                        + customerTypeParams.getMinQuantityPerItem();

                // load random product
                String productCode = serviceManager.getRandomProductCode();
                ProductInfo product = serviceManager.getProductInfoByCode(productCode);

                // create order item
                OrderItem orderItem = new OrderItem(quantity, product);
                orderItems.add(orderItem);

            }
            generatedOrder.setOrderItems(orderItems);

            // price order
            generatedOrder = serviceManager.priceOrder(generatedOrder);
            return generatedOrder;
        }
    */
}