/*******************************************************************************
 * Copyright (c) 2008,2009 Q-ImPrESS consortium
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * This work was funded in the context of the Q-ImPrESS research project  
 * (FP7-215013) by the European Union under the Information and  
 * Communication Technologies priority of the Seventh Research Framework  
 * Programme.
 *
 * Contributors:
 *     itemis 	- initial API and implementation
 *******************************************************************************/
package de.itemis.qimpress.showcase.shipment_simulator.be.generator;

import java.io.StringReader;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import javax.jms.Message;
import javax.jms.MessageListener;
import javax.jms.TextMessage;
import javax.xml.transform.stream.StreamSource;

import org.apache.log4j.Logger;
import org.springframework.oxm.Unmarshaller;

import de.itemis.qimpress.showcase.shipment_simulator.be.domain.ItemShipping;
import de.itemis.qimpress.showcase.shipment_simulator.be.domain.OrderShipping;
import de.itemis.qimpress.showcase.shipment_simulator.be.domain.ShippingState;
import de.itemis.qimpress.showcase.shipment_simulator.be.dto.Order;
import de.itemis.qimpress.showcase.shipment_simulator.be.dto.OrderItem;
import de.itemis.qimpress.showcase.shipment_simulator.be.dto.OrdersBatch;
import de.itemis.qimpress.showcase.shipment_simulator.be.exceptions.DaoException;
import de.itemis.qimpress.showcase.shipment_simulator.be.generator.dao.OrderShippingDao;

/**
 * JMS message listener for processing incoming orders.
 * Each {@link Order} from the message is used to add a new 
 * {@link OrderShipping} to the database for this order with 
 * the <code>OPEN</code> {@link OrderShipping#getShippingState() shipping state}.
 * 
 * @author Wladimir Safonov
 *
 */
public class IncomingOrdersMessageListener implements MessageListener {

    private static final Logger LOG = Logger.getLogger(IncomingOrdersMessageListener.class);
    
    private Unmarshaller unmarshaller;
    private OrderShippingDao orderShippingDao;
    
    /**
     * @param unmarshaller the unmarshaller to set
     */
    public void setUnmarshaller(Unmarshaller unmarshaller) {
        this.unmarshaller = unmarshaller;
    }

    /**
     * @param orderShippingDao the orderShippingDao to set
     */
    public void setOrderShippingDao(OrderShippingDao orderShippingDao) {
        this.orderShippingDao = orderShippingDao;
    }

    /* (non-Javadoc)
     * @see javax.jms.MessageListener#onMessage(javax.jms.Message)
     */
    /* @Override */
    public void onMessage(Message message) {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">> onMessage()");
        }
        
        // unmarshall message to OrdersBatch
        OrdersBatch ordersBatch = null;
        TextMessage textMessage = (TextMessage) message;
        try {
            StreamSource streamSource = new StreamSource(new StringReader(textMessage.getText()));
            ordersBatch = (OrdersBatch) unmarshaller.unmarshal(streamSource);
        } catch (Exception e) {
            LOG.error("Failed to read incoming order message", e);
            // fail silence
            return;
        }
        
        // create and persist OrderShippings for incoming orders
        List<OrderShipping> orderShippings = new LinkedList<OrderShipping>();
        for (Order order : ordersBatch.getOrders()) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Received order: " + order.getNumber());
            }
            
            // create OrderShipping from Order
            OrderShipping orderShipping = new OrderShipping();
            orderShipping.setOrderNumber(order.getNumber());
            orderShipping.setCustomerNumber(order.getCustomerNumber());
            try {
                orderShipping.setShippingState(orderShippingDao.getShippingState(ShippingState.OPEN));
            } catch (DaoException e) {
                LOG.error("Failed to load ShippingState by name", e);
                LOG.error("Order '" + order.getNumber() + "' could not be processed");
                continue;
            }
            orderShipping.setNextProcessingDate(order.getCreationDate());
            
            // create ItemShipping for each OrderItem
            List<ItemShipping> itemShippings = new LinkedList<ItemShipping>();
            for (OrderItem orderItem : order.getOrderItems()) {
                itemShippings.add(new ItemShipping(
                        orderItem.getProductCode(), orderItem.getQuantity()));
            }
            orderShipping.setItemShippings(itemShippings);
            
            orderShippings.add(orderShipping);
        }
        // save created OrderShipping
        try {
            orderShippingDao.save(orderShippings);
        } catch (DaoException e) {
            LOG.error("Failed to save OrderShipping list", e);
            // fail silence
            return;
        }
    }

}
