/*******************************************************************************
 * Copyright (c) 2008,2009 Q-ImPrESS consortium
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * This work was funded in the context of the Q-ImPrESS research project  
 * (FP7-215013) by the European Union under the Information and  
 * Communication Technologies priority of the Seventh Research Framework  
 * Programme.
 *
 * Contributors:
 *     itemis 	- initial API and implementation
 *******************************************************************************/
package de.itemis.qimpress.showcase.shipment_simulator.be.generator;

import java.util.Date;

import org.apache.log4j.Logger;
import org.quartz.JobDetail;
import org.quartz.Scheduler;
import org.quartz.SchedulerException;
import org.quartz.SimpleTrigger;

import de.itemis.qimpress.showcase.shipment_simulator.be.manager.model.GlobalParameters;
import de.itemis.qimpress.showcase.shipment_simulator.be.manager.model.TimeParameters;
import de.itemis.qimpress.showcase.shipment_simulator.be.manager.model.TimeParameters.GeneratorTimeMode;

/**
 * @author Wladimir Safonov
 *
 */
public class GeneratorManagerImpl implements GeneratorManager {

    private static final Logger LOG = Logger.getLogger(GeneratorManagerImpl.class);

    //    this value can be changed for test purpose to achieve a shorter scheduler interval 
    //    for shipments processing (default: one day in simulation time)
    private static final long DAY_MILLIS = 24 * 60 * 60 * 1000;

    /** SchedulerFactoryBean configured with the ShipmentGeneratorJob */
    private Scheduler scheduler;
    /** SimpleTriggerBean configured with a time interval for the ShipmentGeneratorJob */
    private SimpleTrigger trigger;
    /** JobDetail to access and update JobDataMap with generation parameters */
    private JobDetail jobDetail;

    /**
     * @param scheduler the scheduler to set
     */
    public void setScheduler(Scheduler scheduler) {
        this.scheduler = scheduler;
    }

    /**
     * @param trigger the trigger to set
     */
    public void setTrigger(SimpleTrigger trigger) {
        this.trigger = trigger;
    }

    /**
     * @param jobDetail the jobDetail to set
     */
    public void setJobDetail(JobDetail jobDetail) {
        this.jobDetail = jobDetail;
    }

    /* (non-Javadoc)
     * @see de.itemis.qimpress.showcase.order_shipment_generator.be.generator.GeneratorManager#isOrderGeneratorRunning()
     */
    /* @Override */
    public boolean isShipmentGeneratorRunning() {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">> isShipmentGeneratorRunning()");
        }

        try {
            return !scheduler.isInStandbyMode();
        } catch (SchedulerException e) {
            LOG.error("Failed to get scheduler status", e);
            return false;
        }
    }

    /* (non-Javadoc)
     * @see de.itemis.qimpress.showcase.order_shipment_generator.be.generator.GeneratorManager#startOrderGenerator(de.itemis.qimpress.showcase.order_shipment_generator.be.configurator.model.GlobalParameters)
     */
    /* @Override */
    public boolean startShipmentGenerator(GlobalParameters parameters) {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">> startShipmentGenerator()");
        }

        if (isShipmentGeneratorRunning()) {
            return false;
        }

        // set default values of the time parameters for the real time mode
        TimeParameters timeParameters = parameters.getTimeParameters();
        if (timeParameters.getGeneratorTimeMode() == GeneratorTimeMode.REAL_TIME) {
            timeParameters.setTimeScaleFactor(1.0f);
            timeParameters.setStartTimeForOrders(new Date());
        }
        // set start time of the generator
        timeParameters.setStartTimeOfGenerator(new Date());

        try {
            // set trigger interval
            long repeatInterval = Math.round(DAY_MILLIS / parameters.getTimeParameters().getTimeScaleFactor());
            trigger.setRepeatInterval(repeatInterval);

            // save parameters in the JobDataMap & reschedule job
            jobDetail.getJobDataMap().put("arg", parameters);
            scheduler.addJob(jobDetail, true);

            // start scheduler
            scheduler.start();

        } catch (SchedulerException e) {
            LOG.error("Failed to start scheduler with an updated job", e);
            return false;
        }
        return true;
    }

    /* (non-Javadoc)
     * @see de.itemis.qimpress.showcase.order_shipment_generator.be.generator.GeneratorManager#stopOrderGenerator()
     */
    /* @Override */
    public boolean stopShipmentGenerator() {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">> stopShipmentGenerator()");
        }

        try {
            scheduler.standby();
        } catch (SchedulerException e) {
            LOG.error("Failed to pause scheduler");
            return false;
        }
        return true;
    }

}