/*******************************************************************************
 * Copyright (c) 2008,2009 Q-ImPrESS consortium
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * This work was funded in the context of the Q-ImPrESS research project  
 * (FP7-215013) by the European Union under the Information and  
 * Communication Technologies priority of the Seventh Research Framework  
 * Programme.
 *
 * Contributors:
 *     itemis 	- initial API and implementation
 *******************************************************************************/
package de.itemis.qimpress.showcase.pricing_simulator.webservice;

import static org.apache.commons.lang.Validate.isTrue;
import static org.apache.commons.lang.Validate.notEmpty;
import static org.apache.commons.lang.Validate.notNull;

import java.util.List;

import org.apache.log4j.Logger;
import org.springframework.ws.server.endpoint.annotation.Endpoint;
import org.springframework.ws.server.endpoint.annotation.PayloadRoot;

import de.itemis.qimpress.common.Meassurements;
import de.itemis.qimpress.showcase.pricing_simulator.be.domain.CustomerCategoryDiscount;
import de.itemis.qimpress.showcase.pricing_simulator.be.domain.ProductPrice;
import de.itemis.qimpress.showcase.pricing_simulator.be.dto.Order;
import de.itemis.qimpress.showcase.pricing_simulator.be.dto.Price;
import de.itemis.qimpress.showcase.pricing_simulator.be.exceptions.ApplicationException;
import de.itemis.qimpress.showcase.pricing_simulator.be.service.PricingManager;
import de.itemis.qimpress.showcase.pricing_simulator.be.service.ServiceLocator;
import de.itemis.qimpress.showcase.pricing_simulator.webservice.messages.GetCustomerCategoryDiscountsRequest;
import de.itemis.qimpress.showcase.pricing_simulator.webservice.messages.GetCustomerCategoryDiscountsResponse;
import de.itemis.qimpress.showcase.pricing_simulator.webservice.messages.GetListPriceForProductRequest;
import de.itemis.qimpress.showcase.pricing_simulator.webservice.messages.GetListPriceForProductResponse;
import de.itemis.qimpress.showcase.pricing_simulator.webservice.messages.GetPriceForProductRequest;
import de.itemis.qimpress.showcase.pricing_simulator.webservice.messages.GetPriceForProductResponse;
import de.itemis.qimpress.showcase.pricing_simulator.webservice.messages.PriceOrderRequest;
import de.itemis.qimpress.showcase.pricing_simulator.webservice.messages.PriceOrderResponse;

/**
 * Endpoint for Pricing Webservice with XML auto-marshalling.
 * 
 * @author Wladimir Safonov
 *
 */
@Endpoint
public class PricingMarshallingEndpoint {

    private static final Logger LOG = Logger.getLogger(PricingMarshallingEndpoint.class);

    private PricingManager pricingManager = ServiceLocator.getInstance().getPricingManager();

    /**
     * @param pricingManager the pricingManager to set
     */
    public void setPricingManager(PricingManager pricingManager) {
        this.pricingManager = pricingManager;
    }

    public PricingMarshallingEndpoint() {
        
    }

    @PayloadRoot(localPart = "GetListPriceForProductRequest", namespace = "http://www.itemis.com/pricing/schemas")
    public GetListPriceForProductResponse getProductListPrice(GetListPriceForProductRequest request)
            throws ApplicationException {
    	
		long   start = 0;
		long   stop  = 0;

    	if (LOG.isTraceEnabled()) {
    		start = System.currentTimeMillis();
    	}
    	
        if (LOG.isDebugEnabled()) {
            LOG.debug("WS method GetListPriceForProduct called");
        }

        // handle error situations
        notNull(request, "Received null instead of request object for GetListPriceForProduct");
        notEmpty(request.getProductCode(), "ProductCode is null");
        // TODO: add error handling codes
        //        notEmpty(request.getCountryIsoCode(), "CountryIsoCode is null");

        // process webservice request
        GetListPriceForProductResponse response = new GetListPriceForProductResponse();
        ProductPrice productPrice = null;
        try {
            productPrice = pricingManager
                    .findProductCountryPrice(request.getProductCode(), request.getCountryIsoCode());
        } catch (ApplicationException e) {
            LOG.error("Failed to find ProductPrice over PricingManager", e);
            throw e;
        }

        if (productPrice != null) {
            Price price = new Price(productPrice.getPrice(), 
                    productPrice.getCurrency().getIsoCode(), 
                    productPrice.getCurrency().getSign());
            response.setProductPrice(price);
        }
        
        if (LOG.isTraceEnabled()) {
        	stop = System.currentTimeMillis();
        	Meassurements.add(this.getClass().getSimpleName(), "GetListPriceForProductRequest", start, stop);
        	Meassurements.log();
        }
        
        return response;
    }

    @PayloadRoot(localPart = "GetPriceForProductRequest", namespace = "http://www.itemis.com/pricing/schemas")
    public GetPriceForProductResponse getProductVolumePrice(GetPriceForProductRequest request)
            throws ApplicationException {
    	
		long   start = 0;
		long   stop  = 0;

    	if (LOG.isTraceEnabled()) {
    		start = System.currentTimeMillis();
    	}
    	
    	if (LOG.isDebugEnabled()) {
            LOG.debug("WS method GetPriceForProduct called");
        }

        // handle error situations
        notNull(request, "Received null instead of request object for GetPriceForProduct");
        notEmpty(request.getProductCode(), "ProductCode is null");
        // TODO: add error handling codes
        //        notEmpty(request.getCountryIsoCode(), "CountryIsoCode is null");
        isTrue(request.getProductAmount() > 0, "ProductAmount is null");
        //        notEmpty(request.getCustomerCategory(), "CustomerCategory is null");
        //        notEmpty(request.getCustomerType(), "CustomerCategory is null");

        // process webservice request
        GetPriceForProductResponse response = new GetPriceForProductResponse();
        Price price = pricingManager.calculateVolumePriceForCustomerCategory(request.getProductCode(), request
                .getCountryIsoCode(), request.getProductAmount(), request.getCustomerCategory(), request
                .getCustomerType());
        response.setProductPrice(price);

        if (LOG.isTraceEnabled()) {
        	stop = System.currentTimeMillis();
        	Meassurements.add(this.getClass().getSimpleName(), "GetPriceForProductRequest", start, stop);
        	Meassurements.log();
        }
        
        return response;
    }

    @PayloadRoot(localPart = "GetCustomerCategoryDiscountsRequest", namespace = "http://www.itemis.com/pricing/schemas")
    public GetCustomerCategoryDiscountsResponse getCustomerCategoryDiscounts(GetCustomerCategoryDiscountsRequest request)
            throws ApplicationException {
        
		long   start = 0;
		long   stop  = 0;

    	if (LOG.isTraceEnabled()) {
    		start = System.currentTimeMillis();
    	}
    	
    	if (LOG.isDebugEnabled()) {
            LOG.debug("WS method GetCustomerCategoryDiscounts called");
        }

        // no parameters expected -> process webservice request
        GetCustomerCategoryDiscountsResponse response = new GetCustomerCategoryDiscountsResponse();
        List<CustomerCategoryDiscount> discounts = pricingManager.getAllCustomerCategoryDiscounts();
        response.setCustomerCategoryDiscounts(discounts);

        if (LOG.isTraceEnabled()) {
        	stop = System.currentTimeMillis();
        	Meassurements.add(this.getClass().getSimpleName(), "GetCustomerCategoryDiscountsRequest", start, stop);
        	Meassurements.log();
        }
        
        return response;
    }
    
    @PayloadRoot(localPart = "PriceOrderRequest", namespace = "http://www.itemis.com/pricing/schemas")
    public PriceOrderResponse priceOrder(PriceOrderRequest request) throws ApplicationException {
       
		long   start = 0;
		long   stop  = 0;

    	if (LOG.isTraceEnabled()) {
    		start = System.currentTimeMillis();
    	}
    	
    	if (LOG.isDebugEnabled()) {
            LOG.debug("WS method PriceOrder called");
        }
        
        // handle error situations
        notNull(request, "Received null instead of request object for PriceOrderRequest");
        notNull(request.getOrder(), "Received null instead of Order");
        notNull(request.getOrder().getCustomer(), "Recevied null instead of Order.Customer parameter");
        notNull(request.getOrder().getCustomer().getInvoiceAddressCountry(), "Recevied null instead of Order.Customer.InvoiceCountry");
        
        // get Order parameter and process request
        Order pricedOrder = pricingManager.priceOrder(request.getOrder());
        
        // create response
        PriceOrderResponse response = new PriceOrderResponse();
        response.setPricedOrder(pricedOrder);
        
        if (LOG.isTraceEnabled()) {
        	stop = System.currentTimeMillis();
        	Meassurements.add(this.getClass().getSimpleName(), "PriceOrderRequest", start, stop);
        	Meassurements.log();
        }
        return response;
    }
}
