/*******************************************************************************
 * Copyright (c) 2008,2009 Q-ImPrESS consortium
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * This work was funded in the context of the Q-ImPrESS research project  
 * (FP7-215013) by the European Union under the Information and  
 * Communication Technologies priority of the Seventh Research Framework  
 * Programme.
 *
 * Contributors:
 *     itemis 	- initial API and implementation
 *******************************************************************************/
package de.itemis.qimpress.showcase.pricing_simulator.webservice;

import java.io.IOException;
import java.io.StringWriter;
import java.text.SimpleDateFormat;
import java.util.List;

import javax.naming.NamingException;

import org.apache.log4j.Logger;
import org.jdom.Element;
import org.jdom.Namespace;
import org.springframework.ws.server.endpoint.AbstractJDomPayloadEndpoint;

import com.csvreader.CsvWriter;

import de.itemis.qimpress.showcase.pricing_simulator.be.domain.ProductPrice;
import de.itemis.qimpress.showcase.pricing_simulator.be.service.PricingCsvManager;
import de.itemis.qimpress.showcase.pricing_simulator.be.service.ServiceLocator;

/**
 * Webservice endpoint for PricingCsvManager service interface 
 * using JDom for payload processing. Payload data is 
 * represented as CSV for bulk processing.
 * 
 * @author Wladimir Safonov
 *
 */
public class PricingCsvJDomEndpoint extends AbstractJDomPayloadEndpoint {

    private static final Logger LOG = Logger.getLogger(PricingCsvJDomEndpoint.class);
    
    public static final String GET_PRODUCT_PRICES_CSV_REQUEST = "GetProductPricesCSVRequest";
    public static final String GET_PRODUCT_PRICES_CSV_RESPONSE = "GetProductPricesCSVResponse";
    
    // PricingCsv service interface
    private PricingCsvManager pricingCsvManager = ServiceLocator.getInstance().getPricingCsvManager();
    
    private Namespace namespace;
    private final SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd hh:mm:ss");
    
    
    public PricingCsvJDomEndpoint() {
        
        namespace = Namespace.getNamespace("pricing", "http://www.itemis.com/pricing/schemas");
    }


    /* (non-Javadoc)
     * @see org.springframework.ws.server.endpoint.AbstractJDomPayloadEndpoint#invokeInternal(org.jdom.Element)
     */
    @Override
    protected Element invokeInternal(Element request) throws Exception {
        if (LOG.isDebugEnabled()) {
            LOG.debug("invokeInternal()");
        }
        
        String rootElement = request.getName();
        if (GET_PRODUCT_PRICES_CSV_REQUEST.equals(rootElement)) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Root element detected: " + GET_PRODUCT_PRICES_CSV_REQUEST);
            }
            
            // no parameters expected
            List<ProductPrice> productPrices = pricingCsvManager.getAllProductCountryPrices();
            if (LOG.isDebugEnabled()) {
                LOG.debug("Number of ProductPrices loaded: " + productPrices.size());
            }
            return createGetProductPricesCSVResponse(productPrices);
        }
        return null;
    }

    private Element createGetProductPricesCSVResponse(List<ProductPrice> productPrices) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("createGetProductPricesCSVResponse");
        }
        
        Element csvResponseRootElement = new Element(GET_PRODUCT_PRICES_CSV_RESPONSE, namespace);
        Element csvContentElement = new Element("ProductPricesCSV", namespace);
        csvResponseRootElement.setContent(csvContentElement);
        
        StringWriter csvContentWriter = new StringWriter();
        CsvWriter csvWriter = new CsvWriter(csvContentWriter, ';');
        
        try {
            csvWriter.writeRecord(new String[] { "Product_Price", "Product_Price_LC", "Country_ISO_Code",
                    "Product_Key" });
            // set text qualifier
            csvWriter.setTextQualifier('"');
            csvWriter.setForceQualifier(true);
            // iterate over the product prices list and add each price as a CSV line to response
            for (ProductPrice productPrice : productPrices) {
                csvWriter.write(productPrice.getPrice().toString());
                // no price in country currency yet supported
                csvWriter.write(productPrice.getCurrency().getIsoCode());
                csvWriter.write(productPrice.getCountryIsoCode());
                csvWriter.write(productPrice.getProductCode());
                csvWriter.endRecord();
            }
        } catch (IOException e) {
            LOG.error("Failed to write CSV content", e);
        }
        csvContentElement.setText(csvContentWriter.toString());
        return csvResponseRootElement;
    }
}
