/*******************************************************************************
 * Copyright (c) 2008,2009 Q-ImPrESS consortium
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * This work was funded in the context of the Q-ImPrESS research project  
 * (FP7-215013) by the European Union under the Information and  
 * Communication Technologies priority of the Seventh Research Framework  
 * Programme.
 *
 * Contributors:
 *     itemis 	- initial API and implementation
 *******************************************************************************/
package de.itemis.qimpress.showcase.pricing_simulator.jsf.beans;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

import javax.faces.event.ActionEvent;
import javax.naming.NamingException;

import org.apache.log4j.Logger;

import de.itemis.qimpress.showcase.pricing_simulator.be.domain.CustomerCategoryDiscount;
import de.itemis.qimpress.showcase.pricing_simulator.be.domain.CustomerTypeDiscount;
import de.itemis.qimpress.showcase.pricing_simulator.be.domain.VolumeDiscount;
import de.itemis.qimpress.showcase.pricing_simulator.be.exceptions.ApplicationException;
import de.itemis.qimpress.showcase.pricing_simulator.be.service.PricingManager;
import de.itemis.qimpress.showcase.pricing_simulator.be.service.ServiceLocator;

/**
 * @author Claudius Haecker
 * This Class manages qimpressPricing.jspx. 
 */
public class PricingSimulatorFeController implements Serializable {
    private static final long serialVersionUID = 8360776696638832915L;
    private static final Logger LOG = Logger.getLogger(PricingSimulatorFeController.class);
    private PricingManager pricingMgr = ServiceLocator.getInstance().getPricingManager();
    private List<VolumeDiscountDelegate> volumeDiscountDelegateList;

    /**
     * constructor for the PricingSimulatorFeController.
     * The constructor instantiates the pricingManager.
     * @throws ApplicationException if an error occured that cannot be resolved in the business layer
     */
    public PricingSimulatorFeController() {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">>  PricingSimulatorFeController");
        }
        if (pricingMgr == null) {
            LOG.error("PricingManager from ServiceLocater was null");
            throw new IllegalStateException("Could not configure PricingManager.");
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("pricingManager = " + pricingMgr);
        }
    }

    /**
     * This method is to be called after all VolumeDiscountDelegates have been updated.
     * It updates the database with the edits of the user (deletions, updates, adds). 
     */
    public void updateVolumeDiscounts(ActionEvent event) {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">> updateVolumeDiscounts");
        }
        ArrayList<VolumeDiscount> modifieds = new ArrayList<VolumeDiscount>();
        ArrayList<VolumeDiscount> deleteds = new ArrayList<VolumeDiscount>();
        for (VolumeDiscountDelegate discount : volumeDiscountDelegateList) {
            if (discount.isMarkedForDeletion()) {
                deleteds.add(discount.getVolumeDiscount());
            } else if (discount.isModifiedByGui()) {
                modifieds.add(discount.getVolumeDiscount());
            }
        }
        try {
            pricingMgr.saveVolumeDiscountList(modifieds);
        } catch (ApplicationException e) {
            LOG.error("Could not save volume discounts", e);
            // TODO (MH) Read error code and trigger FE error handling
        }
        try {
            pricingMgr.deleteVolumeDiscountList(deleteds);
        } catch (ApplicationException e) {
            LOG.error("Could not delete volume discounts", e);
            // TODO (MH) Read error code and trigger FE error handling
        }
    }

    /**
     * Gets the {@link de.itemis.qimpress.showcase.pricing_simulator.be.domain.VolumeDiscount} from the database
     * and wraps them into {@link de.itemis.qimpress.showcase.pricing_simulator.jsf.beans.VolumeDiscountDelegate}s. 
     * 
     * @return list of all stored {@link de.itemis.qimpress.showcase.pricing_simulator.be.domain.VolumeDiscount}
     */
    public List<VolumeDiscountDelegate> getAllVolumeDiscountsFromBackend() {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">> getAllVolumeDiscountsFromBackend");
        }
        List<VolumeDiscount> volumeDiscountList;
        try {
            volumeDiscountList = pricingMgr.getAllVolumeDiscounts();
        } catch (ApplicationException e) {
            LOG.error("Could not get all volume discounts. " + e.getLocalizedMessage(), e);
            // TODO (MH) Read error code and trigger FE error handling
            ArrayList<VolumeDiscountDelegate> discountList = new ArrayList<VolumeDiscountDelegate>();
            
            return discountList;
        }
        volumeDiscountDelegateList = new ArrayList<VolumeDiscountDelegate>(volumeDiscountList.size());
        for (VolumeDiscount discount : volumeDiscountList) {
            volumeDiscountDelegateList.add(new VolumeDiscountDelegate(discount));
        }
        VolumeDiscount volumeDiscount = new VolumeDiscount();
        volumeDiscount.setPercentage(Float.NaN);
        volumeDiscountDelegateList.add(new VolumeDiscountDelegate(volumeDiscount));
        return volumeDiscountDelegateList;
    }

    /**
     * Loads all stored {@link de.itemis.qimpress.showcase.pricing_simulator.be.domain.CustomerCategoryDiscount} from the database.
     * 
     * @return list of all stored {@link de.itemis.qimpress.showcase.pricing_simulator.be.domain.CustomerCategoryDiscount}
     */
    public List<CustomerCategoryDiscount> getAllCustomerCategoryDiscounts() {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">> getAllCustomerCategoryDiscounts");
        }
        try {
            return pricingMgr.getAllCustomerCategoryDiscounts();
        } catch (ApplicationException e) {
            LOG.error("Could not get all customer category discounts. " + e.getLocalizedMessage(), e);
            // TODO (MH) Read error code and trigger FE error handling
            return new ArrayList<CustomerCategoryDiscount>();
        }
    }
    
    /**
     * Loads all stored
     * {@link de.itemis.qimpress.showcase.pricing_simulator.be.domain.CustomerCategoryDiscount CustomerCategoryDiscounts}
     * from the database.
     * 
     * @return list of all stored 
     * {@link de.itemis.qimpress.showcase.pricing_simulator.be.domain.CustomerCategoryDiscount CustomerCategoryDiscounts}
     */
    public List<CustomerTypeDiscount> getAllCustomerTypeDiscounts() {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">> getAllCustomerTypeDiscounts");
        }
        
        try {
            return pricingMgr.getAllCustomerTypeDiscounts();
        } catch (ApplicationException e) {
            LOG.error("Cound not get all customer type discouns: " + e.getLocalizedMessage(), e);
            // TODO (MH) Read error code and trigger FE error handling
            return new ArrayList<CustomerTypeDiscount>();
        }
    }
}