/*******************************************************************************
 * Copyright (c) 2008,2009 Q-ImPrESS consortium
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * This work was funded in the context of the Q-ImPrESS research project  
 * (FP7-215013) by the European Union under the Information and  
 * Communication Technologies priority of the Seventh Research Framework  
 * Programme.
 *
 * Contributors:
 *     itemis 	- initial API and implementation
 *******************************************************************************/
package de.itemis.qimpress.showcase.pricing_simulator.tests.be.service;

import java.math.BigDecimal;
import java.util.List;

import org.apache.log4j.Logger;
import org.junit.BeforeClass;
import org.junit.Test;

import static org.junit.Assert.*;

import de.itemis.qimpress.showcase.pricing_simulator.be.domain.CustomerCategoryDiscount;
import de.itemis.qimpress.showcase.pricing_simulator.be.domain.CustomerTypeDiscount;
import de.itemis.qimpress.showcase.pricing_simulator.be.domain.ProductPrice;
import de.itemis.qimpress.showcase.pricing_simulator.be.domain.VolumeDiscount;
import de.itemis.qimpress.showcase.pricing_simulator.be.dto.Price;
import de.itemis.qimpress.showcase.pricing_simulator.be.exceptions.ApplicationException;
import de.itemis.qimpress.showcase.pricing_simulator.be.service.PricingManager;
import de.itemis.qimpress.showcase.pricing_simulator.be.service.ServiceLocator;

/**
 * Integration test for 
 * {@link de.itemis.qimpress.showcase.pricing_simulator.be.service.PricingManagerImpl} 
 * service interface. Includes only reading methods.
 * 
 * @author Wladimir Safonov
 *
 */
public class PricingManagerReadOnlyTest {

    private static final Logger LOG = Logger.getLogger(PricingManagerReadOnlyTest.class);
    
    private static PricingManager PRICING_MANAGER = ServiceLocator.getInstance().getPricingManager();
    
    @BeforeClass
    public static void initTestSuite() {
        if (LOG.isDebugEnabled()) {
            LOG.debug("initTestSuite()");
        }
        
        try {
        } catch (Exception e) {
            e.printStackTrace();
            fail("Failed to locate PricingManager");
        }
        assertNotNull("PricingManager not found", PRICING_MANAGER);
    }
    
    @Test
    public void testGetAllVolumeDiscounts() {
        if (LOG.isDebugEnabled()) {
            LOG.debug("testGetAllVolumeDiscounts()");
        }
        
        List<VolumeDiscount> allVolumeDiscounts = null;
        try {
            allVolumeDiscounts = PRICING_MANAGER.getAllVolumeDiscounts();
        } catch (ApplicationException e) {
            e.printStackTrace();
            fail("Failed to load all VolumeDiscounts");
        }
        assertNotNull("Failed to load all VolumeDiscounts", allVolumeDiscounts);
        assertTrue("No VolumeDiscounts loaded", allVolumeDiscounts.size() > 0);
    }
    
    @Test
    public void testGetAllCustomerCategoryDiscounts() {
        if (LOG.isDebugEnabled()) {
            LOG.debug("testGetAllCustomerCategoryDiscounts()");
        }
        
        List<CustomerCategoryDiscount> allCustomerCategoryDiscount = null;
        try {
            allCustomerCategoryDiscount = PRICING_MANAGER.getAllCustomerCategoryDiscounts();
        } catch (ApplicationException e) {
            e.printStackTrace();
            fail("Failed to load all CustomerCategoryDiscount");
        }
        assertNotNull("Failed to load all CustomerCategoryDiscount: result is null", allCustomerCategoryDiscount);
        assertTrue("No CustomerCategoryDiscount loaded", allCustomerCategoryDiscount.size() > 0);
    }
    
    @Test
    public void testGetAllCustomerTypeDiscounts() {
        if (LOG.isDebugEnabled()) {
            LOG.debug("testGetAllCustomerTypeDiscounts()");
        }
        
        List<CustomerTypeDiscount> allCustomerTypeDiscounts = null;
        try {
            allCustomerTypeDiscounts = PRICING_MANAGER.getAllCustomerTypeDiscounts();
        } catch (ApplicationException e) {
            e.printStackTrace();
            fail("Failed to load all CustomerTypeDiscounts");
        }
        assertNotNull("Failed to load all CustomerTypeDiscounts: result is null", allCustomerTypeDiscounts);
        assertTrue("No CustomerTypeDiscounts loaded", allCustomerTypeDiscounts.size() > 0);
    }
    
    @Test
    public void testFindProductCountryPrice() {
        if (LOG.isDebugEnabled()) {
            LOG.debug("testFindProductCountryPrice()");
        }
        
        ProductPrice productPrice = null;
        try {
            productPrice = PRICING_MANAGER.findProductCountryPrice("COAT005", "MN");
        } catch (ApplicationException e) {
            e.printStackTrace();
            fail("Failed to find country-specific ProductPrice");
        }
        assertNotNull("Failed to find country-specific ProductPrice", productPrice);
        assertEquals("Unexpected result for ProductPrice request (country-specific price)", 
                new BigDecimal("381.91"), productPrice.getPrice());
        
        try {
            productPrice = PRICING_MANAGER.findProductCountryPrice("COAT005", "SR");
        } catch (ApplicationException e) {
            e.printStackTrace();
            fail("Failed to find country-specific ProductPrice");
        }
        assertNotNull("Failed to find country-specific ProductPrice", productPrice);
        assertEquals("Unexpected result for ProductPrice request (standard price)", 
                new BigDecimal("442.67"), productPrice.getPrice());
        
        try {
            productPrice = PRICING_MANAGER.findProductCountryPrice("NON-EXISTING", "SAME");
        } catch (ApplicationException e) {
            e.printStackTrace();
            fail("Failed to request non-existing ProductPrice");
        }
        assertNull("Unexpected result for non-existing ProductPrice", productPrice);
    }
    
    @Test
    public void testFindVolumeDiscountByQuantity() {
        if (LOG.isDebugEnabled()) {
            LOG.debug("testFindVolumeDiscountByQuantity()");
        }
        
        VolumeDiscount volumeDiscount = null;
        try {
            volumeDiscount = PRICING_MANAGER.findVolumeDiscountByQuantity(5);
        } catch (ApplicationException e) {
            e.printStackTrace();
            fail("Failed to request non-existing VolumeDiscount");
        }
        assertNull("Unexpected result for non-existing VolumeDiscount", volumeDiscount);
        
        try {
            volumeDiscount = PRICING_MANAGER.findVolumeDiscountByQuantity(333);
        } catch (ApplicationException e) {
            e.printStackTrace();
            fail("Failed to find VolumeDiscount by product quantity");
        }
        assertNotNull("Failed to find VolumeDiscount by product quantity (bound discount interval)", volumeDiscount);
        assertEquals("Unexpected result for VolumeDiscount request (bound discount interval)", 
                10.0, volumeDiscount.getPercentage(), 0.01);

        try {
            volumeDiscount = PRICING_MANAGER.findVolumeDiscountByQuantity(3333);
        } catch (ApplicationException e) {
            e.printStackTrace();
            fail("Failed to find VolumeDiscount by product quantity");
        }
        assertNotNull("Failed to find VolumeDiscount by product quantity (unbound discount interval)", volumeDiscount);
        assertEquals("Unexpected result for VolumeDiscount request (unbound discount interval)", 
                15.0, volumeDiscount.getPercentage(), 0.01);
        
    }
    
    @Test
    public void testFindCustomerTypeDiscountByType() {
        if (LOG.isDebugEnabled()) {
            LOG.debug("testFindCustomerTypeDiscountByType()");
        }
        
        CustomerTypeDiscount customerTypeDiscount = null;
        
        try {
            customerTypeDiscount = PRICING_MANAGER.findCustomerTypeDiscountByType("reseller");
        } catch (ApplicationException e) {
            e.printStackTrace();
            fail("Failed to find CustomerTypeDiscount by type");
        }
        assertNotNull("Failed to find CustomerTypeDiscount by type: null result", customerTypeDiscount);
        assertEquals("Unexpected result for CustomerTypeDiscount for 'reseller'", 
                20.0, customerTypeDiscount.getPercentage(), 0.01);
        
        try {
            customerTypeDiscount = PRICING_MANAGER.findCustomerTypeDiscountByType("non-existing");
        } catch (ApplicationException e) {
            e.printStackTrace();
            fail("Failed to find CustomerTypeDiscount by type");
        }
        assertNull("Unexpected result for non-existing CustomerTypeDiscount", customerTypeDiscount);
    }
    
    @Test
    public void testCalculateVolumePriceForCustomerCategory() {
        if (LOG.isDebugEnabled()) {
            LOG.debug("testCalculateVolumePriceForCustomerCategory()");
        }
        
        Price volumePrice = null;
        try {
            // Country price, volume discount, category discount, not reseller
            volumePrice = PRICING_MANAGER.calculateVolumePriceForCustomerCategory(
                    "COAT005", "MN", 30, "B", "private");
            assertNotNull("Failed to calculate volume price: price is null", volumePrice);
            assertEquals("Failed to calculate volume price (country, volume, category, !reseller)", 
                    new BigDecimal("10340.21"), volumePrice.getValue());
            
            // No country price, no volume discount, category discount = 0, reseller
            volumePrice = PRICING_MANAGER.calculateVolumePriceForCustomerCategory(
                    "COAT005", "VE", 3, "C", "reseller");
            assertNotNull("Failed to calculate volume price: price is null", volumePrice);
            assertEquals("Failed to calculate volume price (!country, !volume, category=0, reseller)", 
                    new BigDecimal("1062.41"), volumePrice.getValue());
            
            // No country price, highest volume discount, no category discount, reseller
            volumePrice = PRICING_MANAGER.calculateVolumePriceForCustomerCategory(
                    "COSA005", "UA", 2000, "Z", "reseller");
            assertNotNull("Failed to calculate volume price: price is null", volumePrice);
            assertEquals("Failed to calculate volume price (!country, high volume, !category, reseller)", 
                    new BigDecimal("63920.00"), volumePrice.getValue());
        } catch (ApplicationException e) {
            e.printStackTrace();
            fail("Failed to calculate volume price");
        }
    }
}
