/*******************************************************************************
 * Copyright (c) 2008,2009 Q-ImPrESS consortium
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * This work was funded in the context of the Q-ImPrESS research project  
 * (FP7-215013) by the European Union under the Information and  
 * Communication Technologies priority of the Seventh Research Framework  
 * Programme.
 *
 * Contributors:
 *     itemis 	- initial API and implementation
 *******************************************************************************/
package de.itemis.qimpress.showcase.pricing_simulator.be.service;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.List;

import org.apache.log4j.Logger;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import de.itemis.qimpress.common.Meassurements;
import de.itemis.qimpress.showcase.pricing_simulator.be.dao.CustomerCategoryDiscountDao;
import de.itemis.qimpress.showcase.pricing_simulator.be.dao.CustomerTypeDiscountDao;
import de.itemis.qimpress.showcase.pricing_simulator.be.dao.ProductPriceDao;
import de.itemis.qimpress.showcase.pricing_simulator.be.dao.VolumeDiscountDao;
import de.itemis.qimpress.showcase.pricing_simulator.be.domain.CustomerCategoryDiscount;
import de.itemis.qimpress.showcase.pricing_simulator.be.domain.CustomerTypeDiscount;
import de.itemis.qimpress.showcase.pricing_simulator.be.domain.ProductPrice;
import de.itemis.qimpress.showcase.pricing_simulator.be.domain.VolumeDiscount;
import de.itemis.qimpress.showcase.pricing_simulator.be.dto.Customer;
import de.itemis.qimpress.showcase.pricing_simulator.be.dto.Order;
import de.itemis.qimpress.showcase.pricing_simulator.be.dto.OrderItem;
import de.itemis.qimpress.showcase.pricing_simulator.be.dto.Price;
import de.itemis.qimpress.showcase.pricing_simulator.be.exceptions.ApplicationException;
import de.itemis.qimpress.showcase.pricing_simulator.be.exceptions.DaoException;
import de.itemis.qimpress.showcase.pricing_simulator.be.exceptions.ApplicationException.ApplicationErrorCode;

/**
 * @author Wladimir Safonov
 *
 */
public class PricingManagerImpl implements PricingManager {

    // PricingManagerImpl logger
    private static final Logger LOG = Logger.getLogger(PricingManagerImpl.class);

    private VolumeDiscountDao volumeDiscountDao;
    private CustomerCategoryDiscountDao customerCategoryDiscountDao;
    private ProductPriceDao productPriceDao;
    private CustomerTypeDiscountDao customerTypeDiscountDao;

    /**
     * @param customerTypeDiscountDao the customerTypeDiscountDao to set
     */
    public void setCustomerTypeDiscountDao(CustomerTypeDiscountDao customerTypeDiscountDao) {
        this.customerTypeDiscountDao = customerTypeDiscountDao;
    }

    /**
     * @param volumeDiscountDao the volumeDiscountDao to set
     */
    public void setVolumeDiscountDao(VolumeDiscountDao volumeDiscountDao) {
        this.volumeDiscountDao = volumeDiscountDao;
    }

    /**
     * @param customerCategoryDiscountDao the customerCategoryDiscountDao to set
     */
    public void setCustomerCategoryDiscountDao(CustomerCategoryDiscountDao customerCategoryDiscountDao) {
        this.customerCategoryDiscountDao = customerCategoryDiscountDao;
    }

    /**
     * @param productPriceDao the productPriceDao to set
     */
    public void setProductPriceDao(ProductPriceDao productPriceDao) {
        this.productPriceDao = productPriceDao;
    }

    /* (non-Javadoc)
     * @see de.itemis.qimpress.showcase.pricing_simulator.be.service.PricingManager#getAllCustomerCategoryDiscounts()
     */
    /* @Override */
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true)
    public List<CustomerCategoryDiscount> getAllCustomerCategoryDiscounts() throws ApplicationException {
       
		long start         = 0;
		long stop          = 0;

    	if (LOG.isTraceEnabled()) {
    		start = System.currentTimeMillis();
    	}
    	
    	if (LOG.isDebugEnabled()) {
            LOG.debug("getAllCustomerCategoryDiscounts()");
        }

        List<CustomerCategoryDiscount> customerCategoryDiscountList = null;
        try {
            customerCategoryDiscountList = customerCategoryDiscountDao.getAll();
        } catch (DaoException e) {
            throw new ApplicationException(ApplicationErrorCode.TE, e);
        }
        
        if (LOG.isTraceEnabled()) {
        	stop = System.currentTimeMillis();
        	Meassurements.add(this.getClass().getSimpleName(), "getAllCustomerCategoryDiscounts", start, stop);
        }
        
        return customerCategoryDiscountList;
    }

    /* (non-Javadoc)
     * @see de.itemis.qimpress.showcase.pricing_simulator.be.service.PricingManager#getAllVolumeDiscounts()
     */
    /* @Override */
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true)
    public List<VolumeDiscount> getAllVolumeDiscounts() throws ApplicationException {
       
		long start         = 0;
		long stop          = 0;

    	if (LOG.isTraceEnabled()) {
    		start = System.currentTimeMillis();
    	}
    	
    	if (LOG.isDebugEnabled()) {
            LOG.debug("getAllVolumeDiscounts()");
        }

        List<VolumeDiscount> volumeDiscountList = null;
        try {
            volumeDiscountList = volumeDiscountDao.getAll();
        } catch (DaoException e) {
            throw new ApplicationException(ApplicationErrorCode.TE, e);
        }
        
        if (LOG.isTraceEnabled()) {
        	stop = System.currentTimeMillis();
        	Meassurements.add(this.getClass().getSimpleName(), "getAllVolumeDiscounts", start, stop);
        }
        
        return volumeDiscountList;
    }

    /* @Override */
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true)
    public List<CustomerTypeDiscount> getAllCustomerTypeDiscounts() throws ApplicationException {
        
		long start         = 0;
		long stop          = 0;

    	if (LOG.isTraceEnabled()) {
    		start = System.currentTimeMillis();
    	}
    	
    	if (LOG.isDebugEnabled()) {
            LOG.debug("getAllCustomerTypeDiscounts()");
        }

        List<CustomerTypeDiscount> customerTypeDiscountList = null;
        try {
            customerTypeDiscountList = customerTypeDiscountDao.getAll();
        } catch (DaoException e) {
            throw new ApplicationException(ApplicationErrorCode.TE, e);
        }
        
        if (LOG.isTraceEnabled()) {
        	stop = System.currentTimeMillis();
        	Meassurements.add(this.getClass().getSimpleName(), "getAllCustomerTypeDiscounts", start, stop);
        }
        
        return customerTypeDiscountList;
    }

    /* (non-Javadoc)
     * @see de.itemis.qimpress.showcase.pricing_simulator.be.service.PricingManager#findProductCountryPrice()
     */
    /* @Override */
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true)
    public ProductPrice findProductCountryPrice(String productCode, String countryCode) throws ApplicationException {
        
		long start         = 0;
		long stop          = 0;

    	if (LOG.isTraceEnabled()) {
    		start = System.currentTimeMillis();
    	}
    	
    	if (LOG.isDebugEnabled()) {
            LOG.debug("findProductCountryPrice()");
        }

        ProductPrice productPrice;
        try {
            productPrice = productPriceDao.findCountryPrice(productCode, countryCode);
        } catch (DaoException e) {
            throw new ApplicationException(ApplicationErrorCode.TE, e);
        }
        
        if (LOG.isTraceEnabled()) {
        	stop = System.currentTimeMillis();
        	Meassurements.add(this.getClass().getSimpleName(), "findProductCountryPrice", start, stop);
        }
        
        return productPrice;
    }

    /* (non-Javadoc)
     * @see de.itemis.qimpress.showcase.pricing_simulator.be.service.PricingManager#findVolumeDiscountByQuantity()
     */
    /* @Override */
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true)
    public VolumeDiscount findVolumeDiscountByQuantity(int productQuantity) throws ApplicationException {
        
		long start         = 0;
		long stop          = 0;

    	if (LOG.isTraceEnabled()) {
    		start = System.currentTimeMillis();
    	}
    	
    	if (LOG.isDebugEnabled()) {
            LOG.debug("findVolumeDiscountByQuantity()");
        }

        VolumeDiscount volumeDiscount;
        try {
            volumeDiscount = volumeDiscountDao.findByQuantity(productQuantity);
        } catch (DaoException e) {
            throw new ApplicationException(ApplicationErrorCode.TE, e);
        }
        
        if (LOG.isTraceEnabled()) {
        	stop = System.currentTimeMillis();
        	Meassurements.add(this.getClass().getSimpleName(), "findVolumeDiscountByQuantity", start, stop);
        }
        
        return volumeDiscount;
    }

    /* @Override */
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true)
    public CustomerTypeDiscount findCustomerTypeDiscountByType(String customerType) throws ApplicationException {
		
    	long start         = 0;
		long stop          = 0;

    	if (LOG.isTraceEnabled()) {
    		start = System.currentTimeMillis();
    	}
    	
    	if (LOG.isDebugEnabled()) {
            LOG.debug("findCustomerTypeDiscountByType()");
        }

        CustomerTypeDiscount typeDiscount = null;
        try {
            typeDiscount = customerTypeDiscountDao.getByName(customerType);
        } catch (DaoException e) {
            throw new ApplicationException(ApplicationErrorCode.TE, e);
        }
        
        if (LOG.isTraceEnabled()) {
        	stop = System.currentTimeMillis();
        	Meassurements.add(this.getClass().getSimpleName(), "findCustomerTypeDiscountByType", start, stop);
        }
        
        return typeDiscount;
    }

    /* (non-Javadoc)
     * @see de.itemis.qimpress.showcase.pricing_simulator.be.service.PricingManager#calculateVolumePriceForCustomerCategory()
     */
    /* @Override */
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true)
    public Price calculateVolumePriceForCustomerCategory(String productCode, String countryCode, int quantity,
            String customerCategoryKey, String customerType) throws ApplicationException {
        
		long start               = 0;
		long stop                = 0;
		long startInternalAction = 0;

    	if (LOG.isTraceEnabled()) {
    		start = System.currentTimeMillis();
    	}
    	
    	if (LOG.isDebugEnabled()) {
            LOG.debug("calculateVolumePriceForCustomerCategory()");
        }

        Price volumePrice = new Price();

        // get product price
        ProductPrice productPrice = this.findProductCountryPrice(productCode, countryCode);
        if (productPrice == null) {
            return null;
        }
        // get discounts
        VolumeDiscount volumeDiscount = this.findVolumeDiscountByQuantity(quantity);
        CustomerCategoryDiscount customerCategoryDiscount = null;
        CustomerTypeDiscount customerTypeDiscount = null;
        try {
            customerCategoryDiscount = this.customerCategoryDiscountDao.getByCategoryKey(customerCategoryKey);
            customerTypeDiscount = this.customerTypeDiscountDao.getByName(customerType);
        } catch (DaoException e) {
            throw new ApplicationException(ApplicationErrorCode.TE, e);
        }

        if (LOG.isTraceEnabled()) {
        	startInternalAction = System.currentTimeMillis();
        }
        
        BigDecimal price = productPrice.getPrice();
        price = price.multiply(new BigDecimal(quantity));
        // apply discounts 
        if (volumeDiscount != null) {
            price = price.multiply(new BigDecimal((100.0 - volumeDiscount.getPercentage()) / 100.0));
        }
        if (customerCategoryDiscount != null) {
            price = price.multiply(new BigDecimal((100.0 - customerCategoryDiscount.getPercentage()) / 100.0));
        }
        if (customerTypeDiscount != null) {
            price = price.multiply(new BigDecimal((100.0 - customerTypeDiscount.getPercentage()) / 100.0));
        }
        price = price.setScale(2, RoundingMode.HALF_UP);
        volumePrice.setValue(price);
        volumePrice.setCurrencyCode(productPrice.getCurrency().getIsoCode());
        volumePrice.setCurrencySymbol(productPrice.getCurrency().getSign());
        
        if (LOG.isTraceEnabled()) {
        	stop = System.currentTimeMillis();
      		
    		Meassurements.add(this.getClass().getSimpleName(), "calculateVolumePriceForCustomerCategory_internalAction", startInternalAction, stop);
    		Meassurements.add(this.getClass().getSimpleName(), "calculateVolumePriceForCustomerCategory", start, stop);
        }
        
        return volumePrice;
    }

    /* (non-Javadoc)
     * @see de.itemis.qimpress.showcase.pricing_simulator.be.service.PricingManager#saveVolumeDiscountList()
     */
    /* @Override */
    @Transactional(propagation = Propagation.REQUIRED)
    public void saveVolumeDiscountList(List<VolumeDiscount> volumeDiscountsList) throws ApplicationException {
        if (LOG.isDebugEnabled()) {
            LOG.debug("saveVolumeDiscountList()");
        }

        try {
            volumeDiscountDao.save(volumeDiscountsList);
        } catch (DaoException e) {
            throw new ApplicationException(ApplicationErrorCode.TE, e);
        }
    }

    /* (non-Javadoc)
     * @see de.itemis.qimpress.showcase.pricing_simulator.be.service.PricingManager#deleteVolumeDiscountList()
     */
    /* @Override */
    @Transactional(propagation = Propagation.REQUIRED)
    public void deleteVolumeDiscountList(List<VolumeDiscount> volumeDiscountsList) throws ApplicationException {
        if (LOG.isDebugEnabled()) {
            LOG.debug("deleteVolumeDiscountList()");
        }

        try {
            volumeDiscountDao.delete(volumeDiscountsList);
        } catch (DaoException e) {
            throw new ApplicationException(ApplicationErrorCode.TE, e);
        }
    }

    /* @Override */
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true)
    public Order priceOrder(Order unpricedOrder) throws ApplicationException {
        if (LOG.isDebugEnabled()) {
            LOG.debug("priceOrder()");
        }

        Order pricedOrder = unpricedOrder;
        Customer customer = unpricedOrder.getCustomer();

        String customerCat = (customer.getCustomerCategory() != null) ? customer.getCustomerCategory()
                .getCustomerCategoryKey() : null;
        String customerType = (customer.getCustomerType() != null) ? customer.getCustomerType().getCustomerTypeName()
                : null;

        BigDecimal totalOrderPrice = new BigDecimal(0);

        // calculate price for each order item
        if (pricedOrder.getOrderItems() != null) {
            for (OrderItem orderItem : pricedOrder.getOrderItems()) {

                Price itemPrice = this.calculateVolumePriceForCustomerCategory(orderItem.getProduct().getProductCode(),
                        customer.getInvoiceAddressCountry().getIsoCode(), orderItem.getQuantity(), customerCat,
                        customerType);
                if (itemPrice != null) {
                    orderItem.setTotalPrice(itemPrice.getValue());
                    totalOrderPrice = totalOrderPrice.add(itemPrice.getValue());
                }
            }
        }

        pricedOrder.setTotalPrice(totalOrderPrice);
        return pricedOrder;
    }

}
