/*******************************************************************************
 * Copyright (c) 2008,2009 Q-ImPrESS consortium
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * This work was funded in the context of the Q-ImPrESS research project  
 * (FP7-215013) by the European Union under the Information and  
 * Communication Technologies priority of the Seventh Research Framework  
 * Programme.
 *
 * Contributors:
 *     itemis 	- initial API and implementation
 *******************************************************************************/
package de.itemis.qimpress.showcase.pdm_simulator.webservice;

import java.io.IOException;
import java.io.StringWriter;
import java.text.SimpleDateFormat;
import java.util.List;

import javax.naming.NamingException;

import org.apache.log4j.Logger;
import org.jdom.Element;
import org.jdom.Namespace;
import org.springframework.ws.server.endpoint.AbstractJDomPayloadEndpoint;

import com.csvreader.CsvWriter;

import de.itemis.qimpress.showcase.pdm_simulator.be.dao.ProductGroupDao;
import de.itemis.qimpress.showcase.pdm_simulator.be.domain.AttributeDefinition;
import de.itemis.qimpress.showcase.pdm_simulator.be.domain.AttributeValue;
import de.itemis.qimpress.showcase.pdm_simulator.be.domain.Product;
import de.itemis.qimpress.showcase.pdm_simulator.be.domain.ProductGroup;
import de.itemis.qimpress.showcase.pdm_simulator.be.service.PDMManager;
import de.itemis.qimpress.showcase.pdm_simulator.be.service.ServiceLocator;

/**
 * Webservice endpoint for PDMManager service interface 
 * using JDom for payload processing. Payload data is 
 * represented as CSV for bulk processing.
 * 
 * 
 * @author Wladimir Safonov
 *
 */
public class PdmCsvJDomEndpoint extends AbstractJDomPayloadEndpoint {

    /** Logger for PdmCsvJDomEndpoint. */
    private static final Logger LOG = Logger.getLogger(PdmCsvJDomEndpoint.class);

    public static final String GET_PRODUCT_GROUPS_CSV_REQUEST = "GetProductGroupsCSVRequest";
    public static final String GET_PRODUCT_GROUPS_CSV_RESPONSE = "GetProductGroupsCSVResponse";
    public static final String GET_PRODUCTS_CSV_REQUEST = "GetProductsCSVRequest";
    public static final String GET_PRODUCTS_CSV_RESPONSE = "GetProductsCSVResponse";
    public static final String GET_ATTRIBUTE_VALUES_CSV_REQUEST = "GetAttributeValuesCSVRequest";
    public static final String GET_ATTRIBUTE_VALUES_CSV_RESPONSE = "GetAttributeValuesCSVResponse";
    public static final String GET_ATTRIBUTE_DEFINITIONS_CSV_REQUEST = "GetAttributeDefinitionsCSVRequest";
    public static final String GET_ATTRIBUTE_DEFINITIONS_CSV_RESPONSE = "GetAttributeDefinitionsCSVResponse";

    // PDM internal service interface
    private PDMManager pdmManager = ServiceLocator.getInstance().getPDMManager();

    private Namespace namespace;
    private final SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd hh:mm:ss");

    public PdmCsvJDomEndpoint() {

        
        namespace = Namespace.getNamespace("pdm", "http://www.itemis.com/pdm/schemas");
    }

    /* (non-Javadoc)
     * @see org.springframework.ws.server.endpoint.AbstractJDomPayloadEndpoint#invokeInternal(org.jdom.Element)
     */
    @Override
    protected Element invokeInternal(Element request) throws Exception {
        if (LOG.isDebugEnabled()) {
            LOG.debug("\tWebservice called: invokeInternal");
        }

        String rootElement = request.getName();
        if (GET_PRODUCT_GROUPS_CSV_REQUEST.equals(rootElement)) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("\troot element detected: " + GET_PRODUCT_GROUPS_CSV_REQUEST);
            }
            // no parameters expected
            List<ProductGroup> productGroups = pdmManager.getProductGroups();
            if (LOG.isDebugEnabled()) {
                LOG.debug("\tproductGroupList size = " + productGroups.size());
            }
            return createGetProductGroupsCSVResponse(productGroups);
        }
        if (GET_PRODUCTS_CSV_REQUEST.equals(rootElement)) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("\troot element detected: " + GET_PRODUCT_GROUPS_CSV_REQUEST);
            }
            // no parameters expected
            List<Product> products = pdmManager.getProducts();
            if (LOG.isDebugEnabled()) {
                LOG.debug("\tproductList size = " + products.size());
            }
            return createGetProductsCSVResponse(products);
        }
        if (GET_ATTRIBUTE_VALUES_CSV_REQUEST.equals(rootElement)) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("\troot element detected: " + GET_ATTRIBUTE_VALUES_CSV_REQUEST);
            }
            // no parameters expected
            List<AttributeValue> attributeValues = pdmManager.getAttributeValues();
            if (LOG.isDebugEnabled()) {
                LOG.debug("\tattributeValuesList size = " + attributeValues.size());
            }
            return createGetAttributeValuesCSVResponse(attributeValues);
        }
        if (GET_ATTRIBUTE_DEFINITIONS_CSV_REQUEST.equals(rootElement)) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("\troot element detected: " + GET_ATTRIBUTE_DEFINITIONS_CSV_REQUEST);
            }
            // no parameters expected
            List<AttributeDefinition> attributeDefinitions = pdmManager.getAttributeDefinitions();
            if (LOG.isDebugEnabled()) {
                LOG.debug("\tattributeDefinitionList size = " + attributeDefinitions.size());
            }
            return createGetAttributeDefinitionsCSVResponse(attributeDefinitions);
        }
        
        return null;
    }

    /**
     * Creates response element with a CSV formatted list of product groups
     * 
     * @param productGroupList list of product groups
     * @return response element with a product group list in CSV format
     */
    private Element createGetProductGroupsCSVResponse(List<ProductGroup> productGroupList) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("\tcreate " + GET_PRODUCT_GROUPS_CSV_RESPONSE);
        }

        Element csvResponseRootElement = new Element(GET_PRODUCT_GROUPS_CSV_RESPONSE, namespace);
        Element csvContentElement = new Element("ProductGroupsCSV", namespace);
        csvResponseRootElement.setContent(csvContentElement);

        StringWriter csvContentWriter = new StringWriter();
        CsvWriter csvWriter = new CsvWriter(csvContentWriter, ';');
        // write CSV header
        try {
            csvWriter.writeRecord(new String[] { "Product_Group_Code", "Product_Group_Name", "Product_Group_Description",
                    "Product_Group_Tree_Depth", "Parent_Product_Group_Code" });
            // set text qualifier
            csvWriter.setTextQualifier('"');
            csvWriter.setForceQualifier(true);
            // iterate the product group list and add each group as a CSV line to response
            for (ProductGroup productGroup : productGroupList) {
                if (!productGroup.isRootProductGroup()) {
                    csvWriter.write(productGroup.getProductGroupCode());
                    csvWriter.write(productGroup.getProductGroupName());
                    csvWriter.write(productGroup.getProductGroupDescription());
                    csvWriter.write(Integer.toString(productGroup.getLevel()));
                    // detach from the root: successors of the root group point to itself
                    if (productGroup.getParentProductGroup().isRootProductGroup()) {
                        csvWriter.write(productGroup.getProductGroupCode());
                    } else {
                        csvWriter.write(productGroup.getParentProductGroup().getProductGroupCode());
                    }
                    csvWriter.endRecord();
                }
            }
        } catch (IOException e) {
            LOG.error("Failed to create CSV content", e);
        }
        csvContentElement.setText(csvContentWriter.toString());
        return csvResponseRootElement;
    }

    /**
     * Creates response element with a CSV formatted list of products
     * 
     * @param productList list of products
     * @return response element with a product list in CSV format
     */
    private Element createGetProductsCSVResponse(List<Product> productList) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("\tcreate " + GET_PRODUCTS_CSV_RESPONSE);
        }

        Element csvResponseRootElement = new Element(GET_PRODUCTS_CSV_RESPONSE, namespace);
        Element csvContentElement = new Element("ProductsCSV", namespace);
        csvResponseRootElement.setContent(csvContentElement);

        StringWriter csvContentWriter = new StringWriter();
        CsvWriter csvWriter = new CsvWriter(csvContentWriter, ';');
        // write CSV header
        try {
            csvWriter.writeRecord(new String[] { "Product_Code", "Product_Name", "Product_Manufacturer_Key",
                    "Product_Group_Code", "Product_Group_Tree_Depth" });
            // set text qualifier
            csvWriter.setTextQualifier('"');
            csvWriter.setForceQualifier(true);
            // iterate the product list and add each product as a CSV line to response
            for (Product product : productList) {
                csvWriter.write(product.getProductCode());
                csvWriter.write(product.getProductName());
                csvWriter.write(product.getProductManufacturerKey());
                csvWriter.write(product.getProductGroup().getProductGroupCode());
                csvWriter.write(Integer.toString(product.getProductGroup().getLevel()));
                csvWriter.endRecord();
            }
        } catch (IOException e) {
            LOG.error("Failed to create CSV content", e);
        }
        csvContentElement.setText(csvContentWriter.toString());
        return csvResponseRootElement;
    }

    /**
     * Creates response element with a CSV formatted list of attribute values
     * 
     * @param attributeValueList list of attribute values
     * @return response element with an attribute value list in CSV format
     */
    private Element createGetAttributeValuesCSVResponse(List<AttributeValue> attributeValueList) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("\tcreate " + GET_ATTRIBUTE_VALUES_CSV_RESPONSE);
        }

        Element csvResponseRootElement = new Element(GET_ATTRIBUTE_VALUES_CSV_RESPONSE, namespace);
        Element csvContentElement = new Element("AttributeValuesCSV", namespace);
        csvResponseRootElement.setContent(csvContentElement);

        StringWriter csvContentWriter = new StringWriter();
        CsvWriter csvWriter = new CsvWriter(csvContentWriter, ';');
        // write CSV header
        try {
            csvWriter.writeRecord(new String[] { "Product_Code", "Product_Manufacturer_Key", "Attribute_Value",
                    "Attribute_Definition_Code" });
            // set text qualifier
            csvWriter.setTextQualifier('"');
            csvWriter.setForceQualifier(true);
            // iterate the attribute value list and add each value as a CSV line to response
            for (AttributeValue attributeValue : attributeValueList) {
                csvWriter.write(attributeValue.getProduct().getProductCode());
                csvWriter.write(attributeValue.getProduct().getProductManufacturerKey());
                csvWriter.write(attributeValue.getAttributeValueText());
                csvWriter.write(attributeValue.getAttributeDefinition().getAttributeDefinitionName());
                csvWriter.endRecord();
            }
        } catch (IOException e) {
            LOG.error("Failed to create CSV content", e);
        }
        csvContentElement.setText(csvContentWriter.toString());
        return csvResponseRootElement;
    }    

    /**
     * Creates response element with a CSV formatted list of attribute definitions
     * 
     * @param attributeDefinitionList list of attribute definitions
     * @return response element with an attribute definition list in CSV format
     */
    private Element createGetAttributeDefinitionsCSVResponse(List<AttributeDefinition> attributeDefinitionList) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("\tcreate " + GET_ATTRIBUTE_DEFINITIONS_CSV_RESPONSE);
        }

        Element csvResponseRootElement = new Element(GET_ATTRIBUTE_DEFINITIONS_CSV_RESPONSE, namespace);
        Element csvContentElement = new Element("AttributeDefinitionsCSV", namespace);
        csvResponseRootElement.setContent(csvContentElement);

        StringWriter csvContentWriter = new StringWriter();
        CsvWriter csvWriter = new CsvWriter(csvContentWriter, ';');
        // write CSV header
        try {
            csvWriter.writeRecord(new String[] { "Attribute_Definition_Code", "Attribute_Definition_Name", "Attribute_Definition_Description" });
            // set text qualifier
            csvWriter.setTextQualifier('"');
            csvWriter.setForceQualifier(true);
            // iterate the attribute value list and add each value as a CSV line to response
            for (AttributeDefinition attributeDefinition : attributeDefinitionList) {
                csvWriter.write(attributeDefinition.getAttributeDefinitionCode());
                csvWriter.write(attributeDefinition.getAttributeDefinitionName());
                csvWriter.write(attributeDefinition.getAttributeDefinitionDescription());
                csvWriter.endRecord();
            }
        } catch (IOException e) {
            LOG.error("Failed to create CSV content", e);
        }
        csvContentElement.setText(csvContentWriter.toString());
        return csvResponseRootElement;
    }    

}
