/*******************************************************************************
 * Copyright (c) 2008,2009 Q-ImPrESS consortium
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * This work was funded in the context of the Q-ImPrESS research project  
 * (FP7-215013) by the European Union under the Information and  
 * Communication Technologies priority of the Seventh Research Framework  
 * Programme.
 *
 * Contributors:
 *     itemis 	- initial API and implementation
 *******************************************************************************/
package de.itemis.qimpress.showcase.pdm_simulator.jsf.beans;

import java.io.Serializable;
import java.text.DateFormat;
import java.util.Collection;
import java.util.Date;
import java.util.LinkedList;
import java.util.List;

import javax.faces.context.FacesContext;
import javax.faces.event.ActionEvent;
import javax.naming.NamingException;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeModel;

import org.apache.log4j.Logger;

import com.icesoft.faces.component.tree.IceUserObject;

import de.itemis.qimpress.showcase.pdm_simulator.be.domain.AttributeValue;
import de.itemis.qimpress.showcase.pdm_simulator.be.domain.Product;
import de.itemis.qimpress.showcase.pdm_simulator.be.domain.ProductGroup;
import de.itemis.qimpress.showcase.pdm_simulator.be.exceptions.ApplicationException;
import de.itemis.qimpress.showcase.pdm_simulator.be.service.PDMManager;
import de.itemis.qimpress.showcase.pdm_simulator.be.service.ServiceLocator;

/**
 * @author Claudius Haecker
  * <p>
 * A basic backing bean for a ice:tree component.  The only instance variable
 * needed is a DefaultTreeModel Object which is bound to the icefaces tree
 * component in the jspx code.</p>
 * <p>
 * The tree created by this backing bean is very simple, containing only text
 * nodes.  The plus and minus icons which expand the tree are rendered because
 * of attributes set at the component level.
 * </p>
*
 */
public class PdmSimulatorFeController implements Serializable {
    private static final long serialVersionUID = 1478131094029853936L;
    private static final Logger LOG = Logger.getLogger(PdmSimulatorFeController.class);
    private PDMManager pdmManager = ServiceLocator.getInstance().getPDMManager();

    // tree default model, used as a value for the tree component
    private DefaultTreeModel treeModel;

    private List<Product> products;
    private Product product = null;

    private boolean attributeDetailsRendered = false;

    private List<AttributeValue> productAttributes = new LinkedList<AttributeValue>();
    private AttributeValue attribute;

    /**
     * @return the attributeDetailsRendered
     */
    public boolean isAttributeDetailsRendered() {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">> isAttributeDetailsRendered");
        }
        return attributeDetailsRendered;
    }

    /**
     * @return the product
     */
    public Product getProduct() {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">> getProduct");
        }
        return product;
    }

    /**
     * constructor for the PdmSimulatorFeController.
     * The constructor instantiates the pdmManager.
     * @throws ApplicationException if an error occured that cannot be resolved in the business layer
     */
    public PdmSimulatorFeController() throws ApplicationException {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">>  PdmSimulatorFeController");
        }
        if (pdmManager == null) {
            LOG.error("PDMManager from ServiceLocater was null");
            throw new IllegalStateException("Could not configure PDMManager.");
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("pdmManager = " + pdmManager);
        }
        //loadFilteredProductsFromBE(pdmManager.getProductGroupRoot());
        // TODO: should be in getTreeModel()
        treeModel = new DefaultTreeModel(getTree(pdmManager.getProductGroupRoot()));
    }

    /**
     * Gets the tree's default model. This is a tree with the names of the product groups
     * @return tree model.
     * @throws ApplicationException problems in Backend
     */
    public DefaultTreeModel getTreeModel() throws ApplicationException {
        // model is accessed by by the ice:tree component

        // (performance problems) treeModel = new DefaultTreeModel(getTree(pdmManager.getProductGroupRoot()));
        return treeModel;
    }

    public boolean isAttributesToRender() {
        return product != null;
    }

    /**
     * Get a list of the products for the currently selected product group.
     * @return list of products
     * @throws ApplicationException problems in Backend
     */
    public List<Product> getProducts() {
        return products;
    }

    /**
     * Actionlistener if someone selects a group in the tree
     * @param event
     * @throws ApplicationException if an error occured that cannot be resolved in the business layer
     */
    public void groupSelected(ActionEvent event) throws ApplicationException {
        updateDataForProductGroup((String) FacesContext.getCurrentInstance().getExternalContext()
                .getRequestParameterMap().get("productGroupId"));
    }

    /**
     * Actionlistener if someone selects an attribute
     * @param event ignored
     */
    public void closeAttributeDetails(ActionEvent event) {
        attributeDetailsRendered = false;
    }

    /**
     * Actionlistener if someone selects an attribute
     * @param event ignored
     */
    public void openAttributeDetails(ActionEvent event) {
        attributeDetailsRendered = true;
        this.attribute = getDataForAttribute((String) FacesContext.getCurrentInstance().getExternalContext()
                .getRequestParameterMap().get("detailAttributeValueId"));
    }

    /**
     * @return 
     * @throws ApplicationException if an error occured that cannot be resolved in the business layer
     */
    AttributeValue getDataForAttribute(String detailAttributeValueId) {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">> productAttributes  = " + detailAttributeValueId);
        }
        if (productAttributes != null) {
            for (AttributeValue av : productAttributes) {
                if (av.getAttributeValueId().equals(detailAttributeValueId)) {
                    return av;
                }
            }
        }
        return null;
    }

    /**
     * @throws ApplicationException if an error occured that cannot be resolved in the business layer
     */
    void updateDataForProductGroup(String productGroupId) throws ApplicationException {
        products = pdmManager.getProducts(productGroupId);
        product = null;
    }

    /**
     * Actionlistener if someone selects a product name in the product list (should get product attributes and update this property)
     * @param event
     * @throws ApplicationException if an error occured that cannot be resolved in the business layer
     */
    public void productSelected(ActionEvent event) throws ApplicationException {
        if (LOG.isDebugEnabled()) {
            LOG.debug("\t=> productSelected *****************************************");
            LOG.debug("\t=> product = d " + product);
        }
        // get product id
        String productId = (String) FacesContext.getCurrentInstance().getExternalContext().getRequestParameterMap()
                .get("productId");
        updateDataForProduct(productId);
    }

    /**
     * @param productId
     * @throws ApplicationException if an error occured that cannot be resolved in the business layer
     */
    void updateDataForProduct(String productId) throws ApplicationException {
        productAttributes = pdmManager.getAttributes(productId);
        if (LOG.isDebugEnabled()) {
            LOG.debug("\t=> productAttributes  = " + productAttributes);
        }
        if (products != null) {
            for (Product prod : products) {
                if (prod.getProductId().equals(productId)) {
                    this.product = prod;
                    break;
                }
            }
        }
    }

    public static String formattedDate(Date date) {
        DateFormat df = DateFormat.getDateTimeInstance(DateFormat.SHORT, DateFormat.SHORT); //, Locale.GERMAN);
        return df.format(date);
    }

    /**
     * recursively generates a tree of type DefaultMutableTreeNode which models the subtree 
     * @param subTree a tree of all product groups
     * @return a tree with the names of the product groups 
     */
    private DefaultMutableTreeNode getTree(ProductGroup subTree) {
        ExtendedDefaultMutableTreeNode tree = new ExtendedDefaultMutableTreeNode();
        if (LOG.isDebugEnabled()) {
            LOG.debug("\t=> subTree ID          = " + subTree.getProductGroupId());
            LOG.debug("\t=> subTree CODE        = " + subTree.getProductGroupCode());
            LOG.debug("\t=> subTree NAME        = " + subTree.getProductGroupName());
            LOG.debug("\t=> subTree DESCRIPTION = " + subTree.getProductGroupDescription());
            LOG.debug("\t=> subTree PARENT      = " + subTree.getParentProductGroup());
        }

        IceUserObject rootObject = new IceUserObject(tree);
        rootObject.setText(subTree.getProductGroupName());
        rootObject.setExpanded(true);
        rootObject.setTooltip(subTree.getProductGroupDescription());
        tree.setUserObject(rootObject);
        tree.setProductGroup(subTree);

        Collection<ProductGroup> childProductGroups = subTree.getChildProductGroups();
        // is leaf if no childProductGroups exist
        rootObject.setLeaf(childProductGroups.size() == 0);
        if (LOG.isDebugEnabled()) {
            LOG.debug("\t=> CHILD-ProductGroups-SIZE  = " + childProductGroups.size());
        }
        for (ProductGroup pg : childProductGroups) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("\t=> ProductGroupName = " + pg.getProductGroupName() + " - parentID:"
                        + pg.getParentProductGroup().getProductGroupId());
            }
            // recursive call
            tree.add(getTree(pg));
        }
        return tree;
    }

    /**
     * @return the productAttributes
     */
    public List<AttributeValue> getProductAttributes() {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">> getProductAttributes");
        }
        return productAttributes;
    }

    /**
     * @return the attribute
     */
    public AttributeValue getAttribute() {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">> getAttribute");
        }
        return attribute;
    }
}