/*******************************************************************************
 * Copyright (c) 2008,2009 Q-ImPrESS consortium
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * This work was funded in the context of the Q-ImPrESS research project  
 * (FP7-215013) by the European Union under the Information and  
 * Communication Technologies priority of the Seventh Research Framework  
 * Programme.
 *
 * Contributors:
 *     itemis 	- initial API and implementation
 *******************************************************************************/
package de.itemis.qimpress.showcase.pdm_simulator.be.service;

import java.util.List;

import de.itemis.qimpress.showcase.pdm_simulator.be.domain.AttributeBinaryValue;
import de.itemis.qimpress.showcase.pdm_simulator.be.domain.AttributeDefinition;
import de.itemis.qimpress.showcase.pdm_simulator.be.domain.AttributeValue;
import de.itemis.qimpress.showcase.pdm_simulator.be.domain.Product;
import de.itemis.qimpress.showcase.pdm_simulator.be.domain.ProductGroup;
import de.itemis.qimpress.showcase.pdm_simulator.be.exceptions.ApplicationException;
import de.itemis.qimpress.showcase.pdm_simulator.be.exceptions.DaoException;

/**
 * @author Claudius Haecker
 */
public interface PDMManager {

    /**
     * provides a means to get all product groups in the PDM
     * @return the root product group
     */
    public ProductGroup getProductGroupRoot() throws ApplicationException;

    /**
     * Selects a product group by the given id from database.
     * @param productGroupID primary key or the product group
     * @return the product group object for the given id
     * @throws ApplicationException if an error occured that cannot be resolved in the business layer
     */
    public ProductGroup getProductGroupByID(String productGroupID) throws ApplicationException;

    /**
     * This service method gets a list with the products for the product group
     * (for the given ID) and the products for all child product groups in this
     * product group tree.
     * @param theProductGroupID
     * @return the product list for the given product group
     * @throws ApplicationException if an error occured that cannot be resolved in the business layer
     */
    public List<Product> getProducts(String theProductGroupID) throws ApplicationException;

    /**
     * @param productId
     * @return list of {@link AttributeValue AttributeValues} for {@link Product} with the given productId
     * @throws ApplicationException if an error occured that cannot be resolved in the business layer
     */
    public List<AttributeValue> getAttributes(String productId) throws ApplicationException;

    /**
     * Loads
     * {@link de.itemis.qimpress.showcase.pdm_simulator.be.domain.AttributeBinaryValue}
     * together with its 
     * {@link de.itemis.qimpress.showcase.pdm_simulator.be.domain.AttributeBinaryValue#getBinaryData() binary data}
     * for the given 
     * {@link de.itemis.qimpress.showcase.pdm_simulator.be.domain.AttributeBinaryValue#getAttributeBinaryValueId() id}.
     * 
     * @param attributeBinaryValueId      
     * {@link de.itemis.qimpress.showcase.pdm_simulator.be.domain.AttributeBinaryValue#getAttributeBinaryValueId() id}
     * to look {@link de.itemis.qimpress.showcase.pdm_simulator.be.domain.AttributeBinaryValue} for.
     * @return {@link de.itemis.qimpress.showcase.pdm_simulator.be.domain.AttributeBinaryValue} with the given id
     * and loaded 
     * {@link de.itemis.qimpress.showcase.pdm_simulator.be.domain.AttributeBinaryValue#getBinaryData() binary data}
     * or null if not found
     */
    public AttributeBinaryValue getAttributeBinaryValueById(String attributeBinaryValueId);
    
    /**
     * Loads {@link de.itemis.qimpress.showcase.pdm_simulator.be.domain.Product} and its corresponding parent
     * {@link de.itemis.qimpress.showcase.pdm_simulator.be.domain.Product#getProductGroup() product group} 
     * for the given {@link de.itemis.qimpress.showcase.pdm_simulator.be.domain.Product#getProductCode() product code}.
     * 
     * @param productCode 
     * {@link de.itemis.qimpress.showcase.pdm_simulator.be.domain.Product#getProductCode() product code} to look
     * {@link de.itemis.qimpress.showcase.pdm_simulator.be.domain.Product} for
     * @return {@link de.itemis.qimpress.showcase.pdm_simulator.be.domain.Product} with the given productCode
     * or null if not found
     * @throws ApplicationException if an error occured that cannot be resolved in the business layer
     */
    public Product getProductByCode(String productCode) throws ApplicationException;
    
    // methods for loading all objects for bulk processing
    
    /**
     * Gets a list of all stored product groups.
     * 
     * @return list of all product groups
     * @throws DaoException if data access results in an error
     */
    public List<ProductGroup> getProductGroups() throws DaoException;
    
    /**
     * Gets a list of all stored products.
     * 
     * @return list of all stored products
     * @throws DaoException if data access results in an error
     */
    public List<Product> getProducts() throws DaoException;
    
    /**
     * Loads all stored attribute values in a list for bulk processing.
     * 
     * @return a list of all stored attribute values
     * @throws DaoException if data access results in an error
     */
    public List<AttributeValue> getAttributeValues() throws DaoException;
    
    /**
     * Loads all stored attribute definitions in a list for bulk processing.
     * 
     * @return a list of all stored attribute definitions
     * @throws DaoException if data access results in an error
     */
    public List<AttributeDefinition> getAttributeDefinitions() throws DaoException;

}
