/*******************************************************************************
 * Copyright (c) 2008,2009 Q-ImPrESS consortium
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * This work was funded in the context of the Q-ImPrESS research project  
 * (FP7-215013) by the European Union under the Information and  
 * Communication Technologies priority of the Seventh Research Framework  
 * Programme.
 *
 * Contributors:
 *     itemis 	- initial API and implementation
 *******************************************************************************/
package de.itemis.qimpress.showcase.order_simulator.webservice;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;

import javax.naming.NamingException;

import static org.apache.commons.lang.Validate.*;

import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import org.jdom.Element;
import org.jdom.Namespace;
import org.springframework.ws.server.endpoint.AbstractJDomPayloadEndpoint;

import de.itemis.qimpress.showcase.order_simulator.be.exceptions.ApplicationException;
import de.itemis.qimpress.showcase.order_simulator.be.manager.OrderSimulatorManager;
import de.itemis.qimpress.showcase.order_simulator.be.manager.ServiceLocator;
import de.itemis.qimpress.showcase.order_simulator.be.manager.model.TimeParameters;
import de.itemis.qimpress.showcase.order_simulator.be.manager.model.TimeParameters.GeneratorTimeMode;

/**
 * @author Wladimir Safonov
 *
 */
public class OrderSimulatorJDomEndpoint extends AbstractJDomPayloadEndpoint {

    private static final Logger LOG = Logger.getLogger(OrderSimulatorJDomEndpoint.class);
    
    private static final String START_ORDER_SIMULATOR_REQUEST = "StartOrderSimulatorRequest";
    private static final String START_ORDER_SIMULATOR_RESPONSE = "StartOrderSimulatorResponse";
    private static final String STOP_ORDER_SIMULATOR_REQUEST = "StopOrderSimulatorRequest";
    private static final String STOP_ORDER_SIMULATOR_RESPONSE = "StopOrderSimulatorResponse";
    private static final String GET_ORDER_SIMULATOR_STATUS_REQUEST = "GetOrderSimulatorStatusRequest";
    private static final String GET_ORDER_SIMULATOR_STATUS_RESPONSE = "GetOrderSimulatorStatusResponse";
    
    private static final String STATUS_RUNNING = "RUNNING";
    private static final String STATUS_STOPPED = "STOPPED";
    
    private OrderSimulatorManager simulatorManager = ServiceLocator.getInstance().getOrderSimulatorManager();
    
    private Namespace namespace;
    private final DateFormat sdf1 = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
    private final DateFormat sdf2 = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSS");
    
    public OrderSimulatorJDomEndpoint() {
        super();
        
        namespace = Namespace.getNamespace("os", "http://www.itemis.com/order_simulator/schemas");
    }

    @Override
    protected Element invokeInternal(Element request) throws Exception {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">> invokeInternal()");
        }
        
        String rootElement = request.getName();
        if (START_ORDER_SIMULATOR_REQUEST.equals(rootElement)) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Detected root element: " + START_ORDER_SIMULATOR_REQUEST);
            }
            
            return processStartOrderSimulatorRequest(request);
        } else if (STOP_ORDER_SIMULATOR_REQUEST.equals(rootElement)) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Detected root element: " + STOP_ORDER_SIMULATOR_REQUEST);
            }
            
            return processStopOrderSimulatorRequest(request);
        } else if (GET_ORDER_SIMULATOR_STATUS_REQUEST.equals(rootElement)) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Detected root element: " + GET_ORDER_SIMULATOR_STATUS_REQUEST);
            }
            
            return processGetOrderSimulatorStatusRequest(request);
        }
        return null;
    }

    
    protected Element processStartOrderSimulatorRequest(Element request) {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">> processStartOrderSimulatorRequest()");
        }
        
        // pre-validate parameters
        notEmpty(request.getChildTextTrim("GeneratorTimeMode", namespace), "'GeneratorTimeMode' parameter is empty");
        
        // unmarshall input parameters
        TimeParameters timeParameters = new TimeParameters();
        GeneratorTimeMode timeMode = null;
        String timeModeString = request.getChildTextTrim("GeneratorTimeMode", namespace);
        try {
            timeMode = GeneratorTimeMode.valueOf(timeModeString);
        } catch (Exception e) {
            String errMsg = "'GeneratorTimeMode' parameter has invalid value: " + timeModeString;
            LOG.error(errMsg);
            throw new IllegalArgumentException(errMsg);
        }
        Date startTimeForOrders = null;
        String startTimeString = request.getChildTextTrim("StartTimeForOrders", namespace);
        if (StringUtils.isNotEmpty(startTimeString)) {
            try {
                startTimeForOrders = sdf1.parse(startTimeString);
            } catch (ParseException e) {
                // try next format
                try {
                    startTimeForOrders = sdf2.parse(startTimeString);
                } catch (ParseException e1) {
                    String errMsg = "'StartTimeForOrders' parameter has wrong format: " + startTimeString;
                    LOG.error(errMsg);
                    throw new IllegalArgumentException(errMsg);
                }
            }
        }
        float timeScaleFactor = 0.0f;
        String timeScaleString = request.getChildTextTrim("TimeScaleFactor", namespace);
        if (StringUtils.isNotEmpty(timeScaleString)) {
            try {
                timeScaleFactor = Float.parseFloat(timeScaleString);
            } catch (NumberFormatException e) {
                String errMsg = "'TimeScaleFactor' parameter has invalid value: " + timeScaleString;
                LOG.error(errMsg);
                throw new IllegalArgumentException(errMsg);
            }
        }
        timeParameters.setGeneratorTimeMode(timeMode);
        timeParameters.setStartTimeForOrders(startTimeForOrders);
        timeParameters.setTimeScaleFactor(timeScaleFactor);
        
        // validate TimeParameters
        if (GeneratorTimeMode.SIMULATED_TIME == timeParameters.getGeneratorTimeMode()) {
            notNull(timeParameters.getStartTimeForOrders(), "'StartTimeForOrders' can not be empty for SIMULATED_TIME mode");
            isTrue(timeParameters.getTimeScaleFactor() != 0.0f, "'TimeScaleFactor' can not be empty for SIMULATED_TIME mode");
        }
        
        // start order generator
        boolean isStarted = simulatorManager.startOrderGenerator(timeParameters);
        
        // create response element
        Element response = new Element(START_ORDER_SIMULATOR_RESPONSE, namespace);
        Element result = new Element("StartResult", namespace).setText(Boolean.toString(isStarted));
        response.addContent(result);
        
        return response;
    }
    
    protected Element processStopOrderSimulatorRequest(Element request) {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">> processStopOrderSimulatorRequest()");
        }

        // no input parameters => process request directly
        
        // stop order generator
        boolean isStopped = false;
        try {
            isStopped = simulatorManager.stopOrderGenerator();
        } catch (ApplicationException e) {
            LOG.error("Error occured while trying to stop the order simulator", e);
        }
        
        // create response element
        Element response = new Element(STOP_ORDER_SIMULATOR_RESPONSE, namespace);
        Element result = new Element("StopResult", namespace).setText(Boolean.toString(isStopped));
        response.addContent(result);
        
        return response;
    }
    
    protected Element processGetOrderSimulatorStatusRequest(Element request) {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">> processGetOrderSimulatorStatusRequest()");
        }
        
        // no input parameter => process request directly
        
        // check order generator status
        boolean isRunning = simulatorManager.isOrderGeneratorRunning();
        
        // create response element
        Element response = new Element(GET_ORDER_SIMULATOR_STATUS_RESPONSE, namespace);
        Element result = new Element("StatusResult", namespace).setText(isRunning ? STATUS_RUNNING : STATUS_STOPPED);
        response.addContent(result);
        
        return response;
    }
}
