/*******************************************************************************
 * Copyright (c) 2008,2009 Q-ImPrESS consortium
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * This work was funded in the context of the Q-ImPrESS research project  
 * (FP7-215013) by the European Union under the Information and  
 * Communication Technologies priority of the Seventh Research Framework  
 * Programme.
 *
 * Contributors:
 *     itemis 	- initial API and implementation
 *******************************************************************************/
package de.itemis.qimpress.showcase.order_simulator.tests.be.services;

import static org.junit.Assert.*;

import java.math.BigDecimal;
import java.util.LinkedList;
import java.util.List;

import org.apache.log4j.Logger;
import org.junit.BeforeClass;
import org.junit.Test;
import org.springframework.context.ApplicationContext;
import org.springframework.context.support.ClassPathXmlApplicationContext;

import de.itemis.qimpress.showcase.order_simulator.be.domain.Order;
import de.itemis.qimpress.showcase.order_simulator.be.domain.OrderItem;
import de.itemis.qimpress.showcase.order_simulator.be.dto.Country;
import de.itemis.qimpress.showcase.order_simulator.be.dto.Customer;
import de.itemis.qimpress.showcase.order_simulator.be.dto.CustomerCategory;
import de.itemis.qimpress.showcase.order_simulator.be.dto.ProductInfo;
import de.itemis.qimpress.showcase.order_simulator.be.exceptions.ApplicationException;
import de.itemis.qimpress.showcase.order_simulator.be.webservices.ServiceManager;

/**
 * JUnit test for the the 
 * {@link de.itemis.qimpress.showcase.order_simulator.be.webservices.ServiceManager} 
 * service methods. As the tested methods call WebServices, it is required that
 * the corresponding WebServices are up and running before these tests get started.
 * 
 * 
 * @author Wladimir Safonov
 *
 */
public class ServiceManagerTest {
    
    private static final Logger LOG = Logger.getLogger(ServiceManagerTest.class);
    
    private static ServiceManager SERVICE_MANAGER;
    
    @BeforeClass
    public static void initTestSuite() {
        if (LOG.isDebugEnabled()) {
            LOG.debug("called initTestSuite()");
        }
        
        ApplicationContext ctxt = new ClassPathXmlApplicationContext("applicationContext.xml");
        assertNotNull("Failed to load context", ctxt);
        SERVICE_MANAGER = (ServiceManager) ctxt.getBean("serviceManager");
        assertNotNull("Failed to load ServiceManager", SERVICE_MANAGER);
    }

    @Test
    public void testGetRandomCustomerNumber() {
        if (LOG.isDebugEnabled()) {
            LOG.debug("called testGetRandomCustomerNumber()");
        }
        
        String customerNumber = null;
        try {
            customerNumber = SERVICE_MANAGER.getRandomCustomerNumber();
        } catch (ApplicationException e) {
            e.printStackTrace();
            fail("Failed to invoke CRM.GetRandomCustomerNumber");
        }
        assertNotNull("Received null instead of customer number", customerNumber);
        assertTrue("Received empty customer number", customerNumber.length() > 0);
    }
    
    @Test
    public void testGetCustomerByNumber() {
        if (LOG.isDebugEnabled()) {
            LOG.debug("called testGetCustomerByNumber()");
        }

        Customer customer = null;
        try {
            customer = SERVICE_MANAGER.getCustomerByNumber("000200401");
        } catch (ApplicationException e) {
            e.printStackTrace();
            fail("Failed to invoke CRM.GetCustomerByNumber");
        }
        assertNotNull("Received null instead of customer", customer);
        assertEquals("Received Customer with unexpected name", "Genesis Computer Services Ltd", customer.getName());
        
        try {
            customer = SERVICE_MANAGER.getCustomerByNumber("NON-EXISTING");
        } catch (ApplicationException e) {
            e.printStackTrace();
            fail("Failed to invoke CRM.GetCustomerByNumber");
        }
        assertNull("Received unexpected result for non-existing customer", customer);
    }
    
    @Test
    public void testGetRandomProductCode() {
        if (LOG.isDebugEnabled()) {
            LOG.debug("called testGetRandomProductCode()");
        }
        
        String productCode = null;
        try {
            productCode = SERVICE_MANAGER.getRandomProductCode();
        } catch (ApplicationException e) {
            e.printStackTrace();
            fail("Failed to invoke PDM.GetRandomProductCode");
        }
        assertNotNull("Received null instead of product code", productCode);
        assertTrue("Received empty product code", productCode.length() > 0);
    }

    @Test
    public void testGetProductInfoByCode() {
        if (LOG.isDebugEnabled()) {
            LOG.debug("called testGetProductInfoByCode()");
        }
        
        ProductInfo productInfo = null;
        try {
            productInfo = SERVICE_MANAGER.getProductInfoByCode("COAT005");
        } catch (ApplicationException e) {
            e.printStackTrace();
            fail("Failed to invoke PDM.GetProductInfo");
        }
        assertNotNull("Received null instead of ProductInfo", productInfo);
        assertEquals("Received ProductInfo with unexpected product name", "ATI Radeon HD4870 X2", productInfo.getProductName());
        assertEquals("Received ProductInfo with unexpected product group name", "Graphic cards", productInfo.getProductGroupName());
        
        try {
            productInfo = SERVICE_MANAGER.getProductInfoByCode("NON-EXISTING");
        } catch (ApplicationException e) {
            e.printStackTrace();
            fail("Failed to invoke PDM.GetProductInfo for non-existing product");
        }
        assertNull("Received unexpected result for non-existing product", productInfo);
    }
    
    @Test
    public void testPriceOrder() {
        if (LOG.isDebugEnabled()) {
            LOG.debug("called testPriceOrder()");
        }

        Customer customer = new Customer();
        customer.setInvoiceAddressCountry(new Country("NAMIBIA", "NA"));
        customer.setCustomerCategory(new CustomerCategory("important customer", "B"));
        Order order = new Order("OR0001", null, customer);

        Order pricedOrder = null;
        try {
            pricedOrder = SERVICE_MANAGER.priceOrder(order);
        } catch (ApplicationException e) {
            e.printStackTrace();
            fail("Failed to invoke PriceOrder");
        }
        assertNotNull("Unexpected result from PriceOrder: null", pricedOrder);
        assertEquals("Unexpected result from PriceOrder: totalPrice is not 0", new BigDecimal(0.0), pricedOrder.getTotalPrice());
        
        OrderItem orderItem = null;
        List<OrderItem> items = new LinkedList<OrderItem>();
        order.setOrderItems(items);
        orderItem = new OrderItem(25, new ProductInfo("COUL005"));
        items.add(orderItem);
        orderItem = new OrderItem(150, new ProductInfo("IDLT010"));
        items.add(orderItem);
        try {
            pricedOrder = SERVICE_MANAGER.priceOrder(order);
        } catch (ApplicationException e) {
            e.printStackTrace();
            fail("Failed to invoke PriceOrder");
        }
        assertNotNull("Unexpected result from PriceOrder: null", pricedOrder);
        assertEquals("Unexpected result from PriceOrder: total price is wrong", 
                new BigDecimal("2171.71"), pricedOrder.getTotalPrice());
        assertEquals("Unexpected result from PriceOrder: total price for item1 is wrong",
                new BigDecimal("223.59"), pricedOrder.getOrderItems().get(0).getTotalPrice());
        assertEquals("Unexpected result from PriceOrder: total price for item2 is wrong",
                new BigDecimal("1948.12"), pricedOrder.getOrderItems().get(1).getTotalPrice());
    }
}
