/*******************************************************************************
 * Copyright (c) 2008,2009 Q-ImPrESS consortium
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * This work was funded in the context of the Q-ImPrESS research project  
 * (FP7-215013) by the European Union under the Information and  
 * Communication Technologies priority of the Seventh Research Framework  
 * Programme.
 *
 * Contributors:
 *     itemis 	- initial API and implementation
 *******************************************************************************/
package de.itemis.qimpress.showcase.order_simulator.be.webservices;

import java.io.IOException;
import java.util.List;
import java.util.Properties;

import org.apache.log4j.Logger;
import org.springframework.ws.client.core.support.WebServiceGatewaySupport;

import de.itemis.qimpress.showcase.order_simulator.be.domain.Order;
import de.itemis.qimpress.showcase.order_simulator.be.dto.Customer;
import de.itemis.qimpress.showcase.order_simulator.be.dto.ProductInfo;
import de.itemis.qimpress.showcase.order_simulator.be.exceptions.ApplicationException;
import de.itemis.qimpress.showcase.order_simulator.be.exceptions.ApplicationException.ApplicationErrorCode;
import de.itemis.qimpress.showcase.order_simulator.be.webservices.messages.CustomerFilter;
import de.itemis.qimpress.showcase.order_simulator.be.webservices.messages.GetProductInfoRequest;
import de.itemis.qimpress.showcase.order_simulator.be.webservices.messages.GetProductInfoResponse;
import de.itemis.qimpress.showcase.order_simulator.be.webservices.messages.GetRandomCustomerNumberRequest;
import de.itemis.qimpress.showcase.order_simulator.be.webservices.messages.GetRandomCustomerNumberResponse;
import de.itemis.qimpress.showcase.order_simulator.be.webservices.messages.GetRandomProductCodeRequest;
import de.itemis.qimpress.showcase.order_simulator.be.webservices.messages.GetRandomProductCodeResponse;
import de.itemis.qimpress.showcase.order_simulator.be.webservices.messages.PriceOrderRequest;
import de.itemis.qimpress.showcase.order_simulator.be.webservices.messages.PriceOrderResponse;
import de.itemis.qimpress.showcase.order_simulator.be.webservices.messages.QueryCustomersRequest;
import de.itemis.qimpress.showcase.order_simulator.be.webservices.messages.QueryCustomersResponse;

/**
 * Implements {@link de.itemis.qimpress.showcase.order_simulator.be.webservices.ServiceManager}
 * relying on Spring WS support for WebService clients and using automatic 
 * XML un-/marshalling for payloads.
 * 
 * @author Wladimir Safonov
 *
 */
public class ServiceManagerImpl extends WebServiceGatewaySupport implements ServiceManager {

    private static final Logger LOG = Logger.getLogger(ServiceManagerImpl.class);

    private static final String CRM_WS_URI = "crm_ws_uri";
    private static final String PDM_WS_URI = "pdm_ws_uri";
    private static final String PRICING_WS_URI = "pricing_ws_uri";

    private Properties webServiceURIs;

    public ServiceManagerImpl() throws ApplicationException {
        webServiceURIs = new Properties();
        try {
            webServiceURIs.load(this.getClass().getClassLoader().getResourceAsStream("webservice.properties"));
        } catch (IOException e) {
            LOG.error("Failed to load webservice.properties file", e);
            throw new ApplicationException(ApplicationErrorCode.TE, e);
        }
    }

    /* (non-Javadoc)
     * @see de.itemis.qimpress.showcase.order_simulator.be.webservices.ServiceManager#getRandomCustomerNumber()
     */
    /* @Override */
    public String getRandomCustomerNumber() throws ApplicationException {
        if (LOG.isDebugEnabled()) {
            LOG.debug("called getRandomCustomerNumber()");
        }

        // create request object and invoke web service
        GetRandomCustomerNumberRequest request = new GetRandomCustomerNumberRequest();
        GetRandomCustomerNumberResponse response = (GetRandomCustomerNumberResponse) getWebServiceTemplate()
                .marshalSendAndReceive(webServiceURIs.getProperty(CRM_WS_URI), request);

        return response.getCustomerNumber();
    }

    /* @Override */
    public Customer getCustomerByNumber(String customerNumber) throws ApplicationException {
        if (LOG.isDebugEnabled()) {
            LOG.debug("called getCustomerByNumber()");
        }

        // create request object
        QueryCustomersRequest request = new QueryCustomersRequest();
        CustomerFilter customerFilter = new CustomerFilter();
        customerFilter.setCustomerNumberPattern(customerNumber);
        request.setCustomerFilter(customerFilter);

        // invoke web service
        QueryCustomersResponse response = (QueryCustomersResponse) getWebServiceTemplate().marshalSendAndReceive(
                webServiceURIs.getProperty(CRM_WS_URI), request);

        // process response
        List<Customer> customers = response.getCustomers();
        if (customers == null || customers.size() == 0) {
            return null;
        }
        if (customers.size() > 1) {
            LOG.error("Received unexpected number of Customers from WebService");
            throw new ApplicationException(ApplicationErrorCode.TE);
        }

        // return Customer
        return customers.get(0);
    }

    /* @Override */
    public String getRandomProductCode() throws ApplicationException {
        if (LOG.isDebugEnabled()) {
            LOG.debug("called getCustomerByNumber()");
        }

        // create request object and invoke web service
        GetRandomProductCodeRequest request = new GetRandomProductCodeRequest();
        GetRandomProductCodeResponse response = (GetRandomProductCodeResponse) getWebServiceTemplate()
                .marshalSendAndReceive(webServiceURIs.getProperty(PDM_WS_URI), request);

        return response.getProductCode();
    }

    /* @Override */
    public ProductInfo getProductInfoByCode(String productCode) throws ApplicationException {
        if (LOG.isDebugEnabled()) {
            LOG.debug("called getProductInfoByCode()");
        }

        // create request object
        GetProductInfoRequest request = new GetProductInfoRequest();
        request.setProductCode(productCode);

        // invoke web service
        GetProductInfoResponse response = (GetProductInfoResponse) getWebServiceTemplate().marshalSendAndReceive(
                webServiceURIs.getProperty(PDM_WS_URI), request);

        return response.getProductInfo();
    }

    /* @Override */
    public Order priceOrder(Order unpricedOrder) throws ApplicationException {
        if (LOG.isDebugEnabled()) {
            LOG.debug("called priceOrder()");
        }

        // create request object 
        PriceOrderRequest request = new PriceOrderRequest();
        request.setOrder(unpricedOrder);

        // invoke web service
        PriceOrderResponse response = (PriceOrderResponse) getWebServiceTemplate().marshalSendAndReceive(
                webServiceURIs.getProperty(PRICING_WS_URI), request);

        return response.getPricedOrder();
    }

}
