/*******************************************************************************
 * Copyright (c) 2008,2009 Q-ImPrESS consortium
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * This work was funded in the context of the Q-ImPrESS research project  
 * (FP7-215013) by the European Union under the Information and  
 * Communication Technologies priority of the Seventh Research Framework  
 * Programme.
 *
 * Contributors:
 *     itemis 	- initial API and implementation
 *******************************************************************************/
package de.itemis.qimpress.showcase.order_simulator.be.webservices;

import de.itemis.qimpress.showcase.order_simulator.be.domain.Order;
import de.itemis.qimpress.showcase.order_simulator.be.dto.Customer;
import de.itemis.qimpress.showcase.order_simulator.be.dto.ProductInfo;
import de.itemis.qimpress.showcase.order_simulator.be.exceptions.ApplicationException;

/**
 * <p>
 * Service interface for the functions of CRM, PDM and Pricing systems 
 * relevant to the Order Shipment OrderGenerator. 
 * <p>
 * Allows to request random customer number and product code, to retrieve 
 * customer and product information by corresponding business identifier
 * as well as to price an order.
 * 
 * @author Wladimir Safonov
 *
 */
public interface ServiceManager {

    /**
     * Retrieves a random 
     * {@link de.itemis.qimpress.showcase.order_simulator.be.dto.Customer#getCustomerNumber() customer number}
     * from the CRM system.
     * 
     * @return random 
     * {@link de.itemis.qimpress.showcase.order_simulator.be.dto.Customer#getCustomerNumber() customer number}
     */
    public String getRandomCustomerNumber() throws ApplicationException;
    
    /**
     * Retrieves
     * {@link de.itemis.qimpress.showcase.order_simulator.be.dto.Customer}
     * from the CRM system by its
     * {@link de.itemis.qimpress.showcase.order_simulator.be.dto.Customer#getCustomerNumber() number}.
     * As the underlying CRM WebService method potentially returns a list of Customers,
     * this service method assumes unique result for the query by customer number and will throw an 
     * {@link de.itemis.qimpress.showcase.order_simulator.be.exceptions.ApplicationException}
     * if the customer result returned from the WebService is not unique.
     * 
     * @param customerNumber
     * {@link de.itemis.qimpress.showcase.order_simulator.be.dto.Customer#getCustomerNumber() customer number}
     * to look up 
     * {@link de.itemis.qimpress.showcase.order_simulator.be.dto.Customer} for
     * @return {@link de.itemis.qimpress.showcase.order_simulator.be.dto.Customer} with
     * the given customer number or null if not found
     * @throws ApplicationException if an error occured that cannot be resolved in the business layer
     */
    public Customer getCustomerByNumber(String customerNumber) throws ApplicationException;
    
    /**
     * Retrieves a random
     * {@link de.itemis.qimpress.showcase.order_simulator.be.dto.ProductInfo#getProductCode() product code}
     * from the PDM system.
     * 
     * @return random 
     * {@link de.itemis.qimpress.showcase.order_simulator.be.dto.ProductInfo#getProductCode() product code}
     * @throws ApplicationException if an error occured that cannot be resolved in the business layer
     */
    public String getRandomProductCode() throws ApplicationException;
    
    /**
     * Retrieves
     * {@link de.itemis.qimpress.showcase.order_simulator.be.dto.ProductInfo}
     * from the PDM system by the
     * {@link de.itemis.qimpress.showcase.order_simulator.be.dto.ProductInfo#getProductCode() product code}.
     * 
     * @param productCode 
     * {@link de.itemis.qimpress.showcase.order_simulator.be.dto.ProductInfo#getProductCode() product code}
     * to look up
     * {@link de.itemis.qimpress.showcase.order_simulator.be.dto.ProductInfo} for
     * @return 
     * {@link de.itemis.qimpress.showcase.order_simulator.be.dto.ProductInfo} 
     * with the given product code or null if not found.
     * @throws ApplicationException if an error occured that cannot be resolved in the business layer
     */
    public ProductInfo getProductInfoByCode(String productCode) throws ApplicationException;
    
    /**
     * Fills the given 
     * {@link de.itemis.qimpress.showcase.order_simulator.be.domain.Order}
     * with price information. Calculates
     * {@link de.itemis.qimpress.showcase.order_simulator.be.domain.OrderItem#getTotalPrice() price}
     * for each order item and 
     * {@link de.itemis.qimpress.showcase.order_simulator.be.domain.Order#getTotalPrice() total price}
     * for the order.
     * 
     * @param unpricedOrder
     * {@link de.itemis.qimpress.showcase.order_simulator.be.domain.Order} without price information
     * @return
     * {@link de.itemis.qimpress.showcase.order_simulator.be.domain.Order} filled with price information
     * @throws ApplicationException if an error occured that cannot be resolved in the business layer
     */
    public Order priceOrder(Order unpricedOrder) throws ApplicationException;
}
