/*******************************************************************************
 * Copyright (c) 2008,2009 Q-ImPrESS consortium
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * This work was funded in the context of the Q-ImPrESS research project  
 * (FP7-215013) by the European Union under the Information and  
 * Communication Technologies priority of the Seventh Research Framework  
 * Programme.
 *
 * Contributors:
 *     itemis 	- initial API and implementation
 *******************************************************************************/
package de.itemis.qimpress.showcase.order_simulator.be.manager;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.jdbc.core.support.JdbcDaoSupport;

import de.itemis.qimpress.showcase.order_simulator.be.dao.OrderDao;
import de.itemis.qimpress.showcase.order_simulator.be.domain.Order;
import de.itemis.qimpress.showcase.order_simulator.be.exceptions.ApplicationException;
import de.itemis.qimpress.showcase.order_simulator.be.exceptions.DaoException;
import de.itemis.qimpress.showcase.order_simulator.be.exceptions.ApplicationException.ApplicationErrorCode;
import de.itemis.qimpress.showcase.order_simulator.be.manager.reports.SalesPerCustomerSummary;

/**
 * @author Wladimir Safonov
 *
 */
public class OrderReportingManagerImpl extends JdbcDaoSupport implements OrderReportingManager {

    private static final Logger LOG = Logger.getLogger(OrderReportingManagerImpl.class);

    private static final String SALES_SUMMARY_SQL = "SELECT customer_number, additional_info1, additional_info2, additional_info3, "
            + "COUNT(*) as orders_number, SUM(total_price) as sales_volume "
            + "FROM order_ "
            + "WHERE creation_date>=? AND creation_date<=? AND additional_info2 LIKE ? AND additional_info3 LIKE ? "
            + "GROUP BY customer_number, additional_info1, additional_info2, additional_info3 "
            + "ORDER BY customer_number";

    private static final RowMapper SALES_SUMMARY_ROW_MAPPER = new RowMapper() {

        /* @Override */
        public Object mapRow(ResultSet rs, int rowNum) throws SQLException {
            SalesPerCustomerSummary summary = new SalesPerCustomerSummary();
            summary.setCustomerNumber(rs.getString(1));
            summary.setCustomerName(rs.getString(2));
            summary.setCustomerCategory(rs.getString(3));
            summary.setCustomerType(rs.getString(4));
            summary.setOrdersNumber(rs.getInt(5));
            summary.setSalesVolume(rs.getBigDecimal(6));
            return summary;
        }

    };

    private OrderDao orderDao;

    /**
     * @param orderDao the orderDao to set
     */
    public void setOrderDao(OrderDao orderDao) {
        this.orderDao = orderDao;
    }

    /* (non-Javadoc)
     * @see de.itemis.qimpress.showcase.order_simulator.be.manager.OrderReportingManager#calculateSalesPerCustomer(java.util.Date, java.util.Date, java.lang.String, java.lang.String)
     */
    /* @Override */
    @SuppressWarnings("unchecked")
    public List<SalesPerCustomerSummary> calculateSalesPerCustomer(Date from, Date till, String customerCategory,
            String customerType) {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">> calculateSalesPerCustomer()");
        }

        // prepare dates interval
        Date beginningDate = getMidnight(from), endingDate = getLastSecond(till);
        // prepare customer category and type
        String customerCategoryParam = StringUtils.isEmpty(customerCategory) ? "%" : "%" + customerCategory + "%";
        String customerTypeParam = StringUtils.isEmpty(customerType) ? "%" : "%" + customerType + "%";

        List<SalesPerCustomerSummary> salesSummaryList = getJdbcTemplate().query(SALES_SUMMARY_SQL,
                new Object[] { beginningDate, endingDate, customerCategoryParam, customerTypeParam },
                SALES_SUMMARY_ROW_MAPPER);

        return salesSummaryList;
    }

    /**
     * Calculates midnight time for the given date.
     * 
     * @param date
     * @return
     */
    private Date getMidnight(Date date) {
        Calendar calend = Calendar.getInstance();
        calend.setTime(date);
        calend.set(Calendar.HOUR_OF_DAY, 0);
        calend.set(Calendar.MINUTE, 0);
        calend.set(Calendar.SECOND, 0);
        calend.set(Calendar.MILLISECOND, 0);
        return calend.getTime();
    }

    /**
     * Sets time to the last second for the given date.
     * 
     * @param date
     * @return
     */
    private Date getLastSecond(Date date) {
        Calendar calend = Calendar.getInstance();
        calend.setTime(getMidnight(date));
        calend.add(Calendar.DATE, 1);
        calend.add(Calendar.SECOND, -1);
        return calend.getTime();
    }

    /* @Override */
    public List<Order> getOrdersForCustomerAndInterval(String customerNumber, Date from, Date till)
            throws ApplicationException {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">> getOrdersForCustomerAndInterval()");
        }

        // prepare dates interval
        Date beginningDate = getMidnight(from), endingDate = getLastSecond(till);
        List<Order> ordersForCustomerAndInterval = null;
        try {
            ordersForCustomerAndInterval = orderDao.getOrdersForCustomerAndInterval(customerNumber, beginningDate,
                    endingDate);
        } catch (DaoException e) {
            LOG.error("Failed to load Orders", e);
            throw new ApplicationException(ApplicationErrorCode.TE, e);
        }
        return ordersForCustomerAndInterval;
    }
}
