/*******************************************************************************
 * Copyright (c) 2008,2009 Q-ImPrESS consortium
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * This work was funded in the context of the Q-ImPrESS research project  
 * (FP7-215013) by the European Union under the Information and  
 * Communication Technologies priority of the Seventh Research Framework  
 * Programme.
 *
 * Contributors:
 *     itemis 	- initial API and implementation
 *******************************************************************************/
package de.itemis.qimpress.showcase.order_simulator.be.generator.order_processing;

import java.util.LinkedList;
import java.util.List;

import org.apache.log4j.Logger;
import org.jdom.output.Format;
import org.jdom.output.XMLOutputter;
import org.jdom.transform.JDOMResult;
import org.springframework.jms.JmsException;
import org.springframework.jms.core.JmsTemplate;
import org.springframework.oxm.Marshaller;

import de.itemis.qimpress.showcase.order_simulator.be.domain.Order;
import de.itemis.qimpress.showcase.order_simulator.be.dto.OrdersBatch;
import de.itemis.qimpress.showcase.order_simulator.be.exceptions.ApplicationException;
import de.itemis.qimpress.showcase.order_simulator.be.exceptions.ApplicationException.ApplicationErrorCode;
import de.itemis.qimpress.showcase.order_simulator.be.manager.model.GlobalParameters;

/**
 * JMS sender of generated orders. Collects generated orders and sends them 
 * in a single batch to the configured JMS topic after reaching certain 
 * threshold of orders number @see {@link GlobalParameters#getOrdersBatchSize()}.
 * 
 * @author Wladimir Safonov
 *
 */
public class OrdersJmsSender implements BufferedOrderProcessor {

    private static final Logger LOG = Logger.getLogger(OrdersJmsSender.class);

    private List<Order> collectedOrders;
    private JmsTemplate jmsTemplate;
    private Marshaller marshaller;
    
    /**
     * @param jmsTemplate the jmsTemplate to set
     */
    public void setJmsTemplate(JmsTemplate jmsTemplate) {
        this.jmsTemplate = jmsTemplate;
    }

    /**
     * @param marshaller the marshaller to set
     */
    public void setMarshaller(Marshaller marshaller) {
        this.marshaller = marshaller;
    }

    public OrdersJmsSender() {
        this.collectedOrders = new LinkedList<Order>();
    }

    /* (non-Javadoc)
     * @see de.itemis.qimpress.showcase.order_simulator.be.generator.order_processing.BufferedOrderProcessor#flush()
     */
    /* @Override */
    public synchronized void flush() throws ApplicationException {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">> flush()");
        }
        
        // skip empty batch
        if (collectedOrders.size() == 0) {
            return;
        }
        
        // create OrdersBatch
        OrdersBatch ordersBatch = new OrdersBatch();
        ordersBatch.setOrders(this.collectedOrders);
        
        // marshall OrdersBatch to XML
        JDOMResult jDomResult = new JDOMResult();
        try {
            marshaller.marshal(ordersBatch, jDomResult);
        } catch (Exception e) {
            LOG.error("Failed to marshall OrdersBatch to XML", e);
            LOG.error("Failed to marshall orders with numbers: '" + 
                    collectedOrders.get(0).getNumber() + "' - '" +
                    collectedOrders.get(collectedOrders.size() - 1).getNumber() + "'");
            emptyOrdersBuffer();
            throw new ApplicationException(ApplicationErrorCode.TE, e);
        }
        
        // output XML to String
        XMLOutputter xmlOutputter = new XMLOutputter(Format.getCompactFormat());
        String ordersBatchMessage = xmlOutputter.outputString(jDomResult.getDocument());
        
        // send message to configured JMS topic
        try {
            jmsTemplate.convertAndSend(ordersBatchMessage);
        } catch (JmsException e) {
            LOG.error("Failed to send OrdersBatch to JMS topic", e);
            LOG.error("Failed to send OrdersBatch to JMS topic. Orders with numbers: '" + 
                    collectedOrders.get(0).getNumber() + "' - '" +
                    collectedOrders.get(collectedOrders.size() - 1).getNumber() + "'");
            emptyOrdersBuffer();
            throw new ApplicationException(ApplicationErrorCode.TE, e);
        }
        emptyOrdersBuffer();
    }

    /* (non-Javadoc)
     * @see de.itemis.qimpress.showcase.order_simulator.be.generator.order_processing.OrderProcessor#process(de.itemis.qimpress.showcase.order_simulator.be.domain.Order, de.itemis.qimpress.showcase.order_simulator.be.manager.model.GlobalParameters)
     */
    /* @Override */
    public void process(Order newOrder, GlobalParameters generatorParams) throws ApplicationException {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">> process()");
        }
        
        collectedOrders.add(newOrder);
        
        // check the flush condition
        if (collectedOrders.size() == generatorParams.getOrdersBatchSize()) {
            flush();
        }
    }

    private void emptyOrdersBuffer() {
        this.collectedOrders = new LinkedList<Order>();
    }
}
