/*******************************************************************************
 * Copyright (c) 2008,2009 Q-ImPrESS consortium
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * This work was funded in the context of the Q-ImPrESS research project  
 * (FP7-215013) by the European Union under the Information and  
 * Communication Technologies priority of the Seventh Research Framework  
 * Programme.
 *
 * Contributors:
 *     itemis 	- initial API and implementation
 *******************************************************************************/
package de.itemis.qimpress.showcase.order_simulator.be.generator.order_processing;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.LinkedList;

import org.apache.log4j.Logger;

import com.csvreader.CsvWriter;

import de.itemis.qimpress.showcase.order_simulator.be.domain.Order;
import de.itemis.qimpress.showcase.order_simulator.be.domain.OrderItem;
import de.itemis.qimpress.showcase.order_simulator.be.exceptions.ApplicationException;
import de.itemis.qimpress.showcase.order_simulator.be.exceptions.ApplicationException.ApplicationErrorCode;
import de.itemis.qimpress.showcase.order_simulator.be.manager.model.GlobalParameters;

/**
 * @author Claudius Haecker
 * @author Wladimir Safonov
 *
 */
public class OrdersCsvWriter implements BufferedOrderProcessor {
    // Logger for the OrdersCsvWriter
    private static final Logger LOG = Logger.getLogger(OrdersCsvWriter.class);

    private static final String FILENAME_1ST_PART = "Order-";
    private static final int ORDER_COUNTER_INITIAL = 0;
    private static final int FILE_COUNTER_INITIAL = 0;

    private int fileCounter = FILE_COUNTER_INITIAL;
    private int orderCounter = ORDER_COUNTER_INITIAL;
    private String dateString = null;
    private LinkedList<String[]> lines = new LinkedList<String[]>();

    private static final DateFormat DATE_FORMAT = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
    
    /** instance of GlobalParameters used for the last processed order */
    private GlobalParameters generatorParameters;

    /** last order file saved */
    private File lastSavedFile;
    
    /**
     * @return the lastSavedFile
     */
    public File getLastSavedFile() {
        return lastSavedFile;
    }

    /* (non-Javadoc)
     * @see de.itemis.qimpress.showcase.order_simulator.be.generator.OrdersBuffer#addOrder(de.itemis.qimpress.showcase.order_simulator.be.domain.Order)
     */
    /* @Override */
    public void process(Order newOrder, GlobalParameters generatorParameters) throws ApplicationException {
        
        // save generator parameters to use later in case of external flush 
        this.generatorParameters = generatorParameters;
        
        orderCounter++;
        for (OrderItem item : newOrder.getOrderItems()) {
            lines.add(new String[] { newOrder.getNumber(), DATE_FORMAT.format(newOrder.getCreationDate()), 
                    newOrder.getCustomer().getCustomerNumber(), item.getProduct().getProductCode(), 
                    item.getProduct().getProductName(), Integer.toString(item.getQuantity()), 
                    item.getTotalPrice().toString()});
        }
        
        // write current orders to CSV file if items-per-file bound is reached
        if (orderCounter >= generatorParameters.getOrdersBatchSize()) {
            writeCsvFile(new File(generatorParameters.getOutputFolder()));
        }
    }

    /**
     * Write collected orders to a CSV file in the given directory.
     * CSV file name will be generated based on the current system time.
     * 
     * @param outputDir directory to write CSV file to
     * @throws ApplicationException 
     */
    protected void writeCsvFile(File outputDir) throws ApplicationException {
        checkDir(outputDir);
        CsvWriter writer = new CsvWriter(createNewFileWriter(outputDir), ';');
        lines.add(0, new String[] { "ORDER_NUMBER", "ORDER_DATE", "CUSTOMER_NUMBER", "PRODUCT_CODE", "PRODUCT_NAME", "QUANTITY",
                "ITEM_TOTAL_PRICE" });
        for (String[] line : lines) {
            try {
                writer.writeRecord(line, true);
            } catch (IOException e) {
                LOG.error("Could not write to file.", e);
                throw new ApplicationException(ApplicationErrorCode.TE, e);
            }
        }
        writer.close();
        lines = new LinkedList<String[]>();
        fileCounter++;
        orderCounter = 0;
    }
    
    /* @Override */
    public void flush() throws ApplicationException {
        if (generatorParameters != null) {
            writeCsvFile(new File(generatorParameters.getOutputFolder()));
        }
    }

    private FileWriter createNewFileWriter(File outputDir) throws ApplicationException {
        if (dateString == null) {
            dateString = DateFormat.getDateTimeInstance().format(new Date());
        }
        lastSavedFile = new File(outputDir, (FILENAME_1ST_PART + dateString + "-" + fileCounter + ".csv").replaceAll(
                "[\\/:*?\"<>|]", "."));
        try {
            return new FileWriter(lastSavedFile);
        } catch (IOException e) {
            LOG.error("could not write to file " + lastSavedFile.getName() + " at path " + lastSavedFile.getPath() + ".", e);
            throw new ApplicationException(ApplicationErrorCode.TE, e);
        }
    }

    private void checkDir(File outputDir) {
        if (!outputDir.exists()) {
            if (!outputDir.mkdirs()) {
                throw new IllegalArgumentException("The given directory does not exist and can not be created: "
                        + outputDir.getAbsolutePath());
            }
        }
    }
}
