/*******************************************************************************
 * Copyright (c) 2008,2009 Q-ImPrESS consortium
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * This work was funded in the context of the Q-ImPrESS research project  
 * (FP7-215013) by the European Union under the Information and  
 * Communication Technologies priority of the Seventh Research Framework  
 * Programme.
 *
 * Contributors:
 *     itemis 	- initial API and implementation
 *******************************************************************************/
package de.itemis.qimpress.showcase.order_simulator.be.generator;

import java.util.Date;
import java.util.LinkedList;
import java.util.List;
import java.util.Random;

import org.apache.log4j.Logger;

import de.itemis.qimpress.showcase.order_simulator.be.domain.Order;
import de.itemis.qimpress.showcase.order_simulator.be.domain.OrderItem;
import de.itemis.qimpress.showcase.order_simulator.be.dto.Customer;
import de.itemis.qimpress.showcase.order_simulator.be.dto.ProductInfo;
import de.itemis.qimpress.showcase.order_simulator.be.exceptions.ApplicationException;
import de.itemis.qimpress.showcase.order_simulator.be.generator.order_processing.OrderProcessor;
import de.itemis.qimpress.showcase.order_simulator.be.manager.model.CustomerTypeParameters;
import de.itemis.qimpress.showcase.order_simulator.be.manager.model.GlobalParameters;
import de.itemis.qimpress.showcase.order_simulator.be.manager.model.TimeParameters;
import de.itemis.qimpress.showcase.order_simulator.be.webservices.ServiceManager;

/**
 * Order generation job. 
 * 
 * @author Wladimir Safonov
 */
public class OrderGeneratorJob {

    private static final Logger LOG = Logger.getLogger(OrderGeneratorJob.class);

    /** service manager for accessing CRM, PDM & Pricing web services */
    private ServiceManager serviceManager;
    /** list of order processors to receive the generated order */
    private List<OrderProcessor> orderProcessors;
    /** OrderNumberGenerator for generating numbers of new orders */
    private OrderNumberGenerator orderNumberGenerator;

    // randomizer
    private Random random = new Random();

    /**
     * @param serviceManager the serviceManager to set
     */
    public void setServiceManager(ServiceManager serviceManager) {
        this.serviceManager = serviceManager;
    }

    /**
     * @param orderProcessors the orderProcessors to set
     */
    public void setOrderProcessors(List<OrderProcessor> orderProcessors) {
        this.orderProcessors = orderProcessors;
    }

    /**
     * @param orderNumberGenerator the orderNumberGenerator to set
     */
    public void setOrderNumberGenerator(OrderNumberGenerator orderNumberGenerator) {
        this.orderNumberGenerator = orderNumberGenerator;
    }

    /**
     * Generates an order and dispatches it to the order handler.
     * 
     * @param params
     */
    public void execute(GlobalParameters params) {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">> execute()");
        }

        // check if an order should be generated according to its probability
        if (random.nextFloat() < params.getOrderProbability()) {

            try {

                // load random customer
                String customerNumber = serviceManager.getRandomCustomerNumber();
                Customer customer = serviceManager.getCustomerByNumber(customerNumber);
                if (customer == null) {
                    LOG.error("Received null instead of Customer for customerNumber='" + customerNumber + "'");
                    return;
                }
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Generating Order for Customer: " + customer.getCustomerNumber() + " ["
                            + customer.getName() + "]");
                }

                // get customer type specific parameters
                if (customer.getCustomerType() == null) {
                    LOG.error("Received null instead of Customer.CustomerType for customerNumber='" + customerNumber
                            + "'");
                    return;
                }
                String customerTypeName = customer.getCustomerType().getCustomerTypeName();
                if (customerTypeName == null) {
                    LOG.error("Received null instead of Customer.CustomerType.Name for customerNumber='"
                            + customerNumber + "'");
                    return;
                }
                CustomerTypeParameters customerTypeParameters = params.getTypeSpecificParameters()
                        .get(customerTypeName);
                if (customerTypeParameters == null) {
                    LOG.error("Found no CustomerTypeParameters for customerTypeName='" + customerTypeName + "'");
                    return;
                }

                // generate order
                Order newOrder = generateOrder(customer, customerTypeParameters);
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Generated Order with number: " + newOrder.getNumber());
                }

                // update order creation date
                TimeParameters timeParams = params.getTimeParameters();
                long generatorElapsedTimeMillis = (newOrder.getCreationDate().getTime() - timeParams
                        .getStartTimeOfGenerator().getTime());
                long scaledElapsedTimeMillis = (long)(generatorElapsedTimeMillis * timeParams.getTimeScaleFactor());
                Date simulatedOrderDate = new Date();
                simulatedOrderDate.setTime(timeParams.getStartTimeForOrders().getTime() + scaledElapsedTimeMillis);
                newOrder.setCreationDate(simulatedOrderDate);

                // process order with the given order processors
                for (OrderProcessor orderProcessor : orderProcessors) {
                    orderProcessor.process(newOrder, params);
                }

            } catch (ApplicationException e) {
                LOG.error("Failed to generate order", e);
            }
        }

    }

    private Order generateOrder(Customer customer, CustomerTypeParameters customerTypeParams)
            throws ApplicationException {

        // create order
        String orderNumber = orderNumberGenerator.generateNextOrderNumber();
        Order generatedOrder = new Order(orderNumber, new Date(), customer);

        // generate order items number
        int orderItemsNum = random.nextInt(customerTypeParams.getMaxItemsPerOrder()
                - customerTypeParams.getMinItemsPerOrder() + 1)
                + customerTypeParams.getMinItemsPerOrder();

        // fill order with items
        List<OrderItem> orderItems = new LinkedList<OrderItem>();
        for (int i = 0; i < orderItemsNum; i++) {

            // generate product quantity for this item
            int quantity = random.nextInt(customerTypeParams.getMaxQuantityPerItem()
                    - customerTypeParams.getMinQuantityPerItem() + 1)
                    + customerTypeParams.getMinQuantityPerItem();

            // load random product
            String productCode = serviceManager.getRandomProductCode();
            ProductInfo product = serviceManager.getProductInfoByCode(productCode);

            // create order item
            OrderItem orderItem = new OrderItem(quantity, product);
            orderItems.add(orderItem);

        }
        generatedOrder.setOrderItems(orderItems);

        // price order
        generatedOrder = serviceManager.priceOrder(generatedOrder);
        return generatedOrder;
    }

}