/*******************************************************************************
 * Copyright (c) 2008,2009 Q-ImPrESS consortium
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * This work was funded in the context of the Q-ImPrESS research project  
 * (FP7-215013) by the European Union under the Information and  
 * Communication Technologies priority of the Seventh Research Framework  
 * Programme.
 *
 * Contributors:
 *     itemis 	- initial API and implementation
 *******************************************************************************/
package de.itemis.qimpress.showcase.order_simulator.be.generator;

import java.util.Date;

import org.apache.log4j.Logger;
import org.quartz.JobDetail;
import org.quartz.Scheduler;
import org.quartz.SchedulerException;
import org.quartz.SimpleTrigger;

import de.itemis.qimpress.showcase.order_simulator.be.exceptions.ApplicationException;
import de.itemis.qimpress.showcase.order_simulator.be.generator.order_processing.BufferedOrderProcessor;
import de.itemis.qimpress.showcase.order_simulator.be.manager.model.GlobalParameters;
import de.itemis.qimpress.showcase.order_simulator.be.manager.model.TimeParameters.GeneratorTimeMode;

/**
 * @author Wladimir Safonov
 *
 */
public class GeneratorManagerImpl implements GeneratorManager {

    private static final Logger LOG = Logger.getLogger(GeneratorManagerImpl.class);
    
    /** SchedulerFactoryBean configured with the OrderGeneratorJob */
    private Scheduler scheduler;
    /** SimpleTriggerBean configured with a time interval for the OrderGeneratorJob */
    private SimpleTrigger trigger;
    /** JobDetail to access and update JobDataMap */
    private JobDetail jobDetail;
    /** BufferedOrderProcessor holding generated orders */
    private BufferedOrderProcessor ordersBuffer;
    
    
    /**
     * @param scheduler the scheduler to set
     */
    public void setScheduler(Scheduler scheduler) {
        this.scheduler = scheduler;
    }

    /**
     * @param trigger the trigger to set
     */
    public void setTrigger(SimpleTrigger trigger) {
        this.trigger = trigger;
    }

    /**
     * @param jobDetail the jobDetail to set
     */
    public void setJobDetail(JobDetail jobDetail) {
        this.jobDetail = jobDetail;
    }

    /**
     * @param ordersBuffer the ordersBuffer to set
     */
    public void setOrdersBuffer(BufferedOrderProcessor ordersBuffer) {
        this.ordersBuffer = ordersBuffer;
    }

    /* (non-Javadoc)
     * @see de.itemis.qimpress.showcase.order_shipment_generator.be.generator.GeneratorManager#isOrderGeneratorRunning()
     */
    /* @Override */
    public boolean isOrderGeneratorRunning() {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">> isOrderGeneratorRunning()");
        }
        
        try {
            return !scheduler.isInStandbyMode();
        } catch (SchedulerException e) {
            LOG.error("Failed to get scheduler status", e);
            return false;
        }
    }

    /* (non-Javadoc)
     * @see de.itemis.qimpress.showcase.order_shipment_generator.be.generator.GeneratorManager#startOrderGenerator(de.itemis.qimpress.showcase.order_shipment_generator.be.configurator.model.GlobalParameters)
     */
    /* @Override */
    public boolean startOrderGenerator(GlobalParameters parameters) {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">> startOrderGenerator()");
        }
        
        if (!isOrderGeneratorRunning()) {
            
            // set default values for the real time mode
            if (parameters.getTimeParameters().getGeneratorTimeMode() == GeneratorTimeMode.REAL_TIME) {
                parameters.getTimeParameters().setTimeScaleFactor(1.0f);
                parameters.getTimeParameters().setStartTimeForOrders(new Date());
            }
            // set start time of the generator
            parameters.getTimeParameters().setStartTimeOfGenerator(new Date());
            
            try {
                // set trigger interval
                long repeatInterval = Math.round(parameters.getOrderInterval() * 1000);
                trigger.setRepeatInterval(repeatInterval);
                scheduler.rescheduleJob(trigger.getName(), trigger.getGroup(), trigger);
                
                // save parameters in the JobDataMap & reschedule job
                jobDetail.getJobDataMap().put("arg", parameters);
                scheduler.addJob(jobDetail, true);
            
                // start scheduler
                scheduler.start();
                
            } catch (SchedulerException e) {
                LOG.error("Failed to start scheduler with an updated job", e);
                return false;
            }
            return true;
        }
        return false;
    }

    /* (non-Javadoc)
     * @see de.itemis.qimpress.showcase.order_shipment_generator.be.generator.GeneratorManager#stopOrderGenerator()
     */
    /** {@inheritDoc} 
     * @throws ApplicationException */
    /* @Override */
    public boolean stopOrderGenerator() throws ApplicationException {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">> stopOrderGenerator()");
        }
        
        try {
            scheduler.standby();
            ordersBuffer.flush();
        } catch (SchedulerException e) {
            LOG.error("Failed to pause scheduler");
            return false;
        }
        return true;
    }

}