/*******************************************************************************
 * Copyright (c) 2008,2009 Q-ImPrESS consortium
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * This work was funded in the context of the Q-ImPrESS research project  
 * (FP7-215013) by the European Union under the Information and  
 * Communication Technologies priority of the Seventh Research Framework  
 * Programme.
 *
 * Contributors:
 *     itemis 	- initial API and implementation
 *******************************************************************************/
package de.itemis.qimpress.showcase.order_simulator.be.domain;

import java.math.BigDecimal;
import java.util.Date;
import java.util.List;

import javax.persistence.CascadeType;
import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.FetchType;
import javax.persistence.GeneratedValue;
import javax.persistence.Id;
import javax.persistence.JoinColumn;
import javax.persistence.OneToMany;
import javax.persistence.Table;
import javax.persistence.Transient;

import de.itemis.qimpress.showcase.order_simulator.be.dto.Customer;

/**
 * Holds all information about the order, including customer details, 
 * order items and product data.
 * 
 * @author Wladimir Safonov
 *
 */
@Entity
@Table(name = "order_")
public class Order {

    private Long id;
    
    private String number;
    private BigDecimal totalPrice;
    private Date creationDate;
    private Customer customer;
    // replicated Customer.customerNumber for persisting
    private String customerNumber; 
    private List<OrderItem> orderItems;
    
    // properties used to persist additional order information
    // e.g. customer data (customer name, type, category)
    // required to implement reporting
    private String additionalInfo1; // customer name
    private String additionalInfo2; // customer category
    private String additionalInfo3; // customer type
    
    public Order() {
        super();
    }

    public Order(String number, Date creationDate, Customer customer) {
        super();
        setNumber(number);
        setCreationDate(creationDate);
        setCustomer(customer);
    }

    /**
     * @return the id
     */
    @Id
    @GeneratedValue
    @Column(name = "order_id")
    public Long getId() {
        return id;
    }

    /**
     * @param id the id to set
     */
    private void setId(Long id) {
        this.id = id;
    }

    /**
     * @return the number
     */
    @Column(name = "order_number", nullable = false, unique = true)
    public String getNumber() {
        return number;
    }
    
    /**
     * @param number the number to set
     */
    public void setNumber(String number) {
        this.number = number;
    }

    /**
     * @return the totalPrice
     */
    @Column(name = "total_price")
    public BigDecimal getTotalPrice() {
        return totalPrice;
    }

    /**
     * @param totalPrice the totalPrice to set
     */
    public void setTotalPrice(BigDecimal totalPrice) {
        this.totalPrice = totalPrice;
    }

    /**
     * @return the creationDate
     */
    @Column(name = "creation_date")
    public Date getCreationDate() {
        return creationDate;
    }

    /**
     * @param creationDate the creationDate to set
     */
    public void setCreationDate(Date creationDate) {
        this.creationDate = creationDate;
    }

    /**
     * @return the customer
     */
    @Transient
    public Customer getCustomer() {
        return customer;
    }

    /**
     * @param customer the customer to set
     */
    public void setCustomer(Customer customer) {
        this.customer = customer;
        // set properties required to be persisted within Order
        if (customer != null) {
            this.setCustomerNumber(customer.getCustomerNumber());
            this.setAdditionalInfo1(customer.getName());
            this.setAdditionalInfo2(customer.getCustomerCategory().getCustomerCategoryKey());
            this.setAdditionalInfo3(customer.getCustomerType().getCustomerTypeName());
        } else {
            this.setCustomerNumber(null);
            this.setAdditionalInfo1(null);
            this.setAdditionalInfo2(null);
            this.setAdditionalInfo3(null);
        }
    }

    /**
     * @return the customerNumber
     */
    @Column(name = "customer_number")
    public String getCustomerNumber() {
        return customerNumber;
    }

    /**
     * @param customerNumber the customerNumber to set
     */
    public void setCustomerNumber(String customerNumber) {
        this.customerNumber = customerNumber;
    }

    /**
     * @return the orderItems
     */
    @OneToMany(fetch = FetchType.EAGER, cascade = {CascadeType.ALL})
    @JoinColumn(name = "order_id")
    public List<OrderItem> getOrderItems() {
        return orderItems;
    }

    /**
     * @param orderItems the orderItems to set
     */
    public void setOrderItems(List<OrderItem> orderItems) {
        this.orderItems = orderItems;
    }

    /**
     * Allows to retrieve the customer name.
     * 
     * @return the customer name
     */
    @Column(name = "additional_info1")
    public String getAdditionalInfo1() {
        return additionalInfo1;
    }

    /**
     * Allows to set the customer name.
     * 
     * @param additionalInfo1 the customer name to set
     */
    public void setAdditionalInfo1(String additionalInfo1) {
        this.additionalInfo1 = additionalInfo1;
    }

    /**
     * Allows to retrieve the customer category.
     * 
     * @return the customer category
     */
    @Column(name = "additional_info2")
    public String getAdditionalInfo2() {
        return additionalInfo2;
    }

    /**
     * Allows to set the customer category.
     * 
     * @param additionalInfo2 the customer category to set
     */
    public void setAdditionalInfo2(String additionalInfo2) {
        this.additionalInfo2 = additionalInfo2;
    }

    /**
     * Allows to retrieve the customer type.
     * 
     * @return the customer type
     */
    @Column(name = "additional_info3")
    public String getAdditionalInfo3() {
        return additionalInfo3;
    }

    /**
     * Allows to set the customer type.
     * 
     * @param additionalInfo3 the customer type to set
     */
    public void setAdditionalInfo3(String additionalInfo3) {
        this.additionalInfo3 = additionalInfo3;
    }

}