/*******************************************************************************
 * Copyright (c) 2008,2009 Q-ImPrESS consortium
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * This work was funded in the context of the Q-ImPrESS research project  
 * (FP7-215013) by the European Union under the Information and  
 * Communication Technologies priority of the Seventh Research Framework  
 * Programme.
 *
 * Contributors:
 *     itemis 	- initial API and implementation
 *******************************************************************************/
package de.itemis.qimpress.showcase.demo_app.be.service;

import java.util.List;

import de.itemis.qimpress.showcase.demo_app.be.dto.Country;
import de.itemis.qimpress.showcase.demo_app.be.dto.CustomerCategoryDiscount;
import de.itemis.qimpress.showcase.demo_app.be.dto.CustomerType;
import de.itemis.qimpress.showcase.demo_app.be.dto.Price;
import de.itemis.qimpress.showcase.demo_app.be.dto.ProductInfo;
import de.itemis.qimpress.showcase.demo_app.be.exceptions.ApplicationException;


/**
 * Service interface for the Webservice Demo-Application
 * 
 * @author Wladimir Safonov
 *
 */
public interface DemoAppManager {

    /**
     * Loads the list of all stored 
     * {@link Country Countries}
     * from the CRM system.
     * 
     * @return list of all stored 
     * {@link Country Countries}
     * @throws ApplicationException if an error occurred that cannot be resolved in the business layer
     */
    public List<Country> getCountries() throws ApplicationException;

    /**
     * Loads the list of all stored
     * {@link CustomerType CustomerTypes}
     * from the CRM system
     * 
     * @return list of all stored
     * {@link CustomerType CustomerTypes}
     * @throws ApplicationException if an error occurred that cannot be resolved in the business layer
     */
    public List<CustomerType> getCustomerTypes() throws ApplicationException;
    
    /**
     * Loads the list of all stored 
     * {@link CustomerCategoryDiscount CustomerCategoryDiscounts}
     * from the Pricing system.
     * 
     * @return list of all stored
     * {@link CustomerCategoryDiscount CustomerCategoryDiscounts}
     * @throws ApplicationException if an error occurred that cannot be resolved in the business layer
     */
    public List<CustomerCategoryDiscount> getAllCustomerCategoryDiscounts() throws ApplicationException;
    
    /**
     * Retrieves volume-dependent price for the given product and country from 
     * the Pricing system. The resulting price depends on the given customer category 
     * and customer type.
     * 
     * @param productCode code of the product to calculate the price for
     * @param countryCode ISO code of the country to look the product price for
     * @param amount product quantity to use in volume discount rules
     * @param customerCategoryKey key of the customer category for category discounts
     * @param customerType customer type for reseller price calculation
     * @return total volume price for the product including applied discounts
     * @throws ApplicationException if an error occurred that cannot be resolved in the business layer
     */
    public Price calculateVolumePriceForCustomerCategory(
            String productCode, String countryCode, int amount, String customerCategoryKey, String customerType) 
                                                                            throws ApplicationException;

    /**
     * Retrieves {@link ProductInfo} with all pertinent information describing a product 
     * with the given product code.
     * 
     * @param productCode product code to look up
     * {@link de.itemis.qimpress.showcase.demo_app.be.dto.ProductInfo} for
     * @return {@link de.itemis.qimpress.showcase.demo_app.be.dto.ProductInfo} or null if not found
     * @throws ApplicationException if an error occurred that cannot be resolved in the business layer
     */
    public ProductInfo getProductInfo(String productCode) throws ApplicationException;
    
    /**
     * Retrieves product {@link Price} for a specific country.
     * 
     * @param productCode code of the product to find the price for
     * @param countryCode ISO country code
     * @return {@link Price} of the product in the specified country
     * @throws ApplicationException if an error occurred that cannot be resolved in the business layer
     */
    public Price getProductPriceForCountry(String productCode, String countryCode) throws ApplicationException;
}
