/*******************************************************************************
 * Copyright (c) 2008,2009 Q-ImPrESS consortium
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * This work was funded in the context of the Q-ImPrESS research project  
 * (FP7-215013) by the European Union under the Information and  
 * Communication Technologies priority of the Seventh Research Framework  
 * Programme.
 *
 * Contributors:
 *     itemis 	- initial API and implementation
 *******************************************************************************/
package de.itemis.qimpress.showcase.crm_simulator.webservice;

import java.io.IOException;
import java.io.StringWriter;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.List;

import javax.naming.NamingException;

import org.apache.log4j.Logger;
import org.jdom.Element;
import org.jdom.JDOMException;
import org.jdom.Namespace;
import org.jdom.xpath.XPath;
import org.springframework.ws.server.endpoint.AbstractJDomPayloadEndpoint;

import com.csvreader.CsvWriter;

import de.itemis.qimpress.common.Meassurements;
import de.itemis.qimpress.showcase.crm_simulator.be.domain.Country;
import de.itemis.qimpress.showcase.crm_simulator.be.domain.Customer;
import de.itemis.qimpress.showcase.crm_simulator.be.domain.CustomerCategory;
import de.itemis.qimpress.showcase.crm_simulator.be.domain.CustomerType;
import de.itemis.qimpress.showcase.crm_simulator.be.filter.CustomerFilter;
import de.itemis.qimpress.showcase.crm_simulator.be.service.CRMManager;
import de.itemis.qimpress.showcase.crm_simulator.be.service.ServiceLocator;

/**
 * JDomEndpoint for the CRMManager service methods getCountries,
 * getCustomerTypes(), getCustomerCategories, queryCustomers.
 */
public class CrmJDomEndpoint extends AbstractJDomPayloadEndpoint {

    /** Logger for CrmJDomEndpoint. */
    private static final Logger LOG = Logger.getLogger(CrmJDomEndpoint.class);

    // constants
    public static final String GET_COUNTRIES_CSV_REQUEST = "GetCountriesCSVRequest";
    public static final String GET_COUNTRIES_CSV_RESPONSE = "GetCountriesCSVResponse";
    public static final String GET_COUNTRIES_REQUEST = "GetCountriesRequest";
    public static final String GET_COUNTRIES_RESPONSE = "GetCountriesResponse";
    public static final String GET_CUSTOMERTYPES_REQUEST = "GetCustomerTypesRequest";
    public static final String GET_CUSTOMERTYPES_RESPONSE = "GetCustomerTypesResponse";
    public static final String GET_CUSTOMERCATEGORIES_REQUEST = "GetCustomerCategoriesRequest";
    public static final String GET_CUSTOMERCATEGORIES_RESPONSE = "GetCustomerCategoriesResponse";
    public static final String GET_QUERYCUSTOMER_REQUEST = "QueryCustomersRequest";
    public static final String GET_QUERYCUSTOMER_RESPONSE = "QueryCustomersResponse";

    // jdom attributes:
    private Namespace namespace;
    private XPath customerFilterXPath;

    // crmManager is not injected - instanced in CrmJDomEndpoint constructor 
    private CRMManager crmManager = ServiceLocator.getInstance().getCRMManager();
    //    public void setCrmManager(CRMManager crmManager) {
    //        if (LOG.isDebugEnabled()) {
    //            LOG.debug("\t=>WS<= called: setCrmManager");
    //        }
    //        this.crmManager = crmManager;
    //    }

    private final SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd'T'hh:mm:ss");

    public CrmJDomEndpoint() {

        if (crmManager == null) {
            
        }
        if (crmManager == null) {
            LOG.error("ERROR: CRMManager is null!" + " Problem getting CRMManager by (BE) ServiceLocator");
        }
        namespace = Namespace.getNamespace("crm", "http://www.itemis.com/crm/schemas");
        try {
            // for the QueryCustomersRequest - customer filter parameter
            customerFilterXPath = XPath.newInstance("/crm:QueryCustomersRequest/customerFilterParameter");
        } catch (JDOMException e) {
            e.printStackTrace();
        }
        customerFilterXPath.addNamespace(namespace);
    }

    // implement method from abstract class AbstractJDomPayloadEndpoint
    @Override
    protected Element invokeInternal(Element element) throws Exception {
        if (LOG.isDebugEnabled()) {
            LOG.debug("\t=>WS<= called: invokeInternal");
        }

        // TODO: check to delete useless LOGs (=> please check the whole class)

        String rootElement = element.getName();
        if (LOG.isDebugEnabled()) {
            LOG.debug("*** rootElement = element.getName() = " + rootElement);
        }
        if (GET_COUNTRIES_CSV_REQUEST.equals(rootElement)) {
            
        	if (LOG.isDebugEnabled()) {
                LOG.debug("*** rootElement: GET_COUNTRIES_CSV_REQUEST");
            }
            List<Country> countriesList = crmManager.getCountries();
            if (LOG.isDebugEnabled()) {
                LOG.debug("*** countriesList-size = " + countriesList.size());
            }
            
            return createGetCountriesCSVResponse(countriesList);
        }
        if (GET_COUNTRIES_REQUEST.equals(rootElement)) {
            
    		long   start = 0;
    		long   stop  = 0;

        	if (LOG.isTraceEnabled()) {
        		start = System.currentTimeMillis();
        	}
        	
        	if (LOG.isDebugEnabled()) {
                LOG.debug("*** rootElement: GET_COUNTRIES_REQUEST");
            }
            // here we do not need to extract parameter(s) from request
            // (no parameter in the getCountries service method)
            // 'service-request' for getCountries method
            // get countries by crmManager
            List<Country> countriesList = crmManager.getCountries();
            
            if (LOG.isDebugEnabled()) {
                LOG.debug("*** countriesList-size = " + countriesList.size());
            }
            
            Element response = createGetCountriesResponse(countriesList);
            
            if (LOG.isTraceEnabled()) {
            	stop = System.currentTimeMillis();
            	Meassurements.add(this.getClass().getSimpleName(), "GetCountriesRequest", start, stop);
            	Meassurements.log();
            }
            
            return response;
        }
        if (GET_CUSTOMERTYPES_REQUEST.equals(rootElement)) {
            
    		long   start = 0;
    		long   stop  = 0;

        	if (LOG.isTraceEnabled()) {
        		start = System.currentTimeMillis();
        	}
        	
        	if (LOG.isDebugEnabled()) {
                LOG.debug("*** rootElement: GET_CUSTOMERTYPES_REQUEST");
            }
            // 'service-request' for getCustomerTypes method
            // get customer types  by crmManager
            List<CustomerType> customerTypesList = crmManager.getCustomerTypes();
            Element response = createGetCustomerTypeResponse(customerTypesList);
            
            if (LOG.isTraceEnabled()) {
            	stop = System.currentTimeMillis();
            	Meassurements.add(this.getClass().getSimpleName(), "GetCustomerTypesRequest", start, stop);
            	Meassurements.log();
            }
                        
            return response;
        }

        if (GET_CUSTOMERCATEGORIES_REQUEST.equals(rootElement)) {
            
    		long   start = 0;
    		long   stop  = 0;

        	if (LOG.isTraceEnabled()) {
        		start = System.currentTimeMillis();
        	}
        	
        	if (LOG.isDebugEnabled()) {
                LOG.debug("*** rootElement: GET_CUSTOMERCATEGORIES_REQUEST");
            }
            // 'service-request' for getCustomerCategories method
            // get customer categories by crmManager
            List<CustomerCategory> customerCategoriesList = crmManager.getCustomerCategories();
            
            if (LOG.isDebugEnabled()) {
                LOG.debug("customerCategoriesList.size() = " + customerCategoriesList.size());
            }
            
            Element response = createGetCustomerCategoriesResponse(customerCategoriesList);

            if (LOG.isTraceEnabled()) {
            	stop = System.currentTimeMillis();
            	Meassurements.add(this.getClass().getSimpleName(), "GetCustomerCategoriesRequest", start, stop);
            	Meassurements.log();
            }
            
            return response;
            //            return new Element(GET_CUSTOMERCATEGORIES_RESPONSE, namespace);
        }
        
        if (GET_QUERYCUSTOMER_REQUEST.equals(rootElement)) {
    		
        	long   start = 0;
    		long   stop  = 0;

        	if (LOG.isTraceEnabled()) {
        		start = System.currentTimeMillis();
        	}
        	
        	if (LOG.isDebugEnabled()) {
                LOG.debug("*** rootElement: GET_QUERYCUSTOMER_REQUEST");
            }
            // for the QueryCustomersRequest - customer filter parameter
            //            customerFilterXPath = XPath.newInstance("/crm:QueryCustomersRequest/customerFilterParameter");
            //            customerFilterXPath.addNamespace(namespace);
            //            LOG.debug("\t=> customerFilterXPath = " +customerFilterXPath);
            //            LOG.debug("\t=> customerFilterXPath.getXPath() = " +customerFilterXPath.getXPath());

            CustomerFilter customerFilter = extractCustomerFilterFromQueryCustomersRequest(element);

            List<Customer> customers = crmManager.queryCustomers(customerFilter);

            if (LOG.isDebugEnabled()) {
                LOG.debug("### customers.size() = " + customers.size());
                for (Customer customer : customers) {
                    LOG.debug("### customer NAME = " + customer.getName());
                }
            }

            // 'service-request' for queryCustomers method
            // get customerFilter parameter from request
            // (there is only one element - the customerFilterParameter)
            //            CustomerFilter filter = (CustomerFilter)
            //                    customerFilterXPath.selectNodes(element).get(0);
            
            Element response = createQueryCustomersResponse(customers);
            
            if (LOG.isTraceEnabled()) {
            	stop = System.currentTimeMillis();
            	Meassurements.add(this.getClass().getSimpleName(), "QueryCustomersRequest", start, stop);
            	Meassurements.log();
            }
            
            return response;
        }

        return null;
    }

    /**
     * extracts the customerFilter parameter
     * from the QueryCustomersRequest element.
     * @param element the request element
     * @return the customerFilter parameter from the request element
     */
    private CustomerFilter extractCustomerFilterFromQueryCustomersRequest(Element element) {
        CustomerFilter customerFilter = new CustomerFilter();
        String filterNamePattern = null;
        String filterNumberPattern = null;
        CustomerCategory filterCategory = null;
        CustomerType filterType = null;
        Country filterDeliveryCountry = null;
        Country filterInvoiceCountry = null;

        Element customerFilterEl = element.getChild("customerFilter", namespace);
        if (customerFilterEl != null) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("ELEMENT - customerFilterEl = " + customerFilterEl);
            }
            Element filterNameElement = customerFilterEl.getChild("customerNamePattern", namespace);
            Element filterNumberElement = customerFilterEl.getChild("customerNumberPattern", namespace);
            Element filterCategoryElement = customerFilterEl.getChild("categoryFilter", namespace);
            Element filterTypeElement = customerFilterEl.getChild("typeFilter", namespace);
            Element filterDelCountryElement = customerFilterEl.getChild("deliveryCountry", namespace);
            Element filterInvCountryElement = customerFilterEl.getChild("invoiceCountry", namespace);
            if (LOG.isDebugEnabled()) {
                LOG.debug("ELEMENT - filterNameElement = " + filterNameElement);
                LOG.debug("ELEMENT - filterNumberElement = " + filterNumberElement);
                LOG.debug("ELEMENT - filterCategoryElement = " + filterCategoryElement);
                LOG.debug("ELEMENT - filterTypeElement = " + filterTypeElement);
                LOG.debug("ELEMENT - filterDelCountryElement = " + filterDelCountryElement);
                LOG.debug("ELEMENT - filterInvCountryElement = " + filterInvCountryElement);
            }
            if (filterNameElement != null) {
                filterNamePattern = filterNameElement.getValue();
                if (LOG.isDebugEnabled()) {
                    LOG.debug("FILTER - filterNamePattern = " + filterNamePattern);
                }
            }
            if (filterNumberElement != null) {
                filterNumberPattern = filterNumberElement.getValue();
                if (LOG.isDebugEnabled()) {
                    LOG.debug("FILTER - filterNumberPattern = " + filterNumberPattern);
                }
            }
            if (filterCategoryElement != null) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("FILTER - filterCategoryElement");
                }
                filterCategory = new CustomerCategory();
                Element categoryIdElement = filterCategoryElement.getChild("customerCategoryId", namespace);
                Element categoryKeyElement = filterCategoryElement.getChild("customerCategoryKey", namespace);
                Element categoryNameElement = filterCategoryElement.getChild("customerCategoryName", namespace);
                if (LOG.isDebugEnabled()) {
                    LOG.debug("categoryIdElement = " + categoryIdElement);
                    LOG.debug("categoryKeyElement = " + categoryKeyElement);
                    LOG.debug("categoryNameElement = " + categoryNameElement);
                }
                //                    Element categoryCreatedElement = filterCategoryElement.getChild("customerCategoryDateCreated", namespace);
                //                    Element categoryModifiedElement = filterCategoryElement.getChild("customerCategoryDateModified", namespace);
                if (categoryIdElement != null && !"".equals(categoryIdElement.getValue())) {
                    filterCategory.setCustomerCategoryId(categoryIdElement.getValue());
                }
                if (categoryKeyElement != null && !"".equals(categoryKeyElement.getValue())) {
                    filterCategory.setCustomerCategoryKey(categoryKeyElement.getValue());
                }
                if (categoryNameElement != null && !"".equals(categoryNameElement.getValue())) {
                    filterCategory.setCustomerCategoryName(categoryNameElement.getValue());
                }
            }
            if (filterTypeElement != null) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("FILTER - filterTypeElement");
                }
                filterType = new CustomerType();
                Element typeIdElement = filterTypeElement.getChild("customerTypeId", namespace);
                Element typeKeyElement = filterTypeElement.getChild("customerTypeKey", namespace);
                Element typeNameElement = filterTypeElement.getChild("customerTypeName", namespace);
                if (typeIdElement != null && !"".equals(typeIdElement.getValue())) {
                    filterType.setCustomerTypeId(typeIdElement.getValue());
                }
                if (typeKeyElement != null && !"".equals(typeKeyElement.getValue())) {
                    filterType.setCustomerTypeKey(typeKeyElement.getValue());
                }
                if (typeNameElement != null && !"".equals(typeNameElement.getValue())) {
                    filterType.setCustomerTypeName(typeNameElement.getValue());
                }
            }
            if (filterDelCountryElement != null) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("FILTER - filterDelCountryElement");
                }
                filterDeliveryCountry = new Country();
                Element delCountryIdElement = filterDelCountryElement.getChild("countryId", namespace);
                Element delCountryIsoCodeElement = filterDelCountryElement.getChild("countryIsoCode", namespace);
                Element delCountryNameElement = filterDelCountryElement.getChild("countryName", namespace);
                if (delCountryIdElement != null && !"".equals(delCountryIdElement.getValue())) {
                    filterDeliveryCountry.setCountryId(delCountryIdElement.getValue());
                }
                if (delCountryIsoCodeElement != null && !"".equals(delCountryIsoCodeElement.getValue())) {
                    filterDeliveryCountry.setIsoCode(delCountryIsoCodeElement.getValue());
                }
                if (delCountryNameElement != null && !"".equals(delCountryNameElement.getValue())) {
                    filterDeliveryCountry.setName(delCountryNameElement.getValue());
                }
            }
            if (filterInvCountryElement != null) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("FILTER - filterInvCountryElement");
                }
                filterInvoiceCountry = new Country();
                Element invCountryIdElement = filterInvCountryElement.getChild("countryId", namespace);
                Element invCountryIsoCodeElement = filterInvCountryElement.getChild("countryIsoCode", namespace);
                Element invCountryNameElement = filterInvCountryElement.getChild("countryName", namespace);
                if (invCountryIdElement != null && !"".equals(invCountryIdElement.getValue())) {
                    filterInvoiceCountry.setCountryId(invCountryIdElement.getValue());
                }
                if (invCountryIsoCodeElement != null && !"".equals(invCountryIsoCodeElement.getValue())) {
                    filterInvoiceCountry.setIsoCode(invCountryIsoCodeElement.getValue());
                }
                if (invCountryNameElement != null && !"".equals(invCountryNameElement.getValue())) {
                    filterInvoiceCountry.setName(invCountryNameElement.getValue());
                }
            }
        } else {
            LOG.debug("THE ELEMENT 'CustomerFilter' is NULL!");
        }
        // set customerFilter:
        customerFilter.setCustomerNamePattern(filterNamePattern);
        customerFilter.setCustomerNumberPattern(filterNumberPattern);
        customerFilter.setCategory(filterCategory);
        customerFilter.setCustomerType(filterType);
        customerFilter.setDeliveryCountry(filterDeliveryCountry);
        customerFilter.setInvoiceCountry(filterInvoiceCountry);
        return customerFilter;
    }

    // response methods:
    private Element createGetCountriesResponse(List<Country> countriesList) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("\t=>WS<= called: createGetCountriesResponse");
        }
        Element responseRootElement = new Element(GET_COUNTRIES_RESPONSE, namespace);
        List<Element> responseCountriesList = new ArrayList<Element>();
        // iterate the countries list and add each country as an element to response
        for (Country country : countriesList) {
            Element nextElement = new Element("Country", namespace);
            nextElement.addContent(new Element("countryId", namespace).addContent(country.getCountryId()));
            nextElement.addContent(new Element("countryIsoCode", namespace).addContent(country.getIsoCode()));
            nextElement.addContent(new Element("countryName", namespace).addContent(country.getName()));
            // TODO check date-format / date-values
            nextElement.addContent(new Element("countryDateCreated", namespace).addContent(sdf.format(country
                    .getDateCreated())));
            nextElement.addContent(new Element("countryDateModified", namespace).addContent(sdf.format(country
                    .getDateLastModified())));

            responseCountriesList.add(nextElement);
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("*** responseCountriesList-size = " + responseCountriesList.size());
        }
        responseRootElement.addContent(responseCountriesList);
        return responseRootElement;
    }

    private Element createGetCountriesCSVResponse(List<Country> countriesList) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("\t=>WS<= called: createGetCountriesCSVResponse");
        }
        Element csvResponseRootElement = new Element(GET_COUNTRIES_CSV_RESPONSE, namespace);
        Element countriesCSV = new Element("CountriesCSV", namespace);
        csvResponseRootElement.setContent(countriesCSV);

        StringWriter csvContentWriter = new StringWriter();
        CsvWriter csvWriter = new CsvWriter(csvContentWriter, ';');
        try {
            // write CSV header
            csvWriter.writeRecord(new String[] { "Country_Name", "Country_Code" });
            // set text qualifier
            csvWriter.setTextQualifier('"');
            csvWriter.setForceQualifier(true);
            // iterate the countries list and add each country as a CSV line to response
            for (Country country : countriesList) {
                csvWriter.write(country.getName());
                csvWriter.write(country.getIsoCode());
                csvWriter.endRecord();
            }
        } catch (IOException e) {
            LOG.error("Could not write countries CSV content", e);
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("*** countriesCsvData written");
        }
        countriesCSV.setText(csvContentWriter.toString());
        return csvResponseRootElement;
    }

    private Element createGetCustomerTypeResponse(List<CustomerType> customerTypesList) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("\t=>WS<= called: createGetCustomerTypeResponse");
        }
        Element responseElement = new Element(GET_CUSTOMERTYPES_RESPONSE, namespace);
        List<Element> responseTypesList = new ArrayList<Element>();
        // iterate the types list and add the elements to response element
        for (CustomerType type : customerTypesList) {
            Element nextElement = new Element("CustomerType", namespace);
            nextElement.addContent(new Element("customerTypeId", namespace).addContent(type.getCustomerTypeId()));
            nextElement.addContent(new Element("customerTypeKey", namespace).addContent(type.getCustomerTypeKey()));
            nextElement.addContent(new Element("customerTypeName", namespace).addContent(type.getCustomerTypeName()));
            nextElement.addContent(new Element("customerTypeDateCreated", namespace).addContent(sdf.format(type
                    .getDateCreated())));
            nextElement.addContent(new Element("customerTypeDateModified", namespace).addContent(sdf.format(type
                    .getDateLastModified())));

            responseTypesList.add(nextElement);
        }

        if (LOG.isDebugEnabled()) {
            LOG.debug("*** responseTypesList-size = " + responseTypesList.size());
        }
        responseElement.addContent(responseTypesList);
        return responseElement;
    }

    private Element createGetCustomerCategoriesResponse(List<CustomerCategory> customerCategoriesList) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("\t=>WS<= called: createGetCountriesResponse");
        }
        Element responseElement = new Element(GET_CUSTOMERCATEGORIES_RESPONSE, namespace);
        List<Element> responseCategoriesList = new ArrayList<Element>();
        // iterate the categories list and add the elements to response element
        for (CustomerCategory category : customerCategoriesList) {
            Element nextElement = new Element("CustomerCategory", namespace);
            nextElement.addContent(new Element("customerCategoryId", namespace).addContent(category
                    .getCustomerCategoryId()));
            nextElement.addContent(new Element("customerCategoryKey", namespace).addContent(category
                    .getCustomerCategoryKey()));
            nextElement.addContent(new Element("customerCategoryName", namespace).addContent(category
                    .getCustomerCategoryName()));
            nextElement.addContent(new Element("customerCategoryDateCreated", namespace).addContent(sdf.format(category
                    .getDateCreated())));
            nextElement.addContent(new Element("customerCategoryDateModified", namespace).addContent(sdf
                    .format(category.getDateLastModified())));

            responseCategoriesList.add(nextElement);
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("*** responseCategoriesList-size = " + responseCategoriesList.size());
        }
        responseElement.addContent(responseCategoriesList);
        return responseElement;
    }

    private Element createQueryCustomersResponse(List<Customer> customersList) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("\t=>WS<= called: createQueryCustomersResponse");
        }
        Element responseElement = new Element(GET_QUERYCUSTOMER_RESPONSE, namespace);
        List<Element> responseCustomersList = new ArrayList<Element>();
        // iterate the customers list and add the elements to response element
        for (Customer customer : customersList) {
            Element nextElement = new Element("Customer", namespace);
            nextElement.addContent(new Element("customerId", namespace).addContent(customer.getCustomerId()));
            nextElement.addContent(new Element("customerName", namespace).addContent(customer.getName()));
            nextElement.addContent(new Element("customerNumber", namespace).addContent(customer.getCustomerNumber()));
            nextElement.addContent(new Element("deliveryAddressField1", namespace).addContent(customer
                    .getDeliveryAddressField1()));
            nextElement.addContent(new Element("deliveryAddressField2", namespace).addContent(customer
                    .getDeliveryAddressField2()));
            nextElement.addContent(new Element("deliveryAddressField3", namespace).addContent(customer
                    .getDeliveryAddressField3()));
            nextElement.addContent(new Element("deliveryAddressStreet", namespace).addContent(customer
                    .getDeliveryAddressStreet()));
            nextElement.addContent(new Element("deliveryAddressStreetNumber", namespace).addContent(customer
                    .getDeliveryAddressStreetNumber()));
            nextElement.addContent(new Element("deliveryAddressZipCode", namespace).addContent(customer
                    .getDeliveryAddressZipCode()));
            nextElement.addContent(new Element("deliveryAddressCity", namespace).addContent(customer
                    .getDeliveryAddressCity()));
            nextElement.addContent(new Element("deliveryAddressLongitude", namespace).addContent(new Double(customer
                    .getDeliveryAddressLongitude()).toString()));
            nextElement.addContent(new Element("deliveryAddressLatitude", namespace).addContent(new Double(customer
                    .getDeliveryAddressLatitude()).toString()));
            nextElement.addContent(new Element("invoiceAddressField1", namespace).addContent(customer
                    .getInvoiceAddressField1()));
            nextElement.addContent(new Element("invoiceAddressField2", namespace).addContent(customer
                    .getInvoiceAddressField2()));
            nextElement.addContent(new Element("invoiceAddressField3", namespace).addContent(customer
                    .getInvoiceAddressField3()));
            nextElement.addContent(new Element("invoiceAddressStreet", namespace).addContent(customer
                    .getInvoiceAddressStreet()));
            nextElement.addContent(new Element("invoiceAddressStreetNumber", namespace).addContent(customer
                    .getInvoiceAddressStreetNumber()));
            nextElement.addContent(new Element("invoiceAddressZipCode", namespace).addContent(customer
                    .getInvoiceAddressZipCode()));
            nextElement.addContent(new Element("invoiceAddressCity", namespace).addContent(customer
                    .getInvoiceAddressCity()));
            nextElement.addContent(new Element("invoiceAddressLongitude", namespace).addContent(new Double(customer
                    .getInvoiceAddressLongitude()).toString()));
            nextElement.addContent(new Element("invoiceAddressLatitude", namespace).addContent(new Double(customer
                    .getInvoiceAddressLatitude()).toString()));
            nextElement.addContent(new Element("commentary", namespace).addContent(customer.getCommentary()));
            nextElement.addContent(new Element("dateCreated", namespace).addContent(sdf.format(customer
                    .getDateCreated())));
            nextElement.addContent(new Element("lastModified", namespace).addContent(sdf.format(customer
                    .getDateLastModified())));
            nextElement.addContent(new Element("discountRate", namespace).addContent(new Float(customer
                    .getDiscountRate()).toString()));
            nextElement.addContent(new Element("contactPersonName", namespace).addContent(customer
                    .getContactPersonName()));
            nextElement.addContent(new Element("contactPersonPhone", namespace).addContent(customer
                    .getContactPersonPhone()));
            nextElement.addContent(new Element("contactPersonEmail", namespace).addContent(customer
                    .getContactPersonEmail()));
            nextElement.addContent(new Element("websiteUrl", namespace).addContent(customer.getWebsiteUrl()));
            // create deliveryAddressCountry element
            Element deliveryAdrCountry = new Element("deliveryAddressCountry", namespace);
            deliveryAdrCountry.addContent(new Element("countryId", namespace).addContent(customer
                    .getDeliveryAddressCountry().getCountryId()));
            deliveryAdrCountry.addContent(new Element("countryIsoCode", namespace).addContent(customer
                    .getDeliveryAddressCountry().getIsoCode()));
            deliveryAdrCountry.addContent(new Element("countryName", namespace).addContent(customer
                    .getDeliveryAddressCountry().getName()));
            deliveryAdrCountry.addContent(new Element("countryDateCreated", namespace).addContent(sdf.format(customer
                    .getDeliveryAddressCountry().getDateCreated())));
            deliveryAdrCountry.addContent(new Element("countryDateModified", namespace).addContent(sdf.format(customer
                    .getDeliveryAddressCountry().getDateLastModified())));
            nextElement.addContent(deliveryAdrCountry);
            // create invoiceAddressCountry element
            Element invoiceAddressCountry = new Element("invoiceAddressCountry", namespace);
            invoiceAddressCountry.addContent(new Element("countryId", namespace).addContent(customer
                    .getDeliveryAddressCountry().getCountryId()));
            invoiceAddressCountry.addContent(new Element("countryIsoCode", namespace).addContent(customer
                    .getDeliveryAddressCountry().getIsoCode()));
            invoiceAddressCountry.addContent(new Element("countryName", namespace).addContent(customer
                    .getDeliveryAddressCountry().getName()));
            invoiceAddressCountry.addContent(new Element("countryDateCreated", namespace).addContent(sdf
                    .format(customer.getDeliveryAddressCountry().getDateCreated())));
            invoiceAddressCountry.addContent(new Element("countryDateModified", namespace).addContent(sdf
                    .format(customer.getDeliveryAddressCountry().getDateLastModified())));
            nextElement.addContent(invoiceAddressCountry);
            // create customerType element
            Element customerTypeElement = new Element("customerType", namespace);
            customerTypeElement.addContent(new Element("customerTypeId", namespace).addContent(customer
                    .getCustomerType().getCustomerTypeId()));
            customerTypeElement.addContent(new Element("customerTypeKey", namespace).addContent(customer
                    .getCustomerType().getCustomerTypeKey()));
            customerTypeElement.addContent(new Element("customerTypeName", namespace).addContent(customer
                    .getCustomerType().getCustomerTypeName()));
            customerTypeElement.addContent(new Element("customerTypeDateCreated", namespace).addContent(sdf
                    .format(customer.getCustomerType().getDateCreated())));
            customerTypeElement.addContent(new Element("customerTypeDateModified", namespace).addContent(sdf
                    .format(customer.getCustomerType().getDateLastModified())));
            nextElement.addContent(customerTypeElement);
            // create customerCategory elment
            Element customerCategoryElement = new Element("customerCategory", namespace);
            customerCategoryElement.addContent(new Element("customerCategoryId", namespace).addContent(customer
                    .getCustomerCategory().getCustomerCategoryId()));
            customerCategoryElement.addContent(new Element("customerCategoryKey", namespace).addContent(customer
                    .getCustomerCategory().getCustomerCategoryKey()));
            customerCategoryElement.addContent(new Element("customerCategoryName", namespace).addContent(customer
                    .getCustomerCategory().getCustomerCategoryName()));
            customerCategoryElement.addContent(new Element("customerCategoryDateCreated", namespace).addContent(sdf
                    .format(customer.getCustomerCategory().getDateCreated())));
            customerCategoryElement.addContent(new Element("customerCategoryDateModified", namespace).addContent(sdf
                    .format(customer.getCustomerCategory().getDateLastModified())));
            nextElement.addContent(customerCategoryElement);

            responseCustomersList.add(nextElement);
        }

        if (LOG.isDebugEnabled()) {
            LOG.debug("*** responseCustomersList-size = " + responseCustomersList.size());
        }
        responseElement.addContent(responseCustomersList);
        return responseElement;
    }

}
