/*******************************************************************************
 * Copyright (c) 2008,2009 Q-ImPrESS consortium
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * This work was funded in the context of the Q-ImPrESS research project  
 * (FP7-215013) by the European Union under the Information and  
 * Communication Technologies priority of the Seventh Research Framework  
 * Programme.
 *
 * Contributors:
 *     itemis 	- initial API and implementation
 *******************************************************************************/
package de.itemis.qimpress.showcase.crm_simulator.webservice.rest;

import java.io.IOException;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Properties;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.log4j.Logger;
import org.json.simple.JSONArray;
import org.json.simple.JSONObject;
import org.springframework.beans.BeansException;
import org.springframework.context.ApplicationContext;
import org.springframework.context.support.ClassPathXmlApplicationContext;
import org.springframework.ws.client.core.WebServiceTemplate;

import de.itemis.qimpress.showcase.crm_simulator.be.domain.Customer;
import de.itemis.qimpress.showcase.crm_simulator.be.filter.CustomerFilter;
import de.itemis.qimpress.showcase.crm_simulator.webservice.messages.QueryCustomersRequest;
import de.itemis.qimpress.showcase.crm_simulator.webservice.messages.QueryCustomersResponse;


/**
 * <p>
 * Servlet for the CRM RESTful Web Service acting on top of the
 * existing SOAP Web Service as an adapter. Uses JSON as a data 
 * representation format.
 * <p>
 * The current implementation supports a simple customer query
 * without filtering possibilities, which delivers a list of 
 * customers with a specific selection of attributes.
 * 
 * @author Wladimir Safonov
 *
 */
public class CrmRestServiceAdapterServlet extends HttpServlet {

    // TODO: add serializationID

    private static final Logger LOG = Logger.getLogger(CrmRestServiceAdapterServlet.class);

    private static final String WEBSERVICE_CONTEXT_PARAM = "webserviceContext";
    private static final String WEBSERVICE_TEMPLATE_BEAN_PARAM = "webserviceTemplateBean";

    private static final String CRM_WS_URI = "crm_ws_uri";
    
    private static final String CUSTOMER_RESOURCE_NAME = "customers";

    private MessageFormat crmWsUrlTemplate;
    private String crmWsUrl;
    private WebServiceTemplate webserviceTemplate;

    @Override
    public void init() throws ServletException {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">> init()");
        }
        super.init();

        // check servlet init params
        String webserviceCtxtFile = getInitParameter(WEBSERVICE_CONTEXT_PARAM);
        String webserviceTemplateBeanId = getInitParameter(WEBSERVICE_TEMPLATE_BEAN_PARAM);
        if (webserviceCtxtFile == null) {
            String errMsg = "Servlet '" + this.getServletName() + "' is not configured properly: missing '"
                    + WEBSERVICE_CONTEXT_PARAM + "' init parameter";
            LOG.error(errMsg);
            throw new IllegalStateException(errMsg);
        }
        if (webserviceTemplateBeanId == null) {
            String errMsg = "Servlet '" + this.getServletName() + "' is not configured properly: missing '"
                    + WEBSERVICE_CONTEXT_PARAM + "' init parameter";
            LOG.error(errMsg);
            throw new IllegalStateException(errMsg);
        }

        // load context and template bean
        ApplicationContext webserviceCtxt;
        try {
            webserviceCtxt = new ClassPathXmlApplicationContext(webserviceCtxtFile);
        } catch (BeansException e) {
            String errMsg = "Failed to load webservice context from '" + webserviceCtxtFile + "'. Check '"
                    + WEBSERVICE_CONTEXT_PARAM + "' servlet init parameter";
            LOG.error(errMsg, e);
            throw new IllegalStateException(errMsg, e);
        }
        try {
            webserviceTemplate = (WebServiceTemplate) webserviceCtxt.getBean(webserviceTemplateBeanId);
        } catch (BeansException e) {
            String errMsg = "Failed to load webservice template bean from the context '" + webserviceCtxtFile + "'. Check '"
                    + WEBSERVICE_TEMPLATE_BEAN_PARAM + "' servlet init parameter";
            LOG.error(errMsg, e);
            throw new IllegalStateException(errMsg, e);
        }
        
        // load URLs properties
        Properties webServiceURIs = new Properties();
        try {
            webServiceURIs.load(this.getClass().getClassLoader().getResourceAsStream("webservice.properties"));
        } catch (IOException e) {
            LOG.error("Failed to load webservice.properties file", e);
        }
        crmWsUrlTemplate = new MessageFormat(webServiceURIs.getProperty(CRM_WS_URI));
    }

    @Override
    protected void doGet(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">> doGet()");
        }

        // check if resource path has been supplied
        String resourcePath = req.getPathInfo();
        if (resourcePath == null) {
            resp.sendError(HttpServletResponse.SC_BAD_REQUEST, "Resource name is missing");
            return;
        }
        
        if (crmWsUrl == null) {
            crmWsUrl = crmWsUrlTemplate.format(new Object[] { req.getServerName(), Integer.toString(req.getServerPort()) });
        }
        
        // determine resource path parts
        String[] tokens = resourcePath.split("/");
        if (tokens.length == 0) {
            resp.sendError(HttpServletResponse.SC_BAD_REQUEST, "Resource name is missing");
            return;
        }
        List<String> resourcePathParts = new ArrayList<String>(tokens.length);
        for (int i = 0; i < tokens.length; i++) {
            if (tokens[i].length() != 0) {
                resourcePathParts.add(tokens[i]);
            }
        }
        
        // analyze resource parts
        if (resourcePathParts.size() == 0) {
            resp.sendError(HttpServletResponse.SC_BAD_REQUEST, "Resource name is missing");
            return;
        }
        if (CUSTOMER_RESOURCE_NAME.equals(resourcePathParts.get(0))) {
            // process GET request on 'customers' resource
            if (resourcePathParts.size() == 1) {
                // process 'GET /customers' request
                processGetAllCustomersRequest(resp);
            } else {
                // no further resource path parameters supported
                resp.sendError(HttpServletResponse.SC_NOT_IMPLEMENTED, "Resource request is not supported");
                return;
            }
        } else {
            // no valid resource found
            resp.sendError(HttpServletResponse.SC_NOT_FOUND, "Resource with the given name was not found");
            return;
        }
    }

    private void processGetAllCustomersRequest(HttpServletResponse resp) throws IOException {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">> processGetAllCustomersRequest()");
        }
        
        // get customer list from the WebService
        List<Customer> customers = getAllCustomers();
        
        // create JSON response
        JSONArray customerArray = new JSONArray();
        for (Customer customer : customers) {
            JSONObject customerObj = new JSONObject();
            customerObj.put("pocname", customer.getContactPersonName());
            customerObj.put("name", customer.getName());
            customerObj.put("poctelephone", customer.getContactPersonPhone());
            customerObj.put("customer_type", customer.getCustomerType().getCustomerTypeName());
            customerObj.put("customer_categorie", customer.getCustomerCategory().getCustomerCategoryKey());
            customerObj.put("invoice_adress_street", customer.getInvoiceAddressStreet());
            customerObj.put("invoice_adress_street_number", customer.getInvoiceAddressStreetNumber());
            customerObj.put("invoice_adress_zip_code", customer.getInvoiceAddressZipCode());
            customerObj.put("invoice_adress_city", customer.getInvoiceAddressCity());
            customerObj.put("invoice_adress_country", customer.getInvoiceAddressCountry().getName());
            customerObj.put("invoice_adress_longitude", customer.getInvoiceAddressLongitude());
            customerObj.put("invoice_adress_latitude", customer.getInvoiceAddressLatitude());
            customerArray.add(customerObj);
        }
        try {
            customerArray.writeJSONString(resp.getWriter());
        } catch (IOException e) {
            LOG.error("Failed to write to response", e);
            throw e;
        }
    }

    private List<Customer> getAllCustomers() {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">> getAllCustomers()");
        }

        // prepare WS request parameters
        QueryCustomersRequest request = new QueryCustomersRequest();
        CustomerFilter customerFilter = new CustomerFilter();
        customerFilter.setCustomerNumberPattern("*");
        request.setCustomerFilter(customerFilter);

        // invoke webservice
        QueryCustomersResponse response = 
            (QueryCustomersResponse) webserviceTemplate.marshalSendAndReceive(
                    crmWsUrl, request);

        return response.getCustomers();
    }

}