/*******************************************************************************
 * Copyright (c) 2008,2009 Q-ImPrESS consortium
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * This work was funded in the context of the Q-ImPrESS research project  
 * (FP7-215013) by the European Union under the Information and  
 * Communication Technologies priority of the Seventh Research Framework  
 * Programme.
 *
 * Contributors:
 *     itemis 	- initial API and implementation
 *******************************************************************************/
package de.itemis.qimpress.showcase.crm_simulator.jsf.beans;

import java.util.ArrayList;
import java.util.List;

import javax.faces.context.FacesContext;
import javax.faces.event.ActionEvent;
import javax.faces.model.SelectItem;

import org.apache.log4j.Logger;

import com.icesoft.faces.component.gmap.GMapLatLng;

import de.itemis.qimpress.showcase.crm_simulator.be.domain.Country;
import de.itemis.qimpress.showcase.crm_simulator.be.domain.Customer;
import de.itemis.qimpress.showcase.crm_simulator.be.domain.CustomerCategory;
import de.itemis.qimpress.showcase.crm_simulator.be.domain.CustomerType;
import de.itemis.qimpress.showcase.crm_simulator.be.exceptions.ApplicationException;
import de.itemis.qimpress.showcase.crm_simulator.be.filter.CustomerFilter;
import de.itemis.qimpress.showcase.crm_simulator.be.service.CRMManager;
import de.itemis.qimpress.showcase.crm_simulator.be.service.ServiceLocator;

/**
 * The CrmSimulatorFeController class controls communication between
 * frontend and backend. The CrmSimulatorFeController also handles
 * navigation to the according (.jspx) pages (see the named showXXX-methods).
 * @author Claudius Haecker
 */
public class CrmSimulatorFeController {

    private static final Logger LOG = Logger.getLogger(CrmSimulatorFeController.class);

    private CRMManager crmManager = ServiceLocator.getInstance().getCRMManager();
    private List<Customer> customers;
    private CustomerFilter customerFilter;

    // navigation constants
    public static final String SHOW_WELCOME = "SHOW_WELCOME";
    public static final String SHOW_QIMPRESS = "SHOW_QIMPRESS";
    private static final String SHOW_QIMPRESS_FILTER = "SHOW_QIMPRESS_FILTER";

    // Select-Items and their String value 
    // (the String-Setter sets the according customer-filter-attribute)
    private ArrayList<SelectItem> customerCategoryItems;
    private String customerFilterCategoryString;

    private String customerFilterDeliveryCountryString;
    private String customerFilterInvoiceCountryString;

    // detailCustomerID and detail Customer
    private String detailCustomerID = "";
    private Customer detailCustomer;

    /**
     * initializes internal settings.
     */
    // attributes and constants for google maps
    // default-values for longitute / latitude (Pforzheim, Bluecherstr.32)
    public static final String DEFAULT_LATITUDE = "48.898544";
    public static final String DEFAULT_LONGITUDE = "8.717322";
    public static final String DEFAULT_ZOOMLEVEL = "3";
    //    public static final String ZOOMLEVEL_SEVEN = "7";

    // attributes for gmap
    private String zoomLevel = DEFAULT_ZOOMLEVEL;
    private String centerLatitude = DEFAULT_LATITUDE;
    private String centerLongitude = DEFAULT_LONGITUDE;
    private String address = "";
    // whether we should search for an address or not
    private boolean locateAddress = false;

    private List<GMapLatLng> mapPoints = new ArrayList<GMapLatLng>();

    /**
     * @return a list of positions of th
     */
    public List<GMapLatLng> getMapPoints() {
        if (LOG.isDebugEnabled()) {
            LOG.debug("\t=> getMapPoints *****");
            LOG.debug("\t=> mapPoints.size() =" + mapPoints.size());
        }
        return mapPoints;
    }

    public void setMapPoints(List<GMapLatLng> mapPoints) {
        this.mapPoints = mapPoints;
    }

    /**
     * constructor for the CrmSimulatorFeController.
     * The constructor instanciates the crmManager and the customerFilter.
     */
    public CrmSimulatorFeController() {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">>  CrmSimulatorFeController");
        }
        if (crmManager == null) {
            LOG.error("CRMManager from ServiceLocater was null");
            throw new IllegalStateException("Could not configure CRMManager.");
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("crmManager = " + crmManager);
        }
        customerFilter = new CustomerFilter();
    }

    /**
     * asserts that
     * @return navigation string
     */
    public String filterCustomers() {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">>  filterCustomers ****************************************************");
            LOG.debug("\t=>  customerFilter.getCustomerNamePattern()" + customerFilter.getCustomerNamePattern());
            LOG.debug("\t=>  customerFilter.getCustomerNumberPattern()" + customerFilter.getCustomerNumberPattern());
            LOG.debug("\t=>  customerFilter.getCategory()" + customerFilter.getCategory());
            if (customerFilter.getCategory() != null) {
                LOG.debug("\t\t=> category ID   = " + customerFilter.getCategory().getCustomerCategoryId());
                LOG.debug("\t\t=> category KEY  = " + customerFilter.getCategory().getCustomerCategoryKey());
                LOG.debug("\t\t=> category NAME = " + customerFilter.getCategory().getCustomerCategoryName());
            }
        }
        loadFilteredCustomersFromBE();
        resetGmapAttributes();

        return showQimpress(true);
    }

    /**
     * load the customers list by the current customerFilter from backend.
     */
    private void loadFilteredCustomersFromBE() {
        try {
            customers = crmManager.queryCustomers(customerFilter);
        } catch (ApplicationException e) {
            LOG.error("Querying for customers failed", e);
            // TODO (MH) Read error code and trigger FE error handling 
        }
    }

    /**
     * resets the google maps attributes.
     * clears address, and resets latitute and longitude values.
     * note: zoom-level will not be displayed correct! (icefaces-bug?!)
     */
    private void resetGmapAttributes() {
        if (LOG.isDebugEnabled()) {
            LOG.debug("==> CALLED: resetGmapAttributes <==");
        }
        // however zoomLevel does not work as expected (icefaces-bug?!)
        this.zoomLevel = DEFAULT_ZOOMLEVEL;
        this.centerLatitude = DEFAULT_LATITUDE;
        this.centerLongitude = DEFAULT_LONGITUDE;
        this.address = "";
    }

    /**
     * resets the customer Filter to default values and navigates back
     * to the qimpress-page. (without reloading the customer list)
     * if the customerFilter is null, this method creates a new instance.
     * the Object type filter attributes are set to null,
     * the String type fiter attributes are set to an empty String.
     * @return jsf navigation string
     */
    public String resetCustomerFilter() {
        if (LOG.isDebugEnabled()) {
            LOG.debug("==> CALLED: resetCustomerFilter <==");
        }
        if (customerFilter == null) {
            customerFilter = new CustomerFilter();
        }
        // reset the customerFilter:
        customerFilter.setCustomerNamePattern("");
        customerFilter.setCustomerNumberPattern("");
        customerFilter.setCategory(null);
        customerFilter.setCustomerType(null);
        customerFilter.setDeliveryCountry(null);
        customerFilter.setInvoiceCountry(null);
        // reset the filter-select-Strings:
        setCustomerFilterCategoryString("");
        setCustomerFilterTypeString("");
        setCustomerFilterDeliveryCountryString("");
        setCustomerFilterInvoiceCountryString("");

        resetGmapAttributes();

        loadFilteredCustomersFromBE();

        return showQimpress(true);
    }

    // ************************************************************
    // ******************** NAVIGATION METHODS ********************
    // ************************************************************

    /**
     * @return the navigation string for the welcome page
     */
    public String showWelcome() {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">>  showWelcome");
        }
        return SHOW_WELCOME;
    }

    /**
     * navigates to the qimpress page without redirect.
     * @return a navigation String configured in faces-config.xml
     */
    public String showQimpress() {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">>  showQimpress()");
        }
        resetGmapAttributes();
        updateMapPoints();
        return showQimpress(false);
    }

    /**
     * navigates to the qimpress page.
     * @param redirect the parameter is used to control redirect.
     * @return a navigation String configured in faces-config.xml
     */
    public String showQimpress(boolean redirect) {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">>  showQimpress(boolean redirect)");
        }
        locateAddress = false;
        if (customers == null) {
            filterCustomers();
        }
        updateMapPoints();
        // redirect is required to reload google map
        if (redirect) {
            return SHOW_QIMPRESS_FILTER;
        }
        return SHOW_QIMPRESS;
    }

    // ************* END OF NAVIGATION METHODS AREA ***************
    // ************************************************************

    /**
     * clears and updates the mapPoints list by the current customers list.
     */
    private void updateMapPoints() {
        if (LOG.isDebugEnabled()) {
            LOG.debug("\t=> CALLED updateMapPoints");
        }
        if (mapPoints == null) {
            mapPoints = new ArrayList<GMapLatLng>();
        }
        mapPoints.clear();
        if (customers == null) {
            loadFilteredCustomersFromBE();
        }
        for (Customer customer : customers) {
            mapPoints.add(new GMapLatLng(String.valueOf(customer.getDeliveryAddressLatitude()), String.valueOf(customer
                    .getDeliveryAddressLongitude())));
            if (LOG.isDebugEnabled()) {
                LOG.debug("\t=>LATITUDE = " + String.valueOf(customer.getDeliveryAddressLatitude())
                        + "\t=>LONGITUDE = " + String.valueOf(customer.getDeliveryAddressLongitude()));
                LOG.debug("\t***** => mapPoints-size = " + mapPoints.size());
            }
        }
    }

    // ************************************************************
    // ****************** GETTER / SETTER METHODS *****************
    // ************************************************************

    /**
     * @return the customers
     */
    public List<Customer> getCustomers() {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">>  getCustomers");
        }
        return customers;
    }

    /**
     * @return the customerFilter
     */
    public CustomerFilter getCustomerFilter() {
        return customerFilter;
    }

    /**
     * @param customerFilter the customerFilter to set
     */
    public void setCustomerFilter(CustomerFilter customerFilter) {
        this.customerFilter = customerFilter;
    }

    /**
     * Getter for customer-category-select-items.
     * @return a select-item list with all existing customer-category names
     */
    public ArrayList<SelectItem> getCustomerCategoryItems() {
        if (LOG.isDebugEnabled()) {
            LOG.debug("==> CALLED: getCustomerCategoryItems <==");
        }
        ArrayList<SelectItem> selectOptions = new ArrayList<SelectItem>();
        // empty select option (no filter for customer category)
        selectOptions.add(new SelectItem(""));
        try {
            List<CustomerCategory> allCategories = crmManager.getCustomerCategories();
            for (CustomerCategory customerCategory : allCategories) {
                selectOptions.add(new SelectItem(customerCategory.getCustomerCategoryKey()));
            }
        } catch (ApplicationException ae) {
            LOG.error("!!! APPLICATION-EXCEPTION - getCustomerCategoryItems - " + "cause: " + ae.getCause());
            // TODO add faces message:
            //          facesContext.addMessage("globalMessage", new FacesMessage(
            //                    FacesMessage.SEVERITY_ERROR, "ERROR getting PiC-SelectItems", "globalMessage"));
        }
        customerCategoryItems = selectOptions;
        return customerCategoryItems;
    }

    /**
     * setter for customerCategoryItems
     * @param customerCategoryItems List of customerCategoryItems to set
     */
    public void setCustomerCategoryItems(ArrayList<SelectItem> customerCategoryItems) {
        this.customerCategoryItems = customerCategoryItems;
    }

    /**
     * the customerFilterCategoryString attribute is used to set the
     * customerFilter attribute category
     * @return key of category to filter for
     * */
    public String getCustomerFilterCategoryString() {
        if (LOG.isDebugEnabled()) {
            LOG.debug("==> CALLED: getCustomerFilterCategoryString <==");
        }
        return customerFilterCategoryString;
    }

    /**
     * sets a new category with customerCategoryKey customerFilterCategoryString.
     * @param customerFilterCategoryString the CustomerCategoryKey to filter for
     */
    public void setCustomerFilterCategoryString(String customerFilterCategoryString) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("==> CALLED: setCustomerFilterCategoryString <==");
        }
        CustomerCategory category = new CustomerCategory();
        category.setCustomerCategoryKey(customerFilterCategoryString);
        customerFilter.setCategory(category);
        this.customerFilterCategoryString = customerFilterCategoryString;
    }

    /**
     * @return the customerFilterDeliveryCountryString
     */
    public String getCustomerFilterDeliveryCountryString() {
        // FIXME: why is this method called for each item in the select menu,
        // i.e. about 250 times per call?
        //        if (LOG.isDebugEnabled()) {
        //            LOG.debug(">>  getCustomerFilterDeliveryCountryString");
        //        }
        return customerFilterDeliveryCountryString;
    }

    /**
     * @param customerFilterDeliveryCountryString the customerFilterDeliveryCountryString to set
     */
    public void setCustomerFilterDeliveryCountryString(String customerFilterDeliveryCountryString) {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">>  setCustomerFilterDeliveryCountryString");
        }
        customerFilter.setDeliveryCountryByName(customerFilterDeliveryCountryString);
        this.customerFilterDeliveryCountryString = customerFilterDeliveryCountryString;
    }

    /**
     * @return the customerFilterInvoiceCountryString
     */
    public String getCustomerFilterInvoiceCountryString() {
        // FIXME: why is this method called for each item in the select menu,
        // i.e. about 250 times per call?
        //        if (LOG.isDebugEnabled()) {
        //            LOG.debug(">>  getCustomerFilterInvoiceCountryString");
        //        }
        return customerFilterInvoiceCountryString;
    }

    /**
     * @param customerFilterInvoiceCountryString the customerFilterInvoiceCountryString to set
     */
    public void setCustomerFilterInvoiceCountryString(String customerFilterInvoiceCountryString) {
        customerFilter.setInvoiceCountryByName(customerFilterInvoiceCountryString);
        this.customerFilterInvoiceCountryString = customerFilterInvoiceCountryString;
    }

    /**
     * Getter for customer-type-select-items.
     * @return a select-item list with all existing customer-type names
     */
    public ArrayList<SelectItem> getCustomerTypeItems() {
        if (LOG.isDebugEnabled()) {
            LOG.debug("==> CALLED: getCustomerTypeItems <==");
        }
        ArrayList<SelectItem> selectOptions = new ArrayList<SelectItem>();
        // empty select option
        selectOptions.add(new SelectItem(""));
        try {
            List<CustomerType> allTypes = crmManager.getCustomerTypes();
            for (CustomerType customerType : allTypes) {
                selectOptions.add(new SelectItem(customerType.getCustomerTypeName()));
            }
        } catch (ApplicationException ae) {
            LOG.error("!!! APPLICATION-EXCEPTION - getCustomerTypeItems - " + "cause: " + ae.getCause());
            // TODO add faces message:
            //          facesContext.addMessage("globalMessage", new FacesMessage(
            //                    FacesMessage.SEVERITY_ERROR, "ERROR getting PiC-SelectItems", "globalMessage"));
        }
        return selectOptions;
    }

    /**
     * @return the customerFilter attribute customerType
     */
    public String getCustomerFilterTypeString() {
        return customerFilter.getCustomerType().getCustomerTypeName();
    }

    /**
     * sets the name of CustomerType - if necessary by adding a type 
     * @param customerFilterTypeString name of the customerType
     * 
     */
    public void setCustomerFilterTypeString(String customerFilterTypeString) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("==> CALLED: setCustomerFilterTypeString <==");
        }
        customerFilter.setCustomerTypeByName(customerFilterTypeString);
    }

    /**
     * Getter for customer-category-select-items.
     * @return a select-item list with all existing customer-category names
     */
    public ArrayList<SelectItem> getCountryItems() {
        if (LOG.isDebugEnabled()) {
            LOG.debug("==> CALLED: getCountryItems <==");
        }
        ArrayList<SelectItem> selectOptions = new ArrayList<SelectItem>();
        // empty select option (no filter for customer category)
        selectOptions.add(new SelectItem(""));
        try {
            List<Country> allCountries = crmManager.getCountries();
            for (Country country : allCountries) {
                selectOptions.add(new SelectItem(country.getName(), truncate(country.getName())));
            }
        } catch (ApplicationException ae) {
            LOG.error("!!! APPLICATION-EXCEPTION - getCountryItems - " + "cause: " + ae.getCause());
            // TODO add faces message:
            //          facesContext.addMessage("globalMessage", new FacesMessage(
            //                    FacesMessage.SEVERITY_ERROR, "ERROR getting PiC-SelectItems", "globalMessage"));
            ae.printStackTrace();
        }
        return selectOptions;
    }

    /**
     * the attribute detailCustomerID is used (on qimpressCRM-page)
     * as parameter to display the data for the detailCustomer in a popup panel.
     * @return the detailCustomerID
     */
    public String getDetailCustomerID() {
        return detailCustomerID;
    }

    public void setDetailCustomerID(String theDetailCustomerID) {
        this.detailCustomerID = theDetailCustomerID;
    }

    public Customer getDetailCustomer() {
        return detailCustomer;
    }

    public void setDetailCustomer(Customer detailCustomer) {
        this.detailCustomer = detailCustomer;
    }

    public String getZoomLevel() {
        return zoomLevel;
    }

    public void setZoomLevel(String zoomLevel) {
        this.zoomLevel = zoomLevel;
    }

    public String getCenterLatitude() {
        return centerLatitude;
    }

    public void setCenterLatitude(String centerLatitude) {
        this.centerLatitude = centerLatitude;
    }

    public String getCenterLongitude() {
        return centerLongitude;
    }

    public void setCenterLongitude(String centerLongitude) {
        this.centerLongitude = centerLongitude;
    }

    public String getAddress() {
        return address;
    }

    public void setAddress(String address) {
        this.address = address;
    }

    public boolean isLocateAddress() {
        return locateAddress;
    }

    public void setLocateAddress(boolean locateAddress) {
        this.locateAddress = locateAddress;
    }

    // ************ END GETTER / SETTER METHODS AREA **************************
    // ************************************************************************

    // *************************************************************************
    // ******************* attributes and methods for popup panel *************

    private boolean draggableRendered = false;
    private boolean autoCentre = false;

    public boolean getDraggableRendered() {
        return draggableRendered;
    }

    public void setDraggableRendered(boolean draggableRendered) {
        this.draggableRendered = draggableRendered;
    }

    public boolean getAutoCentre() {
        return autoCentre;
    }

    public void setAutoCentre(boolean autoCentre) {
        this.autoCentre = autoCentre;
    }

    /**
     * toggles the draggableRendered attribute
     * (for popup-panel on the qimpressCRM page).
     * the method gets the detailCustomerID-parameter from the faces context
     * and sets the detailCustomer.
     * @param event ignored
     */
    public void toggleDraggable(ActionEvent event) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("\t==> toggleDraggable <==");
        }
        // TODO check for exceptions
        setDetailCustomerByRequestParameterID();
        draggableRendered = !draggableRendered;
    }

    // ***************** END - popup attributes, getter/setter ****************
    // ************************************************************************

    /**
     * truncates Strings that are longer than 12 characters
     * after 10 characters and adds three points to the String.
     * */
    /**
     * @param string the string to shorten
     * @return the shortened string
     */
    public static String truncate(String string) {
        if (string.length() > 12) {
            return string.substring(0, 10) + "...";
        } else {
            return string;
        }
    }

    /**
     * updates the data so that the delivery adress of the customer is displayed
     * @return navigation string to show the correct web page with the g-maps app
     */
    public String locateCustomerDeliveryAdress() {
        if (LOG.isDebugEnabled()) {
            LOG.debug("\t=> locateCustomerDeliveryAdress <=");
        }
        // set detail-customer by the request parameter customerDetailID
        setDetailCustomerByRequestParameterID();

        // adress: county, zip-code city, street street-number
        address = detailCustomer.getDeliveryAddressCountry().getName() + ", " + detailCustomer.getDeliveryAddressCity()
                + ", " + detailCustomer.getDeliveryAddressStreet() + " "
                + detailCustomer.getDeliveryAddressStreetNumber();

        locateAddress = true;

        return showQimpress(false);
    }

    /**
     * sets the detailCustomer.The method gets the request parameter 'detailCustomer',
     * iterates through the customers-list and sets the detailCustomer by the
     * detailCustomerID parameter.
     * */
    public void setDetailCustomerByRequestParameterID() {
        String localDetailCustomerID = FacesContext.getCurrentInstance().getExternalContext().getRequestParameterMap()
                .get("detailCustomerID");
        if (localDetailCustomerID != null && !("").equals(localDetailCustomerID)) {
            detailCustomerID = localDetailCustomerID;
            // get detailCustomer from the current customer list
            if (customers != null && customers.size() > 0) {
                for (Customer iterateCustomer : customers) {
                    if (iterateCustomer.getCustomerId().equals(detailCustomerID)) {
                        if (LOG.isDebugEnabled()) {
                            LOG.debug("\t********** SET DETAIL-CUSTOMER **********");
                        }
                        detailCustomer = iterateCustomer;
                        break;
                    }
                }
                //draggableRendered = !draggableRendered;
            }
        } else {
            LOG.error("\t=> ERROR GETTING PARAMETER detailCustomerID FROM FACES-CONTEXT");
            // TODO display ERROR => don not toggle draggableRendered attribute
        }
    }

}
