/*******************************************************************************
 * Copyright (c) 2008,2009 Q-ImPrESS consortium
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * This work was funded in the context of the Q-ImPrESS research project  
 * (FP7-215013) by the European Union under the Information and  
 * Communication Technologies priority of the Seventh Research Framework  
 * Programme.
 *
 * Contributors:
 *     itemis 	- initial API and implementation
 *******************************************************************************/
package de.itemis.qimpress.showcase.crm_simulator.be.service;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.text.DateFormat;
import java.text.ParseException;
import java.util.Date;
import java.util.List;
import java.util.Locale;

import javax.naming.NamingException;

import org.apache.log4j.Logger;
import org.junit.BeforeClass;
import org.junit.Test;

import de.itemis.qimpress.showcase.crm_simulator.be.domain.Customer;
import de.itemis.qimpress.showcase.crm_simulator.be.exceptions.ApplicationException;
import de.itemis.qimpress.showcase.crm_simulator.be.filter.CustomerFilter;

/**
 * Tests whether all attributes of customers are correctly called and
 * whether filtering customers by name and by customer number works.
 * @author Claudius Haecker
 */
public class CRMManagerFilterTest {

    private static final Logger LOG = Logger
            .getLogger(CRMManagerFilterTest.class);
    private static CRMManager crm = ServiceLocator.getInstance().getCRMManager();

    /**
     * The setUp method is called once before running the test.
     * It gets an CRMManager instance (crm) that is used in the
     * different test methods.
     */
    @BeforeClass
    public static void setUp() {
        if (LOG.isDebugEnabled()) {
            LOG.debug("\tTEST - called: setUp");
        }
        
    }

    /**
     * Tests that the CRMManager crm instanciated in the setUp method
     * is not null.
     */
    @Test
    public void testGetCRM() {
        if (LOG.isDebugEnabled()) {
            LOG.debug("\tTEST - called: testGetCRM");
        }
        assertNotNull(crm);
    }


    @Test
    public void testCustomerNamePattern() {
        if (LOG.isDebugEnabled()) {
            LOG.debug("\tTEST - called: testCustomerNamePattern");
        }
        CustomerFilter filter = new CustomerFilter();
        filter.setCustomerNamePattern("*An*");
        List<Customer> customerList = null;
        try {
            customerList = crm.queryCustomers(filter);
        } catch (ApplicationException e) {
            e.printStackTrace();
            fail();
        }
        // currently the filter function for the customerFilter
        // attribute customerNamePatter is implemented with 'ignore case'
        // see CustomerDaoImpl, so we expect list-size:8
        assertEquals(8, customerList.size());
    }


    @Test
    public void testCustomerNumberPattern() {
        if (LOG.isDebugEnabled()) {
            LOG.debug("\tTEST - called: testCustomerNumberPattern");
        }
        CustomerFilter filter = new CustomerFilter();
        List<Customer> customerList = null;
        try {
            filter.setCustomerNumberPattern("*200900*");
            customerList = crm.queryCustomers(filter);
            // => expected one customer - id:3, name:EvoBus GmbH
            assertEquals(1, customerList.size());
            assertEquals(customerList.get(0).getCustomerId(), "3");
            assertEquals(customerList.get(0).getName(), "EvoBus GmbH");

            // new test - change filter number:
            filter.setCustomerNumberPattern("9200");
            customerList.clear();
            customerList = crm.queryCustomers(filter);
            // expected 4 customers
            assertEquals(4, customerList.size());
        } catch (ApplicationException e) {
            e.printStackTrace();
            fail();
        }
    }


    /**
     * test all attributes (of one row) wheather they are  
     */
    @Test
    public void testAllCustomerColumns() {
        if (LOG.isDebugEnabled()) {
            LOG.debug("\tTEST - called: testAllCustomerColumns");
        }
        List<Customer> customerList = null;
        try {
            customerList = crm.queryCustomers(null);
        } catch (ApplicationException e) {
            e.printStackTrace();
            fail();
        }
        for (Customer customer : customerList) {
            if (customer.getCustomerId().equals("8")) {
                assertEquals("Blumen Müller", customer.getName());
                assertEquals("005201200", customer.getCustomerNumber());
                assertEquals("", customer.getDeliveryAddressField2());
                assertEquals("", customer.getDeliveryAddressField3());
                assertEquals("", customer.getDeliveryAddressField1());
                assertEquals("Schreiberstr.", customer.getDeliveryAddressStreet());
                assertEquals("8", customer.getDeliveryAddressStreetNumber());
                assertEquals("70199", customer.getDeliveryAddressZipCode());
                assertEquals("Stuttgart", customer.getDeliveryAddressCity());
                assertEquals(9.15997, customer.getDeliveryAddressLongitude(), 0.0);
                assertEquals(48.762007, customer.getDeliveryAddressLatitude(), 0.0);
                assertEquals("", customer.getInvoiceAddressField1());
                assertEquals("", customer.getInvoiceAddressField2());
                assertEquals("", customer.getInvoiceAddressField3());
                assertEquals("Schreiberstr.", customer.getInvoiceAddressStreet());
                assertEquals("8", customer.getInvoiceAddressStreetNumber());
                assertEquals("70199", customer.getInvoiceAddressZipCode());
                assertEquals("Stuttgart", customer.getInvoiceAddressCity());
                assertEquals("", customer.getCommentary());
                // we don't know the exact value of the creation and modification date
                // because it is different in each test-run.
                DateFormat df = DateFormat.getDateTimeInstance(DateFormat.SHORT, DateFormat.SHORT, Locale.GERMAN);
                Date date = null;
                try {
                    date = df.parse("01.09.2008 10:29:11");
                } catch (ParseException e) {
                    e.printStackTrace();
                    fail("wrong time or date format");
                }
                assertTrue(0 > date.compareTo(customer.getDateCreated()));
                assertTrue(0 > date.compareTo(customer.getDateLastModified()));
                assertNotNull(customer.getDateCreated());
                assertNotNull(customer.getDateLastModified());
                assertEquals(0f, customer.getDiscountRate(), 0.0);
                assertEquals(9.15997, customer.getInvoiceAddressLongitude(), 0.0);
                assertEquals(48.762007, customer.getInvoiceAddressLatitude(), 0.0);
                assertEquals("8", customer.getCustomerId());
                return;
            }
        }
        fail("No customer with CustomerId 8 found.");
        return;

    }
    //    private String customerNamePattern;
    //    private String customerNumberPattern;
    //    private CustomerCategory category;
    //    private CustomerType type;
    //    private Country country;

}
