/*******************************************************************************
 * Copyright (c) 2008,2009 Q-ImPrESS consortium
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * This work was funded in the context of the Q-ImPrESS research project  
 * (FP7-215013) by the European Union under the Information and  
 * Communication Technologies priority of the Seventh Research Framework  
 * Programme.
 *
 * Contributors:
 *     itemis 	- initial API and implementation
 *******************************************************************************/
package de.itemis.qimpress.showcase.crm_simulator.be.dao;

import java.util.List;

import org.apache.log4j.Logger;
import org.hibernate.Criteria;
import org.hibernate.criterion.Restrictions;
import org.springframework.dao.DataAccessException;
import org.springframework.orm.hibernate3.support.HibernateDaoSupport;
import org.springframework.transaction.annotation.Transactional;

import de.itemis.qimpress.showcase.crm_simulator.be.domain.Country;
import de.itemis.qimpress.showcase.crm_simulator.be.domain.Customer;
import de.itemis.qimpress.showcase.crm_simulator.be.exceptions.DaoException;
import de.itemis.qimpress.showcase.crm_simulator.be.filter.CustomerFilter;

/**
 * This class implements the interface defined by {@link CustomerDao} using the hibernate technology. 
 * 
 * @author Claudius Haecker
 */
public class CustomerDaoImpl extends HibernateDaoSupport implements CustomerDao {

    private static final Logger LOG = Logger.getLogger(CustomerDaoImpl.class);

    
    /**
     * retrieves a List of customers that match given filter criteria. Returns all customers if filter== null.
     <table border="1">
                    <TR>
                        <TH></TH><TH>namepattern</TH><TH>numberpattern</TH><TH>customer Category Name</TH>
                        <TH>customer Category Key</TH><TH>customer Category Name</TH><TH>customer Category Key</TH>
                        <TH>name (delivery Country and invoice Country)</TH><TH>isoCode (delivery Country and invoice Country)</TH>
                    </TR>
                    <TR>
                        <TD>
                            if the id of type / category / country matches:
                            all other criteria of type / category / country are ignored:</TD>
                            <TD>N/A</TD><TD>N/A</TD><TD>yes</TD><TD>yes</TD><TD>yes</TD><TD>yes
                        </TD><TD>yes</TD><TD>yes</TD>
                    </TR>
                    <TR>
                        <TD>
                            accepts wildcards (%,* for arbitrary text, _,? for arbitrary letter)</TD><TD>yes</TD><TD>yes</TD>
                            <TD>yes</TD><TD>no</TD><TD>yes</TD><TD>no</TD><TD>yes</TD><TD>no</TD>
                    </TR>
                    <TR>
                        <TD>
                            matches even if occuring only as a substring 
                            </TD><TD>yes</TD><TD>yes</TD><TD>yes</TD><TD>no</TD><TD>yes</TD><TD>no</TD>
                            <TD><strong>no</strong></TD><TD>no</TD>
                    </TR>
                    <TR>
                        <TD>
                            null means: no restriction by this attribute 
                            </TD><TD>yes</TD><TD>yes</TD><TD>yes</TD><TD>yes</TD><TD>yes</TD><TD>yes</TD><TD>yes</TD><TD>yes</TD>
                    </TR>
                    <TR>
                        <TD>
                            ignores case (A=a, B=b, ...)
                            </TD><TD>yes</TD><TD>no</TD><TD>no</TD><TD>no</TD><TD>no</TD><TD>no</TD><TD>no</TD><TD>no</TD>
                    </TR>
    </table>
     *  <p>We don't allow fragments of the country name to match the like statement. Otherwise 'CONGO' (which is a state) would match
                     'CONGO, THE DEMOCRATIC REPUBLIC OF THE', another state.</p>
     * @param filter the criteria
     * @return a list of Customers that match the criterions in the filter object 
     * @throws DaoException if database problems occur
     */
    @Transactional
    public List<Customer> queryCustomers(CustomerFilter filter) throws DaoException {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">> queryCustomers");
        }
        Criteria criteria = null;
        List<Customer> customers = null;
        try {
            criteria = this.getHibernateTemplate().getSessionFactory().getCurrentSession().createCriteria(
                    Customer.class);
        } catch (DataAccessException e) {
            LOG.error(e.getLocalizedMessage(), e);
            throw new DaoException(e);
        }
        if (filter != null) {
            applyPattern(criteria, "name", filter.getCustomerNamePattern(), true);
            applyPattern(criteria, "customerNumber", filter.getCustomerNumberPattern(), false);
            filterCustomerType(filter, criteria);
            filterCustomerCateogory(filter, criteria);
            filterCountry(filter.getDeliveryCountry(), criteria, "deliveryAddressCountry");
            filterCountry(filter.getInvoiceCountry(), criteria, "invoiceAddressCountry");
        }
        try {
            // #################### get the result-set list #################### 
            customers = criteria.list();
        } catch (DataAccessException e) {
            LOG.error(e.getLocalizedMessage(), e);
            e.printStackTrace();
            throw new DaoException(e);
        }
        if (LOG.isDebugEnabled()) {
            if (customers != null) {
                LOG.debug("\t***** CUSTOMERS - SIZE = " + customers.size());
                for (Customer customer : customers) {
                    LOG.debug("\t=> customer NAME : " + customer.getName());
                }
            }
        }
        return customers;
    }

    /**
     * Adds new restrictions to the criteria
     * @param filter the data to filter for
     * @param criteria is changed and extended according to the <code>filter</code> parameter
     */
    private void filterCustomerType(CustomerFilter filter, Criteria criteria) {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">> filterCustomerType");
        }

        if (filter.getCustomerType() != null) {
            if (filter.getCustomerType().getCustomerTypeId() != null) {
                criteria.add(Restrictions.eq("customerType", filter.getCustomerType()));
            } else {
                if (filter.getCustomerType().getCustomerTypeName() != null
                        && !"".equals(filter.getCustomerType().getCustomerTypeName())) {
                    // create new criteria for the join of two tables
                    // public Criteria createCriteria(String associationPath,
                    //         String alias) throws HibernateException
                    String pattern = filter.getCustomerType().getCustomerTypeName();
                    pattern = pattern.replace('*', '%');
                    pattern = pattern.replace('?', '_');
                    // allow fragments of the name to match the like statement
                    pattern = "%" + pattern + "%";
                    criteria.createCriteria("customerType", "ctypeN").add(
                            Restrictions.like("ctypeN.customerTypeName", pattern));
                }
                String searchedCustomerTypeKey = filter.getCustomerType().getCustomerTypeKey();
                if (searchedCustomerTypeKey != null && !"".equals(searchedCustomerTypeKey)) {
                    criteria.createCriteria("customerType", "ctypeK").add(
                            Restrictions.eq("ctypeK.customerTypeKey", searchedCustomerTypeKey));
                }
            }
        }
    }

    /**
     * Adds new restrictions to the criteria
     * @param filter the data to filter for
     * @param criteria is changed and extended according to the <code>filter</code> parameter
     */
    private void filterCustomerCateogory(CustomerFilter filter, Criteria criteria) {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">> filterCustomerCateogory");
        }
        // TODO: (MH) Looks like either CatName or CatId can be set, not both of them. If this is the case
        // add a pre-condition check to fulfill DBC requirements. 
        
        if (filter.getCategory() != null) {
            if (filter.getCategory().getCustomerCategoryId() != null) {
                criteria.add(Restrictions.eq("customerCategory", filter.getCategory()));
            } else {
                if (filter.getCategory().getCustomerCategoryName() != null
                        && !"".equals(filter.getCategory().getCustomerCategoryName())) {
                    // create new criteria for the join of two tables
                    // public Criteria createCriteria(String associationPath,
                    //         String alias) throws HibernateException
                    String pattern = filter.getCategory().getCustomerCategoryName();
                    pattern = pattern.replace('*', '%');
                    pattern = pattern.replace('?', '_');
                    // allow fragments of the name to match the like statement
                    pattern = "%" + pattern + "%";
                    criteria.createCriteria("customerCategory", "ccatN").add(
                            Restrictions.like("ccatN.customerCategoryName", pattern));
                }
                String searchedCustomerCategoryKey = filter.getCategory().getCustomerCategoryKey();
                if (searchedCustomerCategoryKey != null && !"".equals(searchedCustomerCategoryKey)) {
                    criteria.createCriteria("customerCategory", "ccatK").add(
                            Restrictions.eq("ccatK.customerCategoryKey", searchedCustomerCategoryKey));
                }
            }
        }
    }

    /**
     * Adds new restrictions to the criteria
     * @param filter the data to filter for
     * @param criteria is changed and extended according to the <code>filter</code> parameter
     * @param foreignKey how the country is referenced by hibernate
     */
    private void filterCountry(Country searchedCountry, Criteria criteria, String foreignKey) {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">> filterCountry");
        }
        
        // TODO: (MH) Looks like either CountryId or CountryName can be set, not both of them. If this is the case
        // add a pre-condition check to fulfill DBC requirements. 
        
        if (searchedCountry != null) {
            if (searchedCountry.getCountryId() != null) {
                criteria.add(Restrictions.eq(foreignKey, searchedCountry));
            } else {
                if (searchedCountry.getName() != null && !"".equals(searchedCountry.getName())) {
                    // create new criteria for the join of two tables
                    // public Criteria createCriteria(String associationPath,
                    //         String alias) throws HibernateException
                    String pattern = searchedCountry.getName();
                    pattern = pattern.replace('*', '%');
                    pattern = pattern.replace('?', '_');
                    // don't allow fragments of the name to match the like statement
                    // Problem: 'CONGO' (which is a state) would match
                    // 'CONGO, THE DEMOCRATIC REPUBLIC OF THE', another state.
                    criteria.createCriteria(foreignKey, foreignKey + "cntryN").add(
                            Restrictions.like(foreignKey + "cntryN.name", pattern));
                }
                String searchedIsoCode = searchedCountry.getIsoCode();
                if (searchedIsoCode != null && !"".equals(searchedIsoCode)) {
                    criteria.createCriteria(foreignKey, foreignKey + "cntryK").add(
                            Restrictions.eq(foreignKey + "cntryK.isoCode", searchedIsoCode));
                }
            }
        }
    }

    /**
     * Adds new restrictions to the criteria
     * @param criteria is changed and extended according to the <code>filter</code> parameter 
     * @param hibernateTableColumn how the column is referenced by hibernate
     * @param pattern to filter for. May contain '*', '%', '?' and '_'.
     * @param ignoreCase tells whether small and capital letters are treated the same
     */
    private void applyPattern(Criteria criteria, String hibernateTableColumn, String pattern, boolean ignoreCase) {
        if (LOG.isDebugEnabled()) {
            LOG.debug(">> applyPattern");
        }
        if (pattern != null && !"".equals(pattern)) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("\t customer name pattern: " + pattern);
            }
            pattern = pattern.replace('*', '%');
            pattern = pattern.replace('?', '_');
            // allow fragments of the name to match the like statement
            pattern = "%" + pattern + "%";
            if (ignoreCase) {
                criteria.add(Restrictions.ilike(hibernateTableColumn, pattern));
            } else {
                criteria.add(Restrictions.like(hibernateTableColumn, pattern));
            }
        }
    }
}
