package spec.benchmarks._239_nih; 
//Opens uncompressed TIFF files

import java.io.*;

class TiffDecoder {

	// tags
	static final int NEW_SUBFILE_TYPE = 254;
	static final int IMAGE_WIDTH = 256;
	static final int IMAGE_LENGTH = 257;
	static final int BITS_PER_SAMPLE = 258;
	static final int COMPRESSION = 259;
	static final int PHOTO_INTERP = 262;
	static final int STRIP_OFFSETS = 273;
	static final int SAMPLES_PER_PIXEL = 277;
	static final int ROWS_PER_STRIP = 278;
	static final int STRIP_BYTE_COUNT = 279;
	static final int X_RESOLUTION = 282;
	static final int Y_RESOLUTION = 283;
	static final int PLANAR_CONFIGURATION = 284;
	static final int RESOLUTION_UNIT = 296;
	static final int COLOR_MAP = 320;
	static final int IMAGE_HDR = -22222; //43314
	
	//field types
	static final int SHORT = 3;
	static final int LONG = 4;


	private String path;
	private ImageJ ij;
	private File f;
	RandomAccessFile in;
	private boolean littleEndian;
	
	
	TiffDecoder(String path, ImageJ ij) {
		this.path = path;
		this.ij = ij;
	}


	final int getInt() throws IOException {
		int b1 = in.read();
		int b2 = in.read();
		int b3 = in.read();
		int b4 = in.read();
		if (littleEndian)
			return ((b4 << 24) + (b3 << 16) + (b2 << 8) + (b1 << 0));
		else
			return ((b1 << 24) + (b2 << 16) + (b3 << 8) + b4);
	}


	int getShort() throws IOException {
		int b1 = in.read();
		int b2 = in.read();
		if (littleEndian)
			return ((b2 << 8) + b1);
		else
			return ((b1 << 8) + b2);
	}


	int OpenImageFileHeader() throws IOException {
	// Open 8-byte Image File Header at start of file.
	// Returns the offset in bytes to the first IFD or -1
	// if this is not a valid tiff file.
		int byteOrder = in.readShort();
		if (byteOrder==0x4949) // "II"
			littleEndian = true;
		else if (byteOrder==0x4d4d) // "MM"
			littleEndian = false;
		else {
			in.close();
			return -1;
		}
		int magicNumber = getShort(); // 42
		int offset = getInt();
		return offset;
	}
	
	
	int getValue(int fieldType, int count) throws IOException {
		int value = 0;
		int unused;
		if (fieldType==SHORT && count==1) {
				value = getShort();
				unused = getShort();
		}
		else
			value = getInt();
		return value;
	}
	
	
	void getColorMap(int offset, FileInfo fi) throws IOException {
		byte[] colorTable16 = new byte[768*2];
		long saveLoc = in.getFilePointer();
		in.seek(offset);
		int bytesRead = in.read(colorTable16);
		in.seek(saveLoc);
		if (bytesRead!=768*2)
			return;
		fi.colorTable = new byte[768];
		int j = 0;
		if (littleEndian)
			j++;
		for (int i=0; i<768; i++) {
			fi.colorTable[i] = colorTable16[j];
			j += 2;
		}
		fi.fileType = FileInfo.COLOR8;
	}
	

	FileInfo OpenIFD() throws IOException {
	// Get Image File Directory data
	
		int tag, fieldType, count, value;
		int nEntries = getShort();
		if (nEntries<1)
			return null;
		FileInfo fi = new FileInfo();
		for (int i=0; i<nEntries; i++) {
			tag = getShort();
			fieldType = getShort();
			count = getInt();
			value = getValue(fieldType, count);
			if (tag==0)
				return null;
			switch (tag) {
				case IMAGE_WIDTH: 
					fi.width = value;
					break;
				case IMAGE_LENGTH: 
					fi.height = value;
					break;
				case STRIP_OFFSETS: 
					if (count==1)
						fi.offset = value;
					else {
						long saveLoc = in.getFilePointer();
						in.seek(value);
						fi.offset = getInt(); // Assumes contiguous strips
						in.seek(saveLoc);
					}
					break;
				case PHOTO_INTERP:
					fi.blackIsZero = value==1;
					break;
				case BITS_PER_SAMPLE:
						if (count==1) {
							if (value==8)
								fi.fileType = FileInfo.GRAY8;
							else if (value==16) {
								fi.fileType = FileInfo.GRAY16;
								fi.swapBytes = littleEndian;
							}
							else if (value==32) {
								fi.fileType = FileInfo.GRAY32_FLOAT;
								fi.swapBytes = littleEndian;
							}
							else
								throw new IOException("Unsupported TIFF BitsPerSample: " + value);
						}
						break;
				case SAMPLES_PER_PIXEL:
					if (value==3)
						fi.fileType = FileInfo.RGB;
					else if (value!=1)
						throw new IOException("Unsupported TIFF SamplesPerPixel: " + value);
					break;
				case PLANAR_CONFIGURATION:
					if (value==2 && fi.fileType==FileInfo.RGB)
						fi.fileType = FileInfo.RGB_PLANAR;
					break;
				case COMPRESSION: 
					if (value!=1)
						throw new IOException("TIFF compression is not supported");
				case COLOR_MAP: 
					if (count==768)
						getColorMap(value, fi);
					break;
				default:
			}
		}
		fi.fileName = f.getName();
		fi.ij = ij;
		return fi;
	}


	FileInfo getInfo() throws IOException {
		int ifdOffset;
		FileInfo fi = null;
				
		f = new File(path);
		in = new RandomAccessFile(f, "r");
		ifdOffset = OpenImageFileHeader();
		//System.out.println("ifdOffset: " + ifdOffset);
		if (ifdOffset<0) {
			in.close();
			return null;
		}
		in.seek(ifdOffset);
		fi = OpenIFD();
		in.close();
		return fi;
	}


	boolean open() throws IOException {
		FileInfo fi = getInfo();
		if (fi==null)
			return false;
		ImageReader file = new ImageReader(fi);
		FileInputStream in = new FileInputStream(f);
		file.openImage(in);
		in.close();
		return true;
	}
}