package spec.benchmarks._239_nih; 
// Opens raw 8, 16, 24 and 32-bit images

import java.awt.*;
import java.awt.image.*;
import java.io.*;

public class ImageReader {

    private FileInfo fi;
    private int width, height, offset;
    private int bytesPerPixel, bufferSize, byteCount, nPixels;
    private ImageJ ij;
	

	ImageReader (FileInfo fi) {
		this.fi = fi;
	    ij = fi.ij;
	    width = fi.width;
	    height = fi.height;
	    offset = fi.offset;
	}
	
	
	 static ColorModel makeColorModel(byte[] colorTable) {
		byte[] rLUT, gLUT,bLUT;
		int k;
		
		rLUT = new byte[256];
		gLUT = new byte[256];
		bLUT = new byte[256];
		for(int i=0; i<256; i++) {
			rLUT[i] = colorTable[i];
			gLUT[i] = colorTable[256+i];
			bLUT[i] = colorTable[512+i];
		}
		return(new IndexColorModel(8, 256, rLUT, gLUT, bLUT));
	}


	void read8bitImage(FileInputStream in) throws IOException {
		byte[] pixels;
		ImagePlus imp;
		int bytesRead = 0;
		pixels = new byte[nPixels];
		ColorModel cm;

		while (bytesRead<byteCount) {
			bytesRead += in.read(pixels, bytesRead, 8192);
			Info.showProgress((double)bytesRead/byteCount);
		}
		if (fi.fileType==FileInfo.COLOR8)
			cm = makeColorModel(fi.colorTable);
		else
			cm = Converter.makeGrayscaleColorModel(!fi.blackIsZero);
	    Image img = Toolkit.getDefaultToolkit().createImage(new MemoryImageSource(width, height, cm, pixels, 0, width));
        imp = new ImagePlus(fi.fileName, img, ij);
        imp.show();
	}
	
	
	void read16bitImage(FileInputStream in) throws IOException {
		ImagePlus imp;
		int pixelsRead;
		byte[] buffer = new byte[bufferSize];
		short[] pixels = new short[nPixels];

		int bytesRead = 0;
		int totalRead = 0;
		int base = 0;
		while (totalRead<byteCount) {
			bytesRead = in.read(buffer, 0, bufferSize);
			totalRead += bytesRead;
			Info.showProgress((double)totalRead/byteCount);
			pixelsRead = bytesRead/bytesPerPixel;
			if ((base+pixelsRead)>nPixels)
				pixelsRead = nPixels - base;
			int j = 0;
			if (fi.swapBytes)
				for (int i=base; i < (base+pixelsRead); i++) {
					pixels[i] = (short)(((buffer[j+1]&0xff)<<8) | (buffer[j]&0xff));
					j += 2;
				}
			else
				for (int i=base; i < (base+pixelsRead); i++) {
					pixels[i] = (short)(((buffer[j]&0xff)<<8) | (buffer[j+1]&0xff));
					j += 2;
				}
			base += pixelsRead;
		}
        imp = new ImagePlus(fi.fileName, pixels, width, height, ij);
        imp.show();
	}
	
	
	void read32bitImage(FileInputStream in) throws IOException {
		ImagePlus imp;
		int pixelsRead;
		byte[] buffer = new byte[bufferSize];
		float[] pixels = new float[nPixels];
		int tmp;

		int bytesRead = 0;
		int totalRead = 0;
		int base = 0;
		while (totalRead<byteCount) {
			bytesRead = in.read(buffer, 0, bufferSize);
			totalRead += bytesRead;
			Info.showProgress((double)totalRead/byteCount);
			pixelsRead = bytesRead/bytesPerPixel;
			if ((base+pixelsRead)>nPixels)
				pixelsRead = nPixels - base;
			int j = 0;
			if (fi.swapBytes)
				for (int i=base; i < (base+pixelsRead); i++) {
					tmp = (int)(((buffer[j+3]&0xff)<<24) | ((buffer[j+2]&0xff)<<16) | ((buffer[j+1]&0xff)<<8) | (buffer[j]&0xff));
					if (fi.fileType==FileInfo.GRAY32_FLOAT) {
						pixels[i] = Float.intBitsToFloat(tmp);
					}
					else
						pixels[i] = tmp;
					j += 4;
				}
			else
				for (int i=base; i < (base+pixelsRead); i++) {
					tmp = (int)(((buffer[j]&0xff)<<24) | ((buffer[j+1]&0xff)<<16) | ((buffer[j+2]&0xff)<<8) | (buffer[j+3]&0xff));
					if (fi.fileType==FileInfo.GRAY32_FLOAT)
						pixels[i] = Float.intBitsToFloat(tmp);
					else
						pixels[i] = tmp;
					j += 4;
				}
			base += pixelsRead;
		}
        imp = new ImagePlus(fi.fileName, width, height, ij);
		imp.setImage(pixels, width, height, !fi.blackIsZero);
        imp.show();
	}
	
	void readChunkyRGB(FileInputStream in) throws IOException {
		int pixelsRead;
		bufferSize = 24*width;
		byte[] buffer = new byte[bufferSize];
		int[] pixels = new int[nPixels];
		int r, g, b;

		int bytesRead = 0;
		int totalRead = 0;
		int base = 0;
		boolean first = true;
		while (totalRead<byteCount) {
			bytesRead = in.read(buffer, 0, bufferSize);
			totalRead += bytesRead;
			Info.showProgress((double)totalRead/byteCount);
			pixelsRead = bytesRead/bytesPerPixel;
			if ((base+pixelsRead)>nPixels)
				pixelsRead = nPixels - base;
			int j = 0;
			for (int i=base; i < (base+pixelsRead); i++) {
				r = buffer[j++]&0xff;
				g = buffer[j++]&0xff;
				b = buffer[j++]&0xff;
				pixels[i] = 0xff000000 | (r<<16) | (g<<8) | b;
			}
			base += pixelsRead;
		}
	    Image img = Toolkit.getDefaultToolkit().createImage(new MemoryImageSource(width, height, pixels, 0, width));
        ImagePlus imp = new ImagePlus(fi.fileName, img, ij);
        imp.show();
	}


	void readPlanarRGB(FileInputStream in) throws IOException {
		ImagePlus imp;
		int planeSize = nPixels; // 1/3 image size
		byte[] buffer = new byte[planeSize];
		int[] pixels = new int[nPixels];
		int r, g, b;

		int bytesRead;
		Info.showProgress(0.12);
		bytesRead = in.read(buffer, 0, planeSize);
		for (int i=0; i < planeSize; i++) {
			r = buffer[i]&0xff;
			pixels[i] = 0xff000000 | (r<<16);
		}
		
		Info.showProgress(0.37);
		bytesRead = in.read(buffer, 0, planeSize);
		for (int i=0; i < planeSize; i++) {
			g = buffer[i]&0xff;
			pixels[i] |= g<<8;
		}

		Info.showProgress(0.62);
		bytesRead = in.read(buffer, 0, planeSize);
		for (int i=0; i < planeSize; i++) {
			b = buffer[i]&0xff;
			pixels[i] |= b;
		}

		Info.showProgress(0.87);
	    Image img = Toolkit.getDefaultToolkit().createImage(new MemoryImageSource(width, height, pixels, 0, width));
        imp = new ImagePlus(fi.fileName, img, ij);
		Info.showProgress(1.0);
        imp.show();
	}


/*
	void readRGBAImage(FileInputStream in) throws IOException {
		ImagePlus imp;
		int pixelsRead;
		byte[] buffer = new byte[bufferSize];
		int[] pixels = new int[nPixels];
		int tmp;

		int bytesRead = 0;
		int totalRead = 0;
		int base = 0;
		while (totalRead<byteCount) {
			bytesRead = in.read(buffer, 0, bufferSize);
			totalRead += bytesRead;
			Info.showProgress((double)totalRead/byteCount);
			pixelsRead = bytesRead/bytesPerPixel;
			int j = 0;
			for (int i=base; i < (base+pixelsRead); i++) {
				pixels[i] = (int)(((buffer[j]&0xff)<<24) | ((buffer[j+1]&0xff)<<16) | ((buffer[j+2]&0xff)<<8) | (buffer[j+3]&0xff));
				j += 4;
			}
			base += pixelsRead;
		}
	    Image img = Toolkit.getDefaultToolkit().createImage(new MemoryImageSource(width, height, pixels, 0, width));
        imp = new ImagePlus(fi.fileName, img, ij);
        imp.show();
	}
*/
	

	void validateByteCount() {
		int imageSize = width*height*bytesPerPixel;
		nPixels = width*height;
		if (byteCount>imageSize)
			byteCount = imageSize;
		bufferSize = byteCount/25;
		if (bufferSize<8192)
			bufferSize = 8192;
		else
			bufferSize = (bufferSize/8192)*8192;
	}
	
	
	void openImage(FileInputStream in) {
		try {
			byteCount = in.available();
			if (offset!=0) {
				if (offset<0 || (byteCount-offset)<=0) {
					Info.write("Invalid offset: " + "offset=" + offset + ", file byteCount=" + byteCount);
					return;
				}
				long skipCount = in.skip((long)offset);
				if (skipCount!=offset) 
					Info.write("Offset error: " + offset + " " + skipCount);
				byteCount = (int)(byteCount - skipCount);
			}
			switch (fi.fileType) {
				case FileInfo.GRAY8:
				case FileInfo.COLOR8:
					bytesPerPixel = 1;
					validateByteCount();
					read8bitImage(in);
					break;
				case FileInfo.GRAY16:
					bytesPerPixel = 2;
					validateByteCount();
					read16bitImage(in);
					break;
				case FileInfo.GRAY32_INT:
				case FileInfo.GRAY32_FLOAT:
					bytesPerPixel = 4;
					validateByteCount();
					read32bitImage(in);
					break;
				case FileInfo.RGB:
					bytesPerPixel = 3;
					validateByteCount();
					readChunkyRGB(in);
					break;
				case FileInfo.RGB_PLANAR:
					bytesPerPixel = 3;
					validateByteCount();
					readPlanarRGB(in);
					break;
			}
			in.close();
		}
		catch (IOException e) {
			Info.write("" + e);
			return;
		}
	}
	
}

