package spec.benchmarks._239_nih; 
import java.util.*;
import java.awt.*;
import java.awt.image.*;

// This abstract class is the superclass of ByteProcessor and ColorProcessor.
abstract class ImageProcessor extends Object {

	static final int INVERT = 0;
	static final int CLEAR = 1;
	static final int FILL = 2;
	static final int ADD = 3;
	static final int MULT = 4;
	static final int AND = 5;
	static final int OR = 6;
	static final int XOR = 7;
	static final int GAMMA = 8;
	static final int LOG = 9;
	
	private int fgColor = 0;
	private int bgColor = 255;
	
    protected ProgressBar progressBar = null;
    protected boolean pixelsModified;
	protected int width, snapshotWidth;
	protected int height, snapshotHeight;
	protected int roiX, roiY, roiWidth, roiHeight;
	protected int xMin, xMax, yMin, yMax;
	protected boolean newSnapshot = false; // true if pixels = snapshotPixels
	protected int[] mask = null;

	
	protected void showProgress(double percentDone) {
		if (progressBar!=null)
        	progressBar.show(percentDone);
	}


	protected void hideProgress() {
		showProgress(1.0);
		newSnapshot = false;
	}
	
	
	public int getWidth() {
		return width;
	}
	

	public int getHeight() {
		return height;
	}
	

	public void setRoi(Rectangle roi) {
		if (roi != null) {
			roiX = roi.x;
			roiY = roi.y;
			roiWidth = roi.width;
			roiHeight = roi.height;
		}
		else {
			roiX = 0;
			roiY = 0;
			roiWidth = width;
			roiHeight = height;
		}
		//setup limits for 3x3 filters
		xMin = Math.max(roiX, 1);
		xMax = Math.min(roiX + roiWidth - 1, width - 2);
		yMin = Math.max(roiY, 1);
		yMax = Math.min(roiY + roiHeight - 1, height - 2);
		
		mask = null;
	}


	public Rectangle getRoi() {
		return new Rectangle(roiX, roiY, roiWidth, roiHeight);
	}


	public void setMask(int[] mask) {
		this.mask = mask;
	}


	public int[] getMask() {
		return mask;
	}


	private void process(int op, double value) {
		double SCALE = 255.0/Math.log(255.0);
		int v;
		
		byte[] lut = new byte[256];
		for (int i=0; i<256; i++) {
			switch(op) {
				case INVERT:
					v = 255 - i;
					break;
				case CLEAR:
					v = bgColor;
					break;
				case FILL:
					v = fgColor;
					break;
				case ADD:
					v = i + (int)value;
					break;
				case MULT:
					v = (int)Math.round(i * value);
					break;
				case AND:
					v = i & (int)value;
					break;
				case OR:
					v = i | (int)value;
					break;
				case XOR:
					v = i ^ (int)value;
					break;
				case GAMMA:
					v = (int)(Math.exp(value*Math.log(i/255.0))*255.0);
					break;
				case LOG:
					if (i==0)
						v = 0;
					else
						v = (int)(Math.log(i) * SCALE);
					break;
				 default:
				 	v = i;
			}
			if (v < 0)
				v = 0;
			if (v > 255)
				v = 255;
			lut[i] = (byte)(v & 0xff);
		}
		applyTable(lut);
    }


	public float [] getLine(int x, int y, int length) {
		float[] data = new float[length];
		for (int i=0; i<length; i++)
			data[i] = getPixel(x++, y);
		return data;
	}
	
	
	abstract float getPixel(int x, int y);
	abstract Object getPixels();
	abstract void insert(ImageProcessor ip, int xloc, int yloc);

	abstract void applyTable(byte[] lut);

	void invert() {process(INVERT, 0.0);}
	void clear() {process(CLEAR, 0.0);}
	void fill() {process(FILL, 0.0);}

	void add(int value) {process(ADD, value);}
	void multiply(double value) {process(MULT, value);}
	void and(int value) {process(AND, value);}
	void or(int value) {process(OR, value);}
	void xor(int value) {process(XOR, value);}
	void gamma(double value) {process(GAMMA, value);}
	void log() {process(LOG, 0.0);}

	abstract Image createImage();
	abstract Image createImage(Rectangle srcRect, int dstWidth, int dstHeight);
	abstract void snapshot();
	abstract void reset();
	abstract void reset(int[] mask);
	abstract void smooth();
	abstract void sharpen();
	abstract void findEdges();
    abstract void noise(double range);
	abstract void crop();
	abstract void scale(double xScale, double yScale, boolean resizeImage);
  	abstract void rotate(double angle);
	abstract void flipVertical();
	abstract void flipHorizontal();
	abstract void autoThreshold();
	abstract void enhanceContrast();
	abstract void medianFilter();
	
}

